/*

  FITSPNG     FITS to PNG converter
  Copyright (C) 2006-2019  Filip Hroch, Masaryk University, Brno, CZ

  Fitspng is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  Fitspng is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Fitspng.  If not, see <http://www.gnu.org/licenses/>.

*/

#include "fitspng.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <float.h>

#include <assert.h>

#if defined _WIN32 || defined __CYGWIN__
#define DIRSEP '\\'
#else
#define DIRSEP '/'
#endif

void help();
void version();
char *pngsuff(const char *);

int main(int argc,char *argv[])
{
  char *png = NULL;
  int bit_depth, scale, type, verb;
  float qblack,rsense,t,s,f0,fz,st,ss, satur;
  int i, set_fl, set_fn, set_satur, dcspace = 0;
  int nfits = 0;
  int status = 0;

  bit_depth = 8;
  scale = 1;
  type = 0;
  verb = 0;
  set_fl = 0; set_fn = 0; set_satur = 0;

  qblack = 0.25;
  rsense = 1.0;
  t = 0.0;
  s = 1.0;
  f0 = 1.0;
  fz = 0.0;
  satur = 1.0;

  if( argc == 1 ) {
    help();
    return(0);
  }

  for( i = 1; i < argc; i++) {

    /* help */
    if( strcmp(argv[i],"--help") == 0 || strcmp(argv[i],"-h") == 0 ){
      help();
      return(0);
    }

    /* version */
    if( strcmp(argv[i],"--version") == 0 ){
      version();
      return(0);
    }

    /* verbose */
    else if( strcmp(argv[i],"-v") == 0 || strcmp(argv[i],"--verbose") ==0 ){
      verb = 1;
    }

    /* output name */
    else if( strcmp(argv[i],"-o") == 0 && i++ < argc ){
      png = argv[i];
    }

    /* bit depth */
    else if( strcmp(argv[i],"-B") == 0 && i++ < argc ){

      if( !( sscanf(argv[i],"%d",&bit_depth) == 1 &&
	     (bit_depth == 8 ||  bit_depth == 16)) ) {
        fprintf(stderr,"Set bitpix to 8 or 16.\n");
        return(1);
      }
    }

    else if( strcmp(argv[i],"-f") == 0 && i++ < argc ){

      /* linear */
      if( strcmp(argv[i],"linear") == 0 )
	type = 0;

      /* square root */
      else if( strcmp(argv[i],"sqrt") == 0 )
	type = 5;

      /* square */
      else if( strcmp(argv[i],"sqr") == 0 )
	type = 6;

      /* logistic */
      else if( strcmp(argv[i],"logistic") == 0 )
	type = 7;

      else {
        fprintf(stderr,"A wrong type of conversion specified.\n");
        return(1);
      }
    }

    /* intensity scale */
    else if( strcmp(argv[i],"-f0") == 0 && i++ < argc ){
      if( sscanf(argv[i],"%f",&f0) != 1 && f0 > 0.0 ) {
        fprintf(stderr,"-f0: Specify a positive real number.\n");
        return(1);
      }
    }

    /* zero point */
    else if( strcmp(argv[i],"-fz") == 0 && i++ < argc ){
      if( sscanf(argv[i],"%f",&fz) != 1 ) {
        fprintf(stderr,"-fz: Specify a real number.\n");
        return(1);
      }
    }

    /* linear scale */
    else if( strcmp(argv[i],"-fl") == 0 && i++ < argc ){
      if( sscanf(argv[i],"%f,%f",&t,&s) != 2 && s > 0.0 ) {
        fprintf(stderr,"-fl: Specify a pair of real numbers separated by comma(second positive).\n");
        return(1);
      }
      set_fl = 1;
    }

    /* relative intensity scale */
    else if( strcmp(argv[i],"-fr") == 0 && i++ < argc ){
      if( sscanf(argv[i],"%f,%f",&qblack,&rsense) != 2 && rsense > 0.0 ) {
        fprintf(stderr,"-fr: Specify a pair of real numbers separated by comma (firs should be 0 .. 1, second positive).\n");
        return(1);
      }
    }

    /* colour saturation factor */
    else if( strcmp(argv[i],"-fs") == 0 && i++ < argc ){
      if( sscanf(argv[i],"%f",&satur) != 1 && satur >= 0.0 ) {
        fprintf(stderr,"-fs: Specify a positive (or zero) real number.\n");
        return(1);
      }
      set_satur = 1;
    }

    /* display colourspace */
    else if( strcmp(argv[i],"-cs") == 0 && i++ < argc ){

      /* sRGB */
      if( strcmp(argv[i],"sRGB") == 0 )
	dcspace = 0;

      /* AdobeRGB */
      else if( strcmp(argv[i],"AdobeRGB") == 0 )
	dcspace = 1;

      else {
        fprintf(stderr,"A wrong type of display colour-space specified.\n");
        return(1);
      }
    }

    /* scotopic threshold */
    else if( strcmp(argv[i],"-fn") == 0 && i++ < argc ){
      if( sscanf(argv[i],"%f,%f",&st,&ss) != 2 && ss > 0.0) {
        fprintf(stderr,"-fn: Specify a pair of real numbers separated by comma (second positive).\n");
        return(1);
      }
      set_fn = 1;
    }

    /* rescale */
    else if( strcmp(argv[i],"-s") == 0 && i++ < argc ){
      if( !(sscanf(argv[i],"%d",&scale) == 1 || scale > 0 ) ) {
        fprintf(stderr,"-s: Specify a positive decimal number.\n");
        return(1);
      }
    }

    else {

      if( *argv[i] == '-' )
	fprintf(stderr,"Warning: an unknown parameter? (%s).\n",argv[i]);

      /* keep the index of the first FITS file on command line  */
      if( nfits == 0 )
	nfits = i;
    }

  }

  if( nfits == 0 ) {
    fprintf(stderr,"Error: unspecified FITS file(s).\n");
    return 1;
  }

  if( argc - nfits > 1 && png ) {
    fprintf(stderr,"Error: Set of output filename (by -o) is uncompatible"
	  " with list of FITSes.\n");
    return 1;
  }

  if( verb ) {
    fprintf(stderr,"Scaling: qblack=%f rsense=%f\n",qblack,rsense);
    fprintf(stderr,"Itt: (type=%d): f0=%f fz=%f\n",type,f0,fz);
  }

  status = 0;
  for( i = nfits; i < argc; i++ ) {
    char *pngname = png ? strdup(png) : pngsuff(argv[i]);
    int code = fitspng(argv[i],pngname,bit_depth,qblack,rsense,scale,f0,t,s,st,
		       ss,type,satur,dcspace,fz,verb,!set_fl,set_fn,set_satur);
    if( code != 0 ) {
      status = 2;
      fprintf(stderr,"Error: fitspng failed for `%s'.\n",argv[i]);
    }

    free(pngname);
  }

  return status;
}


void help()
{
  printf("%s\n\n%s\n\n%s\n%s\n%s\n%s\n %s\n%s\n %s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n",
	 "FITSPNG    FITS to PNG converter",
	 "Usage: fitspng [options] file(s)",
	 "options:",
	 "\t-f x\t Itt tone function: linear, sqrt, sqr, logistic",
	 "\t-f0 f\t f0 parameter: f0*f(t)+z0 (default: 1)",
	 "\t-fz z\t zero z0 parameter: f0*f(t)+z0 (default: 0)",
	 "\t-fr q,v\t relative scaling parameters: 0<=q<=1, v>0",
	 "\t-fl B,s\t scale parameters: (x-B)/s",
	 "\t-fs x\t colour saturation (default: 1)",
	 "\t-fn tn,sn threshold and width for night/day vision (no default)",
	 "\t-cs x\t output colour-space: sRGB (default) or AdobeRGB",
	 "\t-s s\t zoom-out image by 1/s, to create thumbnails (default: 1)",
	 "\t-o\t output filename (considered for single file input)",
	 "\t-B\t bits per pixel of output image: 8,16 (default: 8)",
	 "\t-v\t verbose mode",
	 "\t--help\t\t give this help",
	 "\t--version\t display software version");
}


void version()
{
  fprintf(stdout,"FITSPNG, %s, (C) 2006-2019 F. Hroch,"
	  " Masaryk University in Brno, CZ\n",VERSION);
  fprintf(stdout,"FITSPNG comes with ABSOLUTELY NO WARRANTY.\n");
  fprintf(stdout,"You may redistribute copies of FITSPNG\n");
  fprintf(stdout,"under the terms of the GNU General Public License.\n");
  fprintf(stdout,"For more information about these matters, see the file named COPYING.\n");
}

char *pngsuff(const char *fitspath)
{
  char *png = 0;
  const char *fits = 0, *dot = 0;

  /* extracting basename */
  if( (fits = strrchr(fitspath,DIRSEP)) == NULL )
    fits = fitspath;
  else
    fits++;

  /* suffix like ".fits" is replaced by ".png" */
  if( (png = malloc(strlen(fits) + 5)) == NULL )
    perror(__FILE__);
  *png = '\0'; /* should be initialised with null-terminating byte */
  assert(png && fits);
  if( (dot = strchr(fits,'.')) != NULL )
    strncat(png,fits,dot-fits);
  else
    strcat(png,fits);
  strcat(png,".png");

  return png;
}
