// C++ implementation of Cromer-Liberman calculation of anomalous scattering
// factors, with corrections from Kissel & Pratt, Acta Cryst. A46, 170 (1990).
// Single header. No dependencies.
//
// Based on the Fortran implementation from Larch [1] (BSD license),
// which is based on the Cowan-Brennan routines [2],
// which was based on the original code from Don T. Cromer [3].
// [1] https://github.com/xraypy/xraylarch directory dylibs/CromerLiberman
// [2] Rev. Sci. Instrum. 63, 850 (1992), https://doi.org/10.1063/1.1142625
// [3] Acta Cryst. A37, 267 (1981), https://doi.org/10.1107/S0567739481000600
//
// Note: Matthew Newville, who maintains Larch, recommended to use
// data from Chris Chantler instead. This data is available from
// https://github.com/xraypy/XrayDB and from the NIST website.
// Use that for better accuracy. This implementation is a trade-off
// between the accuracy and the compactness of data (so we don't need
// separate data files).

#ifndef GEMMI_FPRIM_HPP_
#define GEMMI_FPRIM_HPP_

#include <cassert>
#include <cmath>      // for sqrt, fabs, log, exp
#include <algorithm>  // for sort

namespace gemmi {

namespace impl_fprim {

inline double pow2(double x) { return x * x; }

struct Gaus {
  double xsb, bena, energa, xsedga;
};

typedef double (*sigma_func)(double, double, const Gaus&);

inline double gauss(sigma_func sigma, const double xs_int[], const Gaus& g) {
  return 0.11846344252810 * (*sigma)(0.04691007703067, xs_int[0], g) +
         0.23931433524968 * (*sigma)(0.23076534494716, xs_int[1], g) +
         0.28444444444444 * (*sigma)(0.5, xs_int[2], g) +
         0.23931433524968 * (*sigma)(1. - 0.23076534494716, xs_int[3], g) +
         0.11846344252810 * (*sigma)(1. - 0.04691007703067, xs_int[4], g);
}

inline double sigma0(double x, double xs_int, const Gaus& g) {
  double s = xs_int * g.bena / (x * x);
  double d_prod = pow2(g.energa * x) - pow2(g.bena);
  if (std::fabs(d_prod) < 1e-30)
    return s;
  return g.bena * (s * g.bena - g.xsb * pow2(g.energa)) / d_prod;
}

inline double sigma1(double x, double xs_int, const Gaus& g) {
  return g.bena * g.bena * g.bena * 0.5 * xs_int /
          (x * std::sqrt(x) * (g.energa * g.energa * x - g.bena * g.bena));
}

inline double sigma2(double x, double xs_int, const Gaus& g) {
  if (std::fabs(x) < 1e-18 || g.energa < 1e-18)
    return 0.;
  if (std::fabs(xs_int - g.xsb) < 1e-18)
    return -2 * xs_int * g.bena / (x * x * x);
  double denom = x * x * x * pow2(g.energa) - pow2(g.bena) / x;
  if (std::fabs(denom) < 1e-5)
    denom = x * x * x * pow2(g.energa * 1.001) - pow2(g.bena) / x;
  double bx = g.bena / x;
  return 2 * (xs_int * (bx * bx * bx) / x
              - g.bena * g.xsb * pow2(g.energa)) / denom;
}

inline double sigma3(double x, double xs_int, const Gaus& g) {
  double x2 = x * x;
  return g.bena * g.bena * g.bena * (xs_int - g.xsedga * x2)
            / (x2 * (x2 * pow2(g.energa) - pow2(g.bena)));
}

struct Point {
  double x, y;
  bool operator<(const Point& o) const { return x < o.x; }
};

// Aitken repeated interpolation.
inline double aknint(double x, int n, const Point *data) {
  assert(n >= 3);
  int k = 1;
  while (k < n - 2 && (data[1].x > data[0].x ? data[k].x < x : data[k].x > x))
    ++k;
  double a[3], b[3];
  for (int i = 0; i < 3; ++i) {
    a[i] = data[k+i-1].y;
    b[i] = data[k+i-1].x - x;
  }
  for (int j = 0; j < 2; ++j)
    a[j+1] = (a[0] * b[j+1] - a[j+1] * b[0]) / (b[j+1] - b[0]);
  return (a[1] * b[2] - a[2] * b[1]) / (b[2] - b[1]);
}


#ifdef GEMMI_FPRIM_ALL_DOUBLE
  typedef double float_data_type;
#else
  typedef float float_data_type;
// Compilers with -Wfloat-conversion can complain
# if __GNUC__-0 > 4
#  pragma GCC diagnostic push
#  pragma GCC diagnostic ignored "-Wfloat-conversion"
# endif
# if defined(_MSC_VER)
#  pragma warning(push)
   // warning C4838: conversion from double to float
#  pragma warning(disable: 4838)
   // warning C4305: 'initializing': truncation from 'double' to 'float' ...
#  pragma warning(disable: 4305)
# endif
#endif

struct OrbitalCoef {
  short nparm;
  float_data_type binden;
  float_data_type xnrg[6];
  float_data_type xsc[11];
};

inline void cromer(int z, double energy_ev, int norb, const OrbitalCoef coefs[],
                   double *f1, double *f2) {
  const double finepi = 3.47116243; // = 1/(4*alpha*pi**2)
  const double fscinv = 137.036;
  const double au = 28002200.;
  const double kev2ry = 0.02721;
  const double fourpi = 12.566370614359172;

  // part of xnrg that is always the same
  // log values of { 80., 26.7, 8.9, 3., 1. }
  static const double ln_xnrdat[5] = {
    4.382026634673881, 3.2846635654062037, 2.186051276738094,
    1.0986122886681098, 0.0
  };

  double energy_kev = 0.001 * energy_ev;
  double lne = std::log(energy_kev);
  Gaus g;
  g.energa = energy_kev / kev2ry;
  *f1 = 0.;
  *f2 = 0.;

  for (int i = 0; i < norb; ++i) {
    const OrbitalCoef& orbc = coefs[i];
    g.xsb = 0.;
    g.bena = orbc.binden / kev2ry;
    int nparm = orbc.nparm;

    // GCC-12 prints bogus -Warray-bounds warning (about std::__insertion_sort
    // code) when sorting lndata[11]. To avoid it we use here lndata[16] items.
    //Point lndata[11];
    Point lndata[16];
    for (int j = 0; j < 5; ++j)
      lndata[j].x = ln_xnrdat[j];
    // it could be optimized by storing log values in coefs[]
    for (int j = 5; j < nparm; ++j)
      lndata[j].x = std::log(orbc.xnrg[j-5]);
    for (int j = 0; j < nparm; ++j)
      lndata[j].y = orbc.xsc[j] > 1e-9 ? std::log(orbc.xsc[j]) : 0.;
    std::sort(lndata, lndata + nparm);

    double f1orb = 0.;
    double f1corr = 0.;
    double f2orb = 0.;
    if (g.bena <= g.energa) {
      int i0 = 0;
      while (i0 < nparm && std::fabs(lndata[i0].y) < 1e-9)
        ++i0;
      int inxs = nparm - i0;
      double akn = aknint(lne, inxs, &lndata[i0]);
      g.xsb = std::exp(akn) / au;
      f2orb = g.energa * fscinv * g.xsb / fourpi;
      double var = g.energa - g.bena;
      if (std::fabs(var) <= 1e-9)
        var = 1.;
      f1corr = -finepi * g.xsb * g.energa *
               std::log((g.energa + g.bena) / var);
    }

    double xs_int[5];
    for (int j = 5; j < 10; ++j)
      xs_int[j - 5] = orbc.xsc[j] / au;

    if (g.bena > g.energa && nparm == 11) {
      g.xsedga = orbc.xsc[10] / au;
      f1orb = gauss(sigma3, xs_int, g);
      f1corr = g.xsedga * finepi * pow2(g.bena) *
               std::log((-g.bena + g.energa) / (-g.bena - g.energa)) / g.energa;
    } else if (nparm == 11) {
      f1orb = gauss(sigma0, xs_int, g);
    } else if (nparm == 10 && z >= 79 && i == 0) {
      f1orb = gauss(sigma1, xs_int, g);
    } else {
      f1orb = gauss(sigma2, xs_int, g);
    }
    *f1 += 2 * finepi * f1orb + f1corr;
    *f2 += f2orb;
  }
}

// This is comment from the Fortran code in Larch:
//  Note: the Jensen correction to f' was subsequently shown to be incorrect
//  (see L. Kissel and R.H. Pratt, Acta Cryst. A46, 170 (1990))
//  and that the relativistic correction that Ludwig used is also
//  wrong. (...)
//  Kissel and Pratt give better corrections. The relativistic correction
//  that Ludwig used is (5/3)(e_tot/mc^2). Kissel and Pratt say that this
//  should be simply (e_tot/mc^2), but their correction (kpcor) apparently
//  takes this into account. So we can use the old relcor and simply add
//  the (energy independent) kpcor term.
inline double calculate_correction(int z) {
  static float_data_type kpcor_minus_relcor[92] = {
    0, 0, 0, -0.001, -0.001, -0.002, -0.003, -0.004, -0.005, -0.007, -0.008,
    -0.01, -0.013, -0.015, -0.018, -0.021, -0.024, -0.027, -0.031, -0.035,
    -0.04, -0.044, -0.049, -0.054, -0.06, -0.065, -0.071, -0.078, -0.085,
    -0.092, -0.099, -0.107, -0.115, -0.123, -0.132, -0.141, -0.15, -0.16,
    -0.17, -0.181, -0.191, -0.203, -0.214, -0.226, -0.238, -0.251, -0.264,
    -0.277, -0.291, -0.305, -0.32, -0.335, -0.35, -0.366, -0.382, -0.398,
    -0.415, -0.432, -0.45, -0.468, -0.4879, -0.5062, -0.5252, -0.545, -0.565,
    -0.586, -0.607, -0.629, -0.651, -0.674, -0.697, -0.721, -0.745, -0.77,
    -0.795, -0.821, -0.847, -0.874, -0.901, -0.929, -0.957, -0.986, -1.015,
    -1.045, -1.076, -1.107, -1.139, -1.171, -1.204, -1.238, -1.272, -1.306
  };
  return kpcor_minus_relcor[z-1];
}

inline OrbitalCoef* get_orbital_coefficients(int z, int* n_orb) {
  static const short indices[91] = {  // from Z=3
    0, 2, 4, 7, 10, 14, 18, 22, 26, 30, 34, 39, 45, 52, 59, 66, 73, 80, 87,
    94, 101, 109, 118, 127, 136, 145, 154, 163, 172, 181, 190, 199, 208, 217,
    226, 235, 247, 259, 272, 285, 299, 313, 327, 341, 355, 369, 383, 397, 411,
    425, 439, 453, 467, 484, 501, 518, 536, 554, 572, 590, 608, 626, 645, 664,
    683, 702, 721, 740, 759, 778, 798, 818, 838, 859, 880, 901, 922, 943, 965,
    987, 1010, 1033, 1057, 1081, 1105, 1129, 1153, 1177, 1201, 1225, 1249
  };

  static OrbitalCoef coefs[] = {
// Z=3,  2 orbitals
{10, 0.05475, {24.8800831, 1.0281179, 0.218999997, 0.09252656251, 0.06027210504, 0},
 {0.0013015531, 0.05167718977, 2.04572558, 73.6826706, 2367.26001, 0.06531799585, 2174.78979, 189729.859, 1618910.88, 3022227, 0}},
{10, 0.00534, {2.4266603, 0.100276709, 0.0213600006, 0.009024509229, 0.005878594704, 0},
 {2.456936636e-05, 0.000866183138, 0.03219022229, 1.14365172, 35.4625931, 2.25543308, 17745.8359, 440576.219, 1333729.63, 1448481.13, 0}},

// Z=4,  2 orbitals
{10, 0.111, {50.4418144, 2.08440351, 0.444000006, 0.187588111, 0.122195505, 0},
 {0.005501731299, 0.21635069, 8.28356647, 277.734161, 7976.125, 0.0256038066, 866.962646, 81105.0859, 771925.063, 2051785.63, 0}},
{10, 0.00842, {3.82630682, 0.15811421, 0.03367999941, 0.0142296562, 0.009269244038, 0},
 {0.0002081731509, 0.007329026237, 0.263835013, 8.6520052, 237.105286, 4.02921534, 31112.8418, 621490.375, 1648233.5, 1031229.06, 0}},

// Z=5,  3 orbitals
{10, 0.188, {85.4329758, 3.53034091, 0.751999974, 0.317716777, 0.206961751, 0},
 {0.01651980355, 0.639404237, 23.6173916, 741.657715, 19577.5059, 0.01335961837, 447.89505, 43871.5781, 435450.094, 1167093.38, 0}},
{10, 0.01347, {6.12118196, 0.252945185, 0.05387999862, 0.02276406996, 0.01482858974, 0},
 {0.0008280635811, 0.02931262553, 1.02559197, 31.2025394, 776.768555, 3.395962, 26979.2715, 581240.375, 1365937.5, 870618.875, 0}},
{10, 0.0047, {2.13582444, 0.08825852722, 0.01879999973, 0.007942919619, 0.005174043588, 0},
 {1.49853463e-07, 1.31118104e-05, 0.001204593806, 0.10465923, 7.67965651, 0.400328189, 30321.8242, 1377606.13, 6894477, 14448549, 0}},

// Z=6,  3 orbitals
{10, 0.2838, {128.967453, 5.3293128, 1.13520002, 0.479617149, 0.312424183, 0},
 {0.0402277112, 1.52750432, 54.3804855, 1610.34399, 39562.0547, 0.008320189081, 275.130463, 27737.3262, 281872.813, 765563.438, 0}},
{10, 0.01951, {8.86594391, 0.366366804, 0.07804000378, 0.03297156841, 0.02147778682, 0},
 {0.002234945307, 0.07857172936, 2.65592504, 75.4173965, 1707.09705, 2.68837976, 21804.5703, 492388.406, 1077516.38, 678007.188, 0}},
{10, 0.0064, {2.90835667, 0.120181821, 0.02559999935, 0.01081589051, 0.0070455065, 0},
 {1.561928684e-06, 0.0001310063381, 0.01182175893, 0.94114393, 63.7416344, 1.06329751, 76115.4297, 3070108, 11207961, 18938980, 0}},

// Z=7,  4 orbitals
{10, 0.4016, {182.49939, 7.54140949, 1.60640001, 0.678697169, 0.442105532, 0},
 {0.08473966271, 3.15485168, 108.397171, 3054.15259, 70342.9141, 0.005561249331, 182.452774, 18816.3184, 196109.391, 545683.125, 0}},
{10, 0.02631, {11.9560728, 0.49405998, 0.105240002, 0.04446345195, 0.02896363661, 0},
 {0.004933742341, 0.170914933, 5.56366348, 148.861649, 3093.78003, 2.19790149, 17981.0859, 406345.625, 854337.875, 509074.75, 0}},
{10, 0.0092, {4.18076277, 0.172761381, 0.03680000082, 0.01554784272, 0.010127916, 0},
 {5.528737347e-06, 0.000451105152, 0.03958846256, 2.99486256, 182.204178, 0.817081749, 60970.375, 2508748.5, 7526410.5, 10705870, 0}},
{10, 0.0092, {4.18076277, 0.172761381, 0.03680000082, 0.01554784272, 0.010127916, 0},
 {2.35968605e-06, 0.000212471612, 0.0193909537, 1.48126984, 90.318634, 0.402362198, 30338.6758, 1251751.25, 3764252.5, 5362577, 0}},

// Z=8,  4 orbitals
{10, 0.532, {241.757156, 9.99011421, 2.12800002, 0.899070919, 0.585657716, 0},
 {0.160504997, 5.85706615, 194.312134, 5221.68848, 111657.734, 0.0042040525, 135.399689, 14075.4473, 147497.984, 417832.938, 0}},
{10, 0.0237, {10.770009, 0.445048332, 0.09480000287, 0.04005259648, 0.02609039098, 0},
 {0.009546677582, 0.325384855, 10.1911001, 256.266083, 4902.52197, 5.65027618, 33413.8359, 514644.063, 839019.875, 466361.438, 0}},
{10, 0.0071, {3.22645831, 0.133326709, 0.02840000018, 0.01199887879, 0.007816108875, 0},
 {1.560978126e-05, 0.001245745225, 0.105102435, 7.58270359, 419.3974, 5.73374319, 230141.391, 3959252.5, 8621531, 10632627, 0}},
{10, 0.0071, {3.22645831, 0.133326709, 0.02840000018, 0.01199887879, 0.007816108875, 0},
 {1.317946135e-05, 0.001174610457, 0.102651693, 7.48369217, 415.102356, 5.65493631, 228981.328, 3953836.5, 8638009, 10673776, 0}},

// Z=9,  4 orbitals
{10, 0.6854, {311.466827, 12.8707218, 2.74160004, 1.15831435, 0.754529715, 0},
 {0.280725479, 10.0407209, 322.387848, 8296.8457, 164967.891, 0.003362095449, 102.271927, 10743.9775, 113532.219, 328530.594, 0}},
{10, 0.031, {14.0873537, 0.58213073, 0.124000005, 0.05238947272, 0.03412667289, 0},
 {0.01684946194, 0.564888954, 17.0781956, 405.708954, 7241.59912, 4.16125727, 25509.5801, 409591.969, 663627.438, 342294.719, 0}},
{10, 0.0086, {3.90810442, 0.16149433, 0.03440000117, 0.01453385316, 0.009467399679, 0},
 {3.755814032e-05, 0.002953280695, 0.240616351, 16.3673668, 847.311951, 6.02639341, 224444.438, 3326680.75, 6272801.5, 6917325.5, 0}},
{10, 0.0086, {3.90810442, 0.16149433, 0.03440000117, 0.01453385316, 0.009467399679, 0},
 {4.756415365e-05, 0.004174075089, 0.351481318, 24.1645908, 1255.78772, 8.86773872, 334560.344, 4983429, 9439702, 10439668, 0}},

// Z=10,  4 orbitals
{10, 0.8669, {393.946014, 16.2790031, 3.46759987, 1.46504617, 0.954335868, 0},
 {0.461048841, 16.1760311, 504.382843, 12496.9648, 235837.344, 0.002751233522, 77.5974884, 8264.11914, 88570.9688, 262354.688, 0}},
{10, 0.045, {20.4493828, 0.84502846, 0.180000007, 0.07604923099, 0.04953871667, 0},
 {0.02783099376, 0.915888429, 26.7597961, 604.638611, 10104.3848, 2.09819627, 14937.3867, 281951.719, 503242.344, 240853.344, 0}},
{10, 0.0183, {8.31608295, 0.343644917, 0.07320000231, 0.03092668764, 0.02014574595, 0},
 {8.132271614e-05, 0.00627292227, 0.495196134, 31.8960037, 1569.13635, 0.647712648, 45953.0586, 1562452.63, 3068620.5, 2868416, 0}},
{10, 0.0183, {8.31608295, 0.343644917, 0.07320000231, 0.03092668764, 0.02014574595, 0},
 {0.0001363403426, 0.01177385636, 0.960985005, 62.6071243, 3095.17505, 1.25661778, 91011.9609, 3113751.25, 6153211, 5780000.5, 0}},

// Z=11,  4 orbitals
{11, 1.0721, {22.8543644, 4.64584494, 2.14420009, 1.39372301, 1.12486768, 1.07317209},
 {0.719678938, 24.7575035, 749.831604, 17837.6504, 0, 40.4996109, 5139.83252, 44760.4258, 137345.703, 242031.281, 194894.047}},
{10, 0.0633, {28.7654667, 1.18867338, 0.253199995, 0.10697592, 0.06968446076, 0},
 {0.04549028352, 1.47548819, 41.757679, 896.968445, 13934.4736, 1.17296302, 9320.54199, 201357.344, 518224.594, 355563.563, 0}},
{10, 0.0311, {14.1327963, 0.584008574, 0.124399997, 0.05255847052, 0.03423675895, 0},
 {0.0001788535155, 0.01378671546, 1.06470704, 65.0138245, 2978.45313, 0.172413975, 16576.0566, 938347.875, 2437551.5, 1397262.13, 0}},
{10, 0.0311, {14.1327963, 0.584008574, 0.124399997, 0.05255847052, 0.03423675895, 0},
 {0.0003001188161, 0.02587055042, 2.06276727, 127.485764, 5874.59863, 0.329745412, 32774.9375, 1869307.13, 4888040, 2814550, 0}},

// Z=12,  4 orbitals
{11, 1.305, {27.8191833, 5.6550951, 2.6099999, 1.69649148, 1.36923075, 1.30630505},
 {1.0774889, 36.3590317, 1071.03394, 24479.2656, 0, 31.9452724, 4077.84326, 35751.1758, 110366.938, 192795.563, 195278.125}},
{10, 0.0894, {40.6261101, 1.67878985, 0.357600003, 0.151084468, 0.09841691703, 0},
 {0.07123360038, 2.27467704, 62.3694611, 1277.60583, 18539.1563, 0.615424693, 5559.92969, 139329.063, 436742.563, 487870.688, 0}},
{10, 0.0514, {23.3577404, 0.965210319, 0.205600008, 0.08686511964, 0.0565842241, 0},
 {0.0003576920135, 0.02731011063, 2.05052972, 119.007393, 5120.1167, 0.0463600345, 5738.63086, 450838.188, 2111516.75, 1120995.88, 0}},
{10, 0.0514, {23.3577404, 0.965210319, 0.205600008, 0.08686511964, 0.0565842241, 0},
 {0.0005990033969, 0.05106269196, 3.96210837, 232.971024, 10091.8086, 0.08693627268, 11313.0215, 896493.5, 4229249, 2257596.5, 0}},

// Z=13,  5 orbitals
{11, 1.5596, {33.2465858, 6.75838041, 3.11919999, 2.02746964, 1.63636184, 1.56115961},
 {1.5575105, 51.5725327, 1478.3385, 32428.127, 0, 25.9100971, 3315.86548, 29194.6914, 90483.5703, 157688.125, 167248.469}},
{10, 0.1177, {53.4864998, 2.21021891, 0.470800012, 0.198910996, 0.129571274, 0},
 {0.107318684, 3.36706495, 89.3895264, 1748.37085, 23709.1934, 0.384381026, 3783.99194, 104031.25, 369606.75, 494166.094, 0}},
{10, 0.0731, {33.2188873, 1.37270176, 0.292400002, 0.123537749, 0.08047289401, 0},
 {0.0006674015895, 0.05023531988, 3.65229011, 201.835403, 8114.76563, 0.02148255892, 2929.53931, 269588.344, 1592395.75, 1433475.88, 0}},
{10, 0.0731, {33.2188873, 1.37270176, 0.292400002, 0.123537749, 0.08047289401, 0},
 {0.001112943166, 0.0934844017, 7.03663921, 394.388367, 15981.2051, 0.0394811146, 5758.07031, 535215.063, 3186256, 2885157.25, 0}},
{10, 0.0083757, {3.80615973, 0.157281682, 0.03350266069, 0.0141547313, 0.009220438078, 0},
 {0.007879838347, 0.224162638, 5.65236759, 109.529068, 1489.20581, 58.8560448, 45791.7344, 266193.563, 71344.7734, 347980.688, 0}},

// Z=14,  6 orbitals
{11, 1.8389, {39.200531, 7.96870089, 3.67779994, 2.390558, 1.92940867, 1.84073889},
 {2.1863687, 71.0673218, 1984.17834, 41781.957, 0, 21.3957233, 2740.57544, 24228.9063, 75374.5, 131083.328, 142535.563}},
{10, 0.1487, {67.5738525, 2.79234958, 0.594799995, 0.251300454, 0.163697943, 0},
 {0.156227812, 4.81014633, 123.655083, 2312.4209, 29410.8008, 0.263393164, 2770.06958, 81446.5391, 310726.656, 444896.5, 0}},
{10, 0.0992, {45.0795288, 1.86281824, 0.396800011, 0.167646304, 0.10920535, 0},
 {0.001176461577, 0.08714818954, 6.12822819, 323.639862, 12196.46, 0.0112606911, 1652.95496, 172008.016, 1209017.13, 1656120.25, 0}},
{10, 0.0992, {45.0795288, 1.86281824, 0.396800011, 0.167646304, 0.10920535, 0},
 {0.001959844725, 0.161601678, 11.7720785, 631.226135, 24000.7852, 0.02011139318, 3237.92676, 340892.344, 2417061, 3331674.75, 0}},
{10, 0.011357, {5.1610465, 0.213269562, 0.04542867094, 0.01919342019, 0.01250265632, 0},
 {0.01411816571, 0.401187301, 9.87669754, 183.10257, 2358.03467, 44.4429398, 42208.6055, 313736.125, 162653.281, 80005.2031, 0}},
{10, 0.0050831, {2.30989599, 0.0954516679, 0.02033221535, 0.00859027449, 0.00559573248, 0},
 {6.249973376e-05, 0.003996924497, 0.261709213, 13.4452877, 475.203278, 32.8313789, 122243.977, 216556.953, 11214985, 32290612, 0}},

// Z=15,  7 orbitals
{11, 2.1455, {45.7364426, 9.29732323, 4.29099989, 2.78913593, 2.25109935, 2.14764547},
 {2.99109888, 95.5345459, 2601.354, 52644.3086, 0, 17.7680607, 2290.75854, 20345.6113, 63556.0977, 110514.297, 121777.781}},
{10, 0.1893, {86.023735, 3.55475307, 0.757200003, 0.319913775, 0.208392873, 0},
 {0.220793724, 6.66793537, 166.183395, 2980.1311, 35851.7344, 0.176890522, 1945.68689, 61948.0781, 254934.203, 385898.719, 0}},
{10, 0.1322, {60.0757446, 2.4825058, 0.528800011, 0.223415747, 0.145533741, 0},
 {0.00197823206, 0.144055828, 9.80049324, 496.497589, 17670.9141, 0.00606521545, 949.569885, 110775.891, 887976.25, 1629468.25, 0}},
{10, 0.1322, {60.0757446, 2.4825058, 0.528800011, 0.223415747, 0.145533741, 0},
 {0.003278829157, 0.265864342, 18.7691917, 966.596558, 34750.7227, 0.01049841568, 1852.47876, 219093.813, 1773571.75, 3276616.5, 0}},
{10, 0.0144615, {6.57174635, 0.271563828, 0.05784596503, 0.02443967387, 0.01592008211, 0},
 {0.02243389189, 0.631072283, 15.1140366, 268.409637, 3296.84058, 34.8709259, 38114.8672, 321233.156, 227968.109, 4926.22803, 0}},
{10, 0.00638493, {2.90150857, 0.119898833, 0.02553972043, 0.01079042256, 0.007028916851, 0},
 {0.0001338380243, 0.008599226363, 0.548111379, 26.8109093, 882.155823, 30.0270767, 129890.227, 204304.078, 12901790, 30727232, 0}},
{10, 0.00633669, {2.87958717, 0.118992977, 0.02534676343, 0.01070889924, 0.006975811906, 0},
 {5.532951764e-05, 0.003981444519, 0.261431783, 12.9882307, 431.338715, 14.9381475, 65373.6758, 102237.445, 6433917.5, 15425960, 0}},

// Z=16,  7 orbitals
{11, 2.472, {52.6965675, 10.7121801, 4.94400024, 3.21358371, 2.59366918, 2.47447205},
 {4.0029788, 125.645302, 3336.60425, 64908.2656, 0, 15.0581884, 1949.6781, 17367.4648, 54399.3633, 94464.5781, 105087.281}},
{10, 0.2292, {104.155533, 4.30401182, 0.916800022, 0.387344092, 0.25231722, 0},
 {0.303693503, 8.99559212, 217.438217, 3737.79248, 42614.5664, 0.130291492, 1515.5553, 50511.7813, 218041.891, 344550.156, 0}},
{10, 0.1648, {74.8901901, 3.09468198, 0.659200013, 0.2785092, 0.181421801, 0},
 {0.003195327241, 0.228410363, 15.0264416, 729.671326, 24527.0723, 0.004100469407, 656.268738, 81419.5938, 699896, 1532038, 0}},
{10, 0.1648, {74.8901901, 3.09468198, 0.659200013, 0.2785092, 0.181421801, 0},
 {0.00527478708, 0.419763833, 28.6901493, 1417.97913, 48208.3945, 0.006840255111, 1274.83459, 160726.359, 1396900.5, 3080348.75, 0}},
{10, 0.0176882, {8.0380764, 0.332156867, 0.07075292617, 0.02989280969, 0.01947227307, 0},
 {0.03336768597, 0.925767303, 21.5492477, 366.828888, 4318.39111, 28.5174408, 34478.0508, 311271.25, 263840.156, 5097.19873, 0}},
{10, 0.00781363, {3.55075431, 0.146727577, 0.03125452623, 0.01320490334, 0.00860171672, 0},
 {0.0002531304199, 0.01616474427, 0.997793913, 46.4188728, 1431.60376, 26.2621441, 127479.82, 190660.828, 13336919, 26273610, 0}},
{10, 0.00773488, {3.51496625, 0.145248711, 0.03093951195, 0.01307181176, 0.008515020832, 0},
 {0.0002077342215, 0.01485415734, 0.94839263, 44.8675652, 1397.84741, 26.2266979, 128772.211, 192191.422, 13350413, 26480780, 0}},

// Z=17,  7 orbitals
{11, 2.8224, {60.1661758, 12.2306061, 5.64479971, 3.66910148, 2.96131539, 2.82522225},
 {5.25414228, 162.141266, 4201.44678, 78930.8828, 0, 12.9650497, 1677.90186, 14986.1406, 47054.5273, 81640.5781, 91375.2969}},
{10, 0.2702, {122.787186, 5.07392645, 1.08080006, 0.456633389, 0.29745248, 0},
 {0.408017427, 11.8550882, 278.075928, 4582.91162, 49648.3242, 0.106112912, 1241.04224, 42662.8477, 190254.953, 310957.031, 0}},
{10, 0.2016, {91.6132355, 3.7857275, 0.806400001, 0.340700567, 0.221933454, 0},
 {0.004981699865, 0.349480033, 22.2610455, 1037.18677, 33091.1836, 0.002997984877, 469.98114, 61464.7539, 560663.125, 1384550.63, 0}},
{10, 0.2, {90.8861542, 3.75568223, 0.800000012, 0.337996602, 0.220172077, 0},
 {0.008185919374, 0.63939172, 42.3526001, 2009.42236, 64820.0625, 0.004982291255, 933.104126, 123397.305, 1131530.13, 2813498, 0}},
{10, 0.0175, {7.95253801, 0.328622192, 0.0700000003, 0.02957470156, 0.01926505752, 0},
 {0.04733937234, 1.29461861, 29.2959709, 478.582855, 5403.11523, 39.7703285, 42692.8984, 342222.719, 275203.469, 19442.5215, 0}},
{10, 0.0068, {3.09012914, 0.127693191, 0.02720000036, 0.01149188355, 0.007485850714, 0},
 {0.0004393915879, 0.02781425416, 1.66258872, 73.4911575, 2122.91431, 66.6478119, 197430.234, 462803.625, 20593570, 28671536, 0}},
{10, 0.0068, {3.09012914, 0.127693191, 0.02720000036, 0.01149188355, 0.007485850714, 0},
 {0.0005393139436, 0.03811880946, 2.36163688, 106.294724, 3105.21143, 96.364151, 295091.563, 665788.375, 30513632, 43049564, 0}},

// Z=18,  7 orbitals
{11, 3.2029, {68.2774429, 13.879467, 6.40580034, 4.16374922, 3.36054349, 3.20610285},
 {6.78045702, 205.872787, 5211.41455, 0, 0, 11.2206945, 1451.25696, 13007.5967, 40969.3633, 71071.4063, 79858.5313}},
{10, 0.314, {145.417831, 6.00909138, 1.27999997, 0.540794551, 0.352275312, 0},
 {0.537745595, 15.3215637, 349.145355, 5530.56934, 57249.9609, 0.08058060706, 990.483459, 35425.1172, 163749.75, 276381.094, 0}},
{10, 0.2473, {112.380722, 4.64390087, 0.989199996, 0.417932779, 0.272242785, 0},
 {0.007535587996, 0.518801868, 32.0660934, 1437.97949, 43965.0234, 0.00203356077, 325.793732, 45366.8281, 443202.438, 1192494.38, 0}},
{10, 0.2452, {111.426414, 4.60446596, 0.980799973, 0.414383799, 0.269930959, 0},
 {0.0123210391, 0.945011258, 60.8128281, 2780.07568, 86040.3672, 0.003250034759, 644.774658, 91015.125, 894592.375, 2427436.25, 0}},
{10, 0.0253, {11.497098, 0.475093782, 0.101199999, 0.04275656864, 0.02785176784, 0},
 {0.06486418098, 1.74860966, 38.5277634, 606.492554, 6609.77051, 19.1251431, 27166.3066, 268914.438, 284672.375, 42450.8164, 0}},
{10, 0.0124, {5.6349411, 0.23285228, 0.04960000142, 0.02095578797, 0.01365066878, 0},
 {0.0007182126865, 0.04501340538, 2.61000037, 110.124474, 3000.1792, 13.2102203, 91612.4297, 198423.344, 10275668, 15258303, 0}},
{10, 0.0124, {5.6349411, 0.23285228, 0.04960000142, 0.02095578797, 0.01365066878, 0},
 {0.001170826145, 0.08175848424, 4.92371416, 211.84494, 5842.46826, 25.1585808, 181295.203, 407273, 20135490, 30561258, 0}},

// Z=19,  7 orbitals
{11, 3.6074, {76.900322, 15.6323299, 7.21479988, 4.68959618, 3.7849524, 3.61100745},
 {8.61559391, 257.381226, 6361.01416, 0, 0, 9.75314999, 1266.26306, 11374.3389, 35897.2773, 62242.4883, 70476.5469}},
{10, 0.3771, {171.365829, 7.08133841, 1.50839996, 0.637292564, 0.41513446, 0},
 {0.695426166, 19.4439659, 430.745331, 6562.66797, 64891.7344, 0.06338828057, 788.96814, 29275.7754, 139685.906, 254293.172, 0}},
{10, 0.2963, {134.647827, 5.56404305, 1.18519998, 0.500741959, 0.326184928, 0},
 {0.01109857671, 0.749247313, 44.9230194, 1933.97839, 56408.4883, 0.001546652056, 238.654984, 34589.043, 352950.938, 918931.813, 0}},
{10, 0.2936, {133.420868, 5.51334143, 1.17439997, 0.496178985, 0.323212624, 0},
 {0.01805134118, 1.35872793, 84.9236145, 3731.17749, 110303.227, 0.002377696102, 470.76355, 69328.625, 712621.875, 1866130.5, 0}},
{10, 0.0339, {15.4052029, 0.636588097, 0.135600001, 0.05729041994, 0.03731916845, 0},
 {0.08954766393, 2.38756752, 51.3165588, 778.871826, 8213.99219, 11.3997135, 19505.7285, 222586.125, 398011.063, 198676.953, 0}},
{10, 0.0178, {8.08886719, 0.334255695, 0.07119999826, 0.03008169681, 0.01959531568, 0},
 {0.001234294148, 0.07772114873, 4.40419769, 177.011063, 4530.33447, 6.19070959, 62596.0508, 371854.625, 2104738.75, 26572800, 0}},
{10, 0.0178, {8.08886719, 0.334255695, 0.07119999826, 0.03008169681, 0.01959531568, 0},
 {0.002018780215, 0.141051933, 8.31027031, 341.028473, 8845.39355, 11.7073212, 123850.641, 762296.438, 4081948, 52652676, 0}},

// Z=20,  7 orbitals
{11, 4.0381, {86.0817184, 17.4987278, 8.07619953, 5.24950361, 4.23685122, 4.0421381},
 {10.801218, 317.499695, 7660.65479, 0, 0, 8.5894022, 1112.51978, 10011.2959, 31645.1211, 54821.7813, 62620.7578}},
{10, 0.4378, {198.949783, 8.22118759, 1.75119996, 0.739874542, 0.481956691, 0},
 {0.886400521, 24.3066006, 523.445618, 7676.92627, 72531.1406, 0.05192638189, 645.348877, 24651.0664, 120790.688, 221150.344, 0}},
{10, 0.35, {159.050766, 6.57244349, 1.39999998, 0.591494024, 0.385301143, 0},
 {0.01598180458, 1.05744219, 61.5380554, 2544.06226, 70789.5391, 0.00115960429, 180.25058, 26956.6484, 284359, 774663.75, 0}},
{10, 0.3464, {157.41481, 6.50484133, 1.38559997, 0.585410118, 0.38133803, 0},
 {0.02587469853, 1.9095788, 115.956467, 4897.30762, 138282.609, 0.0017266043, 354.880554, 54049.7734, 574529.375, 1575394.13, 0}},
{10, 0.0437, {19.8586235, 0.820616543, 0.174799994, 0.07385225594, 0.04810759798, 0},
 {0.121023893, 3.18752408, 66.8213043, 979.660217, 10024.3721, 7.39177656, 14683.541, 186835.031, 421672.531, 383247.125, 0}},
{10, 0.0254, {11.5425415, 0.476971626, 0.101599999, 0.04292556643, 0.02796185389, 0},
 {0.001983818365, 0.123837739, 6.82274866, 261.554993, 6312.27588, 2.69190574, 38421.3984, 406731.313, 350113.156, 4626078.5, 0}},
{10, 0.0254, {11.5425415, 0.476971626, 0.101599999, 0.04292556643, 0.02796185389, 0},
 {0.003220621264, 0.223583385, 12.8490229, 503.650238, 12333.3105, 5.03576231, 75807.5547, 827790.938, 700751.563, 9119556, 0}},

// Z=21,  7 orbitals
{11, 4.4928, {95.7747269, 19.4691277, 8.98559952, 5.8406105, 4.71393061, 4.497293},
 {13.3754511, 387.105499, 9124.26074, 0, 0, 7.63245869, 985.740845, 8887.63965, 28145.1992, 48765.2773, 55765.875}},
{10, 0.5004, {227.397141, 9.39671612, 2.00160003, 0.845667481, 0.550870538, 0},
 {1.11466527, 29.9885063, 628.362183, 8892.73633, 80391.9453, 0.04445857555, 544.732727, 21299.0742, 106495.711, 198788.484, 0}},
{10, 0.4067, {184.816986, 7.63717937, 1.62679994, 0.68731606, 0.447719902, 0},
 {0.02259455808, 1.4645592, 82.830307, 3300.45557, 88189.5781, 0.0009547692607, 142.253525, 21883.9707, 237792.031, 660498.188, 0}},
{10, 0.4022, {182.772049, 7.55267668, 1.60880005, 0.679711163, 0.442766041, 0},
 {0.03633770719, 2.63203478, 155.574493, 6340.27686, 172182.266, 0.001371221151, 279.049683, 43849.1992, 480606.969, 1344320, 0}},
{10, 0.0538, {24.4483738, 1.01027846, 0.215200007, 0.0909210816, 0.05922628939, 0},
 {0.157317132, 4.07192469, 83.1233444, 1180.57031, 11781.1436, 5.22513533, 11549.5293, 158022.641, 381183.219, 393901.906, 0}},
{10, 0.0323, {14.678113, 0.606542647, 0.129199997, 0.05458644778, 0.03555779159, 0},
 {0.002940912731, 0.180802792, 9.67005634, 354.71637, 8119.83496, 1.61626744, 27698.6602, 365879.313, 277917.594, 1868140.75, 0}},
{10, 0.0323009, {14.678113, 0.606542647, 0.129199997, 0.05458644778, 0.03555779159, 0},
 {0.004759747069, 0.325096637, 18.151268, 681.726746, 15855.2959, 2.99188566, 54482.2539, 742810.5, 575778.188, 3671602.75, 0}},

// Z=22,  7 orbitals
{11, 4.9664, {105.870644, 21.5214291, 9.93280029, 6.45628738, 5.2108407, 4.97136641},
 {16.3783951, 466.729553, 10741.3164, 0, 0, 6.81414413, 881.864502, 7959.78369, 25236.7109, 43721.8906, 50025.1211}},
{10, 0.5637, {256.162628, 10.5853901, 2.25480008, 0.952643394, 0.620554984, 0},
 {1.38360548, 36.5285339, 744.958191, 10184.4697, 88024.4688, 0.03695865348, 473.029541, 18804.6387, 95396.0859, 180583.906, 0}},
{10, 0.4615, {209.719788, 8.66623592, 1.84599996, 0.779927135, 0.508047044, 0},
 {0.03134495765, 1.98984599, 109.354668, 4199.1748, 107379.313, 0.0008467737352, 120.095116, 18723.7754, 206431.688, 581895.938, 0}},
{10, 0.4555, {206.99321, 8.55356598, 1.82200003, 0.769787252, 0.501441896, 0},
 {0.05011180788, 3.55951881, 204.702057, 8047.52686, 209401.109, 0.001190659008, 235.652359, 37602.7813, 418222.25, 1187201.38, 0}},
{10, 0.0603, {27.402174, 1.13233817, 0.2412, 0.101905972, 0.06638187915, 0},
 {0.199400857, 5.07217216, 100.853806, 1388.92346, 13522.3828, 4.71482325, 10629.2871, 147303.156, 361211.375, 393500.281, 0}},
{10, 0.0346, {15.7233038, 0.649733007, 0.138400003, 0.05847340822, 0.03808977082, 0},
 {0.004219437949, 0.254280508, 13.171936, 462.252045, 10030.2881, 1.74592924, 28436.168, 360342.531, 275575.281, 1358419, 0}},
{10, 0.0346, {15.7233038, 0.649733007, 0.138400003, 0.05847340822, 0.03808977082, 0},
 {0.006784337573, 0.454848289, 24.6371078, 886.506653, 19570.9004, 3.21035886, 55870.5195, 732970.688, 575839.313, 2673432.25, 0}},

// Z=23,  8 orbitals
{11, 5.4651, {116.501617, 23.6824989, 10.9301996, 7.10459423, 5.73408651, 5.47056532},
 {19.8593178, 557.293579, 12524.9434, 0, 0, 6.15256977, 793.15094, 7166.36816, 22744.7285, 39397.2227, 45205.3203}},
{10, 0.6282, {285.473389, 11.7965975, 2.51279998, 1.0616473, 0.691560507, 0},
 {1.69736111, 43.9810371, 873.232361, 11541.8584, 95371.6563, 0.03313467652, 418.870239, 16851.0996, 86395.4922, 165305.313, 0}},
{10, 0.5205, {236.531204, 9.77416229, 2.08200002, 0.879636109, 0.572997808, 0},
 {0.04274908826, 2.6584549, 142.06488, 5268.31689, 129182.891, 0.0007210918702, 102.273438, 16152.1201, 180568.219, 515993.25, 0}},
{10, 0.5129, {233.07753, 9.63144684, 2.05159998, 0.866792262, 0.564631283, 0},
 {0.06792763621, 4.73335934, 265.0383, 10073.6582, 251756.672, 0.0009996999288, 200.471283, 32481.7402, 366493.25, 1054867.88, 0}},
{10, 0.0665, {30.2196445, 1.24876428, 0.266000003, 0.112383865, 0.07320721447, 0},
 {0.248241663, 6.20659971, 120.277809, 1608.43457, 15290.3027, 4.38909674, 9963.03711, 138626.078, 342480.813, 385242.594, 0}},
{10, 0.0378, {17.1774826, 0.709823906, 0.151199996, 0.06388135254, 0.04161252454, 0},
 {0.00589512568, 0.348192871, 17.4781113, 587.836182, 12128.6934, 1.71744359, 27477.3926, 344220.781, 266125, 1052709.13, 0}},
{10, 0.0378, {17.1774826, 0.709823906, 0.151199996, 0.06388135254, 0.04161252454, 0},
 {0.009413871914, 0.619588792, 32.5734215, 1124.89624, 23648.6406, 3.13177872, 53901.5117, 701182.125, 560652, 2076638.38, 0}},
{10, 0.0022, {0.999747634, 0.04131250456, 0.008799999952, 0.00371796242, 0.002421892947, 0},
 {7.826552064e-06, 0.00130264333, 0.193255246, 20.0691166, 1380.06238, 1381.32471, 5239243, 9182668, 10412750, 10710503, 0}},

// Z=24,  9 orbitals
{11, 5.9892, {127.674065, 25.9536381, 11.9784002, 7.78592062, 6.28398228, 5.99518919},
 {23.8584499, 659.70166, 14486.9111, 0, 0, 5.58467245, 716.948975, 6486.02197, 20613.4258, 35712.5781, 40916.9805}},
{10, 0.6946, {315.647583, 13.0434837, 2.77839994, 1.1738621, 0.764657617, 0},
 {2.0621798, 52.4386368, 1014.19061, 12981.0391, 102526.242, 0.03025845997, 375.973938, 15277.1689, 79048.1406, 151559.484, 0}},
{10, 0.5837, {265.251221, 10.9609585, 2.3348, 0.986443043, 0.642572224, 0},
 {0.05744704604, 3.50119233, 182.161697, 6545.91699, 154516.766, 0.0006451446097, 87.9408264, 14092.8447, 160078.109, 468845.625, 0}},
{10, 0.5745, {261.070465, 10.7881966, 2.2980001, 0.970895171, 0.632444322, 0},
 {0.09077656269, 6.2059207, 338.71524, 12490.7891, 301108, 0.0008811053121, 171.919571, 28343.875, 325225.781, 960945.375, 0}},
{10, 0.0741, {33.673317, 1.39148021, 0.296400011, 0.125227734, 0.08157375455, 0},
 {0.302706599, 7.43203974, 140.450073, 1828.28052, 17014.6992, 3.90112567, 9012.97559, 126689.531, 309776.031, 321821.813, 0}},
{10, 0.0425, {19.3133068, 0.798082411, 0.170000002, 0.07182427496, 0.04678656533, 0},
 {0.007973670959, 0.460968494, 22.4341736, 725.293579, 14298.7051, 1.479128, 24396.9609, 310075.875, 223923.734, 1302581.75, 0}},
{10, 0.0425, {19.3133068, 0.798082411, 0.170000002, 0.07182427496, 0.04678656533, 0},
 {0.01263785642, 0.815583825, 41.6369019, 1384.22351, 27850.6758, 2.66806769, 47729.0195, 632149.5, 472555.719, 2572403, 0}},
{10, 0.0023, {1.04519069, 0.04319034144, 0.009200000204, 0.00388696068, 0.002531978767, 0},
 {1.598890958e-05, 0.002490416402, 0.352081299, 35.3093948, 2332.49268, 1990.50281, 5731099.5, 11103850, 15900373, 17578990, 0}},
{10, 0.0023, {1.04519069, 0.04319034144, 0.009200000204, 0.00388696068, 0.002531978767, 0},
 {2.306908527e-06, 0.0004717691918, 0.0779812336, 8.36031055, 565.273315, 482.098938, 1422491.13, 2800122.25, 4079126.25, 4554578.5, 0}},

// Z=25,  9 orbitals
{11, 6.539, {139.394363, 28.3361435, 13.0780001, 8.50065708, 6.8608427, 6.5455389},
 {28.4208565, 774.182983, 16600.3594, 0, 0, 5.0947504, 650.499634, 5888.91309, 18730.2344, 32452.0781, 37298.293}},
{10, 0.769, {349.457245, 14.4405975, 3.07599998, 1.29959691, 0.84656167, 0},
 {2.47825813, 61.8827705, 1166.31567, 14459.9814, 109372.039, 0.02737544663, 333.641083, 13715.877, 71665.7734, 139514.672, 0}},
{10, 0.6514, {296.016174, 12.2322559, 2.60559988, 1.10085487, 0.717100441, 0},
 {0.0760230422, 4.54017782, 229.838562, 7990.76465, 180338.203, 0.0005802452797, 75.9000168, 12276.8643, 140810.891, 412271.844, 0}},
{10, 0.6403, {290.972015, 12.0238161, 2.5611999, 1.0820961, 0.704880893, 0},
 {0.119360931, 8.00909424, 425.921082, 15215.748, 351552.875, 0.000786130724, 148.008759, 24696.9512, 286419.781, 845586.25, 0}},
{10, 0.0839, {38.1267395, 1.57550859, 0.335599989, 0.14178957, 0.0923621878, 0},
 {0.369769335, 8.92678928, 164.650696, 2087.09033, 18974.6934, 3.25346279, 7935.61279, 114448.82, 292233.063, 339664.719, 0}},
{10, 0.0486, {22.0853348, 0.912630737, 0.194399998, 0.08213317394, 0.0535018146, 0},
 {0.01085056923, 0.614776731, 29.0170918, 900.749329, 16910.4805, 1.21343887, 20973.2129, 287629, 238536.297, 650341.813, 0}},
{10, 0.0486, {22.0853348, 0.912630737, 0.194399998, 0.08213317394, 0.0535018146, 0},
 {0.01704382896, 1.08166397, 53.681572, 1716.14941, 32937.5547, 2.16276741, 40929.7813, 586108.75, 510700.938, 1289947.25, 0}},
{10, 0.0072616, {3.29989052, 0.136361137, 0.02904636599, 0.01227196585, 0.007993998006, 0},
 {2.829528239e-05, 0.004537343513, 0.637378931, 61.7960281, 3924.17334, 42.0984802, 1352013.25, 6780917, 5107832, 3621647.25, 0}},
{10, 0.0071437, {3.24635458, 0.134148881, 0.02857512981, 0.01207287051, 0.007864307612, 0},
 {4.043392892e-06, 0.0008592494996, 0.141584083, 14.6815014, 954.535461, 10.6563883, 346167.406, 1705718.13, 1301684.5, 926458.625, 0}},

// Z=26,  9 orbitals
{11, 7.112, {151.609222, 30.8191853, 14.224, 9.24555302, 7.46204519, 7.11911201},
 {33.5967026, 901.826782, 18909.1035, 0, 0, 4.64799356, 593.258423, 5374.18408, 17106.834, 29643.8125, 34122.8516}},
{10, 0.8461, {384.493866, 15.8884125, 3.38439989, 1.42989457, 0.931437969, 0},
 {2.9529779, 72.4341125, 1331.50403, 16011.6406, 115667.063, 0.02514671162, 299.203613, 12430.0166, 65528.4492, 128504.328, 0}},
{10, 0.7211, {327.690002, 13.5411119, 2.88440013, 1.21864665, 0.793830454, 0},
 {0.09943902493, 5.81886387, 286.807678, 9667.25781, 209151.594, 0.0005338012706, 66.8126221, 10892.3428, 126080.281, 372604.25, 0}},
{10, 0.7081, {321.78241, 13.2969923, 2.83240008, 1.19667697, 0.77951926, 0},
 {0.155008718, 10.2135296, 529.719604, 18373.6797, 407788.563, 0.0007196301012, 129.786392, 21899.6582, 256620.547, 765136.438, 0}},
{10, 0.0929, {42.2166176, 1.74451435, 0.371600002, 0.156999409, 0.102269933, 0},
 {0.443230689, 10.5243616, 189.634857, 2344.99683, 20846.9414, 2.90282345, 7199.48584, 105066.07, 271252.344, 318037.219, 0}},
{10, 0.054, {24.5392609, 1.01403415, 0.216000006, 0.09125907719, 0.0594464615, 0},
 {0.01434014179, 0.79649514, 36.4943275, 1089.90161, 19528.9355, 1.07614696, 18903.8281, 265300.969, 225420.891, 544294.313, 0}},
{10, 0.054, {24.5392609, 1.01403415, 0.216000006, 0.09125907719, 0.0594464615, 0},
 {0.02241079509, 1.39496934, 67.266304, 2072.01074, 38024.9414, 1.8962189, 36795.9922, 540918.875, 485733.313, 1083631.13, 0}},
{10, 0.00356, {1.63595068, 0.0676022768, 0.01439999975, 0.006083938293, 0.003963097464, 0},
 {4.385668217e-05, 0.006915745325, 0.94388485, 88.1911469, 5344.8877, 904.588257, 4360846, 6479449.5, 6033488, 5995079.5, 0}},
{10, 0.00356, {1.63595068, 0.0676022768, 0.01439999975, 0.006083938293, 0.003963097464, 0},
 {1.248026183e-05, 0.002616845304, 0.418635994, 41.8461266, 2598.32959, 436.012756, 2171087.5, 3276084.5, 3056089.75, 3007959.75, 0}},

// Z=27,  9 orbitals
{11, 7.7089, {164.333557, 33.4057961, 15.4177999, 10.0215197, 8.08832359, 7.716609},
 {39.4313164, 1043.13147, 21369.3008, 0, 0, 4.28146601, 543.329163, 4923.98242, 15683.9902, 27184.9941, 31321.7324}},
{10, 0.9256, {420.621094, 17.3812962, 3.70239997, 1.5642482, 1.01895642, 0},
 {3.49209929, 84.1281891, 1508.71106, 17605.7676, 121267.266, 0.02347338758, 270.816925, 11345.8652, 60245.4531, 118855.93, 0}},
{10, 0.7936, {360.63623, 14.9025459, 3.17440009, 1.34117043, 0.873642802, 0},
 {0.128448009, 7.37002563, 353.904449, 11571.7295, 239973.906, 0.0004955939366, 59.5110245, 9754.95801, 113755.008, 338721.719, 0}},
{10, 0.7786, {353.819794, 14.6208706, 3.11439991, 1.31582069, 0.857129872, 0},
 {0.199023858, 12.8761759, 651.491699, 21956.6172, 468721.469, 0.0006732242764, 115.055496, 19594.3516, 231614.719, 696498.438, 0}},
{10, 0.1007, {45.7611771, 1.89098597, 0.402799994, 0.170181274, 0.110856637, 0},
 {0.527080059, 12.2973413, 216.458298, 2613.66479, 22692.3418, 2.7349987, 6767.11426, 98785.1719, 255470.359, 301436.031, 0}},
{10, 0.0595, {27.0386295, 1.11731541, 0.238000005, 0.100553982, 0.06550119072, 0},
 {0.0186880827, 1.01760614, 45.2823868, 1302.37842, 22269.2207, 0.97313112, 17233.8945, 245676.391, 212413.641, 467092.094, 0}},
{10, 0.0595, {27.0386295, 1.11731541, 0.238000005, 0.100553982, 0.06550119072, 0},
 {0.0289880801, 1.7726568, 83.1538925, 2470.61206, 43352.5391, 1.69353569, 33457.4688, 501269.844, 460572.344, 933397.188, 0}},
{10, 0.0029, {1.31784916, 0.05445738882, 0.01159999985, 0.004900950473, 0.003192495089, 0},
 {6.596191088e-05, 0.01024698, 1.36048698, 122.744514, 7140.04639, 2703.98633, 5272268, 5839494, 6107424.5, 6859508, 0}},
{10, 0.0029, {1.31784916, 0.05445738882, 0.01159999985, 0.004900950473, 0.003192495089, 0},
 {2.791640509e-05, 0.005796468351, 0.903280079, 87.2194901, 5200.96289, 1960.57092, 3948209.75, 4443491.5, 4623939.5, 5131735.5, 0}},

// Z=28,  9 orbitals
{11, 8.3328, {177.633469, 36.1094093, 16.6655998, 10.8325853, 8.74293137, 8.34113312},
 {45.9692841, 1198.94641, 23975.1094, 0, 0, 3.95673132, 499.031403, 4524.92383, 14424.6455, 25003.709, 28779.6367}},
{11, 1.0081, {21.4900513, 4.36850691, 2.01620007, 1.31052339, 1.05771756, 1.00910807},
 {4.09878159, 97.0018768, 1697.90784, 19234.1719, 0, 175.452911, 8759.09961, 41247.1641, 85691.0078, 115743.359, 125379.633}},
{10, 0.8719, {396.21817, 16.3728962, 3.48760009, 1.47349608, 0.959840178, 0},
 {0.164238393, 9.24035645, 432.607697, 13745.6699, 277654.313, 0.0004599943059, 52.9345589, 8730.52051, 102716.945, 308360.781, 0}},
{10, 0.8547, {388.401947, 16.0499077, 3.41880012, 1.44442844, 0.940905392, 0},
 {0.252628744, 16.0622597, 793.752747, 26041.0527, 541307, 0.0006338458625, 101.764656, 17515.0469, 209193.891, 634732.625, 0}},
{10, 0.1118, {50.8053589, 2.09942627, 0.4472, 0.188940093, 0.123076193, 0},
 {0.62044394, 14.2436867, 245.278503, 2896.93066, 24583.4199, 2.32250309, 6039.97852, 89677.125, 235188.766, 278220.125, 0}},
{10, 0.0681, {30.9467335, 1.27880979, 0.272399992, 0.115087837, 0.07496859133, 0},
 {0.0240989048, 1.28517413, 55.5597305, 1541.9696, 25191.3965, 0.763312697, 14306.6006, 218936.094, 199737.891, 384962.188, 0}},
{10, 0.0681, {30.9467335, 1.27880979, 0.272399992, 0.115087837, 0.07496859133, 0},
 {0.03709980473, 2.22665858, 101.645332, 2918.94946, 49043.4727, 1.30890584, 27669.1055, 446311.094, 435478.656, 771988, 0}},
{10, 0.0036, {1.63595068, 0.0676022768, 0.01439999975, 0.006083938293, 0.003963097464, 0},
 {9.720342496e-05, 0.01483889204, 1.91603458, 167.094742, 9355.44727, 1645.66589, 4322541, 4872569, 4804619.5, 5332216, 0}},
{10, 0.0036, {1.63595068, 0.0676022768, 0.01439999975, 0.006083938293, 0.003963097464, 0},
 {5.445885472e-05, 0.01115813758, 1.69257736, 158.038025, 9076.38281, 1582.09875, 4310802.5, 4948048.5, 4847296.5, 5294123, 0}},

// Z=29,  9 orbitals
{11, 8.9789, {191.406631, 38.9092216, 17.9577999, 11.6725111, 9.42083168, 8.9878788},
 {53.2751999, 1370.01917, 0, 0, 0, 3.67557979, 460.38797, 4176.02441, 13321.418, 23094.584, 26531.3652}},
{11, 1.0961, {23.3659821, 4.74984646, 2.19219995, 1.42492282, 1.15004885, 1.0971961},
 {4.77699661, 111.155106, 1901.578, 20946.1465, 0, 159.816452, 8044.51953, 38107.1836, 79578.3594, 107951.578, 98264.7188}},
{10, 0.951, {432.163635, 17.8582687, 3.8039999, 1.6071738, 1.04691827, 0},
 {0.207876742, 11.4773922, 524.464844, 16219.5195, 335328.406, 0.0004353651893, 47.9885559, 7946.92432, 94237.9219, 285755.063, 0}},
{10, 0.9311, {423.120483, 17.4845772, 3.72440004, 1.57354307, 1.02501106, 0},
 {0.317746878, 19.8555527, 959.083496, 30679.8008, 642349.813, 0.0006144578801, 91.8376389, 15937.9561, 192159.094, 590128.125, 0}},
{10, 0.1198, {54.4408035, 2.24965358, 0.479200006, 0.202459961, 0.13188307, 0},
 {0.722816169, 16.2981033, 274.493652, 3175.11328, 26342.4785, 2.21426272, 5730.03467, 84846.7969, 216483.391, 234566.063, 0}},
{10, 0.0736, {33.4461021, 1.38209105, 0.294400007, 0.124382742, 0.08102332801, 0},
 {0.03050390072, 1.59164131, 66.8567581, 1792.96069, 27991.2402, 0.722355902, 13405.5195, 202315.969, 164977.609, 484288.344, 0}},
{10, 0.0736, {33.4461021, 1.38209105, 0.294400007, 0.124382742, 0.08102332801, 0},
 {0.04648753628, 2.73926282, 121.781357, 3385.16455, 54478.8164, 1.22389197, 25846.0547, 413024.906, 362458.844, 968524.375, 0}},
{10, 0.0016, {0.727089226, 0.03004545718, 0.006400000304, 0.002703972859, 0.001761376741, 0},
 {0.0001341273601, 0.01955577545, 2.42353272, 204.523788, 11062.6572, 31380.8027, 4706225, 6409198, 8098560, 9343090, 0}},
{10, 0.0016, {0.727089226, 0.03004545718, 0.006400000304, 0.002703972859, 0.001761376741, 0},
 {0.0001116064741, 0.02188922837, 3.18954372, 288.272064, 16006.7266, 45622.2109, 7098642, 9932945, 12575751, 14364417, 0}},

// Z=30,  9 orbitals
{11, 9.6589, {205.896057, 41.8546371, 19.3171997, 12.5561171, 10.1339865, 9.66825867},
 {61.385128, 1556.50842, 0, 0, 0, 3.41738033, 424.861633, 3855.01636, 12305.6807, 21342.2305, 24597.9238}},
{11, 1.1936, {25.4444256, 5.17235374, 2.38720012, 1.55167222, 1.25234771, 1.19479358},
 {5.53230429, 126.530487, 2115.26636, 22648.3418, 0, 144.20369, 7324.24951, 34937.3711, 73252.3047, 99417.7656, 107036.82}},
{11, 1.0428, {22.2297649, 4.51887608, 2.0855999, 1.35563314, 1.09412551, 1.04384279},
 {0.260641187, 14.1196003, 629.8302, 18958.1777, 0, 27.0905952, 5521.70898, 53473.457, 167394.891, 300247.656, 1333241}},
{11, 1.0197, {21.7373333, 4.4187746, 2.0394001, 1.32560337, 1.06988859, 1.02071965},
 {0.395631582, 24.3039589, 1147.85352, 35802.5313, 0, 51.2061539, 11047.5537, 108699.633, 343778.188, 622572.875, 2775663.5}},
{10, 0.1359, {61.7571373, 2.55198598, 0.543600023, 0.229668677, 0.149606928, 0},
 {0.843118131, 18.7137108, 308.69693, 3499.22192, 28351.6152, 1.81062496, 4876.25684, 74630.8828, 200414.328, 237135.188, 0}},
{10, 0.0866, {39.3537025, 1.62621033, 0.346399993, 0.146352515, 0.09533450752, 0},
 {0.03875831515, 1.98363042, 81.0585175, 2100.58325, 31268.1074, 0.500795305, 10337.084, 176975.906, 176503.344, 277110.688, 0}},
{10, 0.0866, {39.3537025, 1.62621033, 0.346399993, 0.146352515, 0.09533450752, 0},
 {0.05864527449, 3.39699197, 147.175522, 3960.25513, 60918.1406, 0.83180958, 19836.1797, 360242.563, 388554.844, 559752.813, 0}},
{10, 0.0081, {3.68088913, 0.152105123, 0.03240000084, 0.01368886139, 0.008916969411, 0},
 {0.000197031346, 0.02921710163, 3.58537722, 293.221649, 15270.9492, 132.655106, 1802453.38, 3597675, 2548656.25, 2326434.25, 0}},
{10, 0.0081, {3.68088913, 0.152105123, 0.03240000084, 0.01368886139, 0.008916969411, 0},
 {0.0001636254747, 0.03274735808, 4.72995329, 414.446899, 22173.4941, 185.929138, 2675326.5, 5466531.5, 3880210.5, 3443699.5, 0}},

// Z=31,  9 orbitals
{11, 10.367, {220.99942, 44.9248581, 20.7341995, 13.4771624, 10.8773575, 10.3774672},
 {70.3515015, 1759.00732, 0, 0, 0, 3.18604612, 392.892303, 3565.45703, 11387.8984, 19762.4121, 22863.1387}},
{11, 1.2977, {27.6635666, 5.62346125, 2.59540009, 1.68700147, 1.36157143, 1.29899764},
 {6.36879921, 143.186615, 2339.80249, 24352.5195, 0, 130.087662, 6664.97998, 32003.0059, 67459.7031, 93002.2422, 97879.8594}},
{11, 1.1423, {24.3508453, 4.95004988, 2.28460002, 1.48498249, 1.19852281, 1.14344227},
 {0.323959708, 17.2229671, 750.230164, 21983.3828, 0, 23.8705845, 4887.52295, 47695.3945, 150168.469, 264292.719, 1161609.88}},
{11, 1.1154, {23.7774067, 4.83348131, 2.23079991, 1.45001268, 1.17029881, 1.1165154},
 {0.488045543, 29.4898682, 1362.54773, 41448.9023, 0, 44.8492928, 9771.77344, 97009.0781, 308733.25, 547964.938, 2423530.5}},
{10, 0.1581, {71.8455048, 2.96886683, 0.632400036, 0.267186314, 0.174046025, 0},
 {0.980047762, 21.4168129, 346.233002, 3849.9126, 30489.5234, 1.32787907, 3933.51074, 63336.6367, 180888.016, 223851.828, 0}},
{10, 0.1068, {48.5332031, 2.00553417, 0.42719999, 0.180490181, 0.11757189, 0},
 {0.04889723286, 2.45786095, 97.8323288, 2453.60913, 34869.2188, 0.302032202, 7056.104, 145255.656, 194211.641, 153909.406, 0}},
{10, 0.1029, {46.7609253, 1.93229842, 0.411599994, 0.173899248, 0.113278531, 0},
 {0.07358220965, 4.1898675, 176.9366, 4610.36816, 67755.8047, 0.556340277, 14740.4688, 307470.469, 427068.25, 334535.438, 0}},
{10, 0.0174, {7.90709448, 0.326744318, 0.06960000098, 0.02940570191, 0.0191549696, 0},
 {0.0002864034032, 0.04266745225, 5.1559391, 408.520752, 20417.7539, 8.49055195, 496642.375, 3697577, 1791704.5, 1466864.75, 0}},
{10, 0.0174, {7.90709448, 0.326744318, 0.06960000098, 0.02940570191, 0.0191549696, 0},
 {0.0002360964572, 0.04774709046, 6.80612516, 578.010925, 29699.7559, 11.3159914, 733049.25, 5574049, 2732383, 2190759.75, 0}},

// Z=32,  9 orbitals
{11, 11.1031, {236.689011, 48.1142426, 22.2061996, 14.4339571, 11.6495829, 11.1142035},
 {80.224617, 1978.03235, 0, 0, 0, 2.97958446, 364.208313, 3305.44434, 10563.2871, 18339.8008, 21252.4922}},
{11, 1.4143, {30.1491718, 6.1287365, 2.82859993, 1.83858073, 1.48391032, 1.41571426},
 {7.28964853, 161.231857, 2578.17993, 26116.5684, 0, 116.147964, 6021.05811, 29166.0547, 61864.875, 85900.4766, 91946.7969}},
{11, 1.2478, {26.5998287, 5.40722418, 2.49559999, 1.62213182, 1.30921543, 1.24904776},
 {0.399251997, 20.8421402, 887.159058, 25318.959, 0, 21.1208858, 4340.14258, 42661.2109, 135139.531, 236204.406, 1025463.19}},
{11, 1.2167, {25.9368572, 5.27245522, 2.43339992, 1.58170199, 1.27658463, 1.21791673},
 {0.597486913, 35.5078125, 1605.66748, 47667.3398, 0, 39.4249001, 8668.75293, 86800.4922, 278110.656, 490046.188, 2151649.5}},
{10, 0.18, {81.7975388, 3.38011384, 0.720000029, 0.304196924, 0.198154882, 0},
 {1.13551497, 24.4313889, 387.218262, 4224.90674, 32690.6797, 1.06691349, 3305.13623, 55324.1563, 165297.047, 210959.625, 0}},
{10, 0.1279, {58.1216927, 2.40175867, 0.511600018, 0.216148809, 0.140800044, 0},
 {0.06146106869, 3.03055429, 117.449364, 2851.64014, 38660.9688, 0.194859102, 5125.17773, 121457.352, 203937.344, 119521.008, 0}},
{10, 0.1208, {54.8952332, 2.2684319, 0.483200014, 0.204149932, 0.132983938, 0},
 {0.09169522673, 5.1381321, 211.667389, 5344.65771, 75049.6797, 0.374449104, 11211.8652, 263858.656, 451811.719, 274452.563, 0}},
{10, 0.0287, {13.0421619, 0.53894037, 0.114799999, 0.04850250855, 0.03159469366, 0},
 {0.0004109158472, 0.0609209016, 7.22076845, 554.85022, 26606.1992, 1.40949738, 170372.672, 2955295.25, 1788004.63, 925162.063, 0}},
{10, 0.0287, {13.0421619, 0.53894037, 0.114799999, 0.04850250855, 0.03159469366, 0},
 {0.0003384672746, 0.06817138195, 9.52997208, 785.18158, 38735.9609, 1.7875061, 250344.313, 4433116, 2723435, 1394704.25, 0}},

// Z=33,  9 orbitals
{11, 11.8667, {252.966965, 51.423233, 23.7334003, 15.4266319, 12.4507666, 11.8785667},
 {91.0590286, 2214.20728, 0, 0, 0, 2.77999401, 338.074158, 3071.3042, 9819.9209, 17059.8027, 19765.7715}},
{11, 1.5265, {32.5409851, 6.61494493, 3.05300021, 1.98444009, 1.60163271, 1.52802658},
 {8.30234146, 180.560242, 2823.69092, 27807.3613, 0, 106.028755, 5528.84961, 26914.7109, 57309.3477, 79884.0313, 86638.7656}},
{11, 1.3586, {28.9617939, 5.88736582, 2.71720004, 1.7661711, 1.4254688, 1.35995865},
 {0.488622487, 25.0425739, 1041.92041, 28968.4316, 0, 18.8089008, 3873.24927, 38317.7344, 122051.578, 212553.844, 910310.875}},
{11, 1.3231, {28.2050266, 5.73353004, 2.64619994, 1.72002137, 1.38822162, 1.32442307},
 {0.725964069, 42.4437675, 1879.37708, 54474.1758, 0, 34.8414116, 7723.30566, 77956.7031, 251346.531, 441191.563, 1911485.38}},
{10, 0.2035, {92.4766541, 3.8214066, 0.81400001, 0.343911529, 0.224025086, 0},
 {1.31026614, 27.7612228, 431.583313, 4623.30566, 34990.5742, 0.87553215, 2808.80566, 48673.4727, 152043.938, 200912.766, 0}},
{10, 0.1464, {66.5286636, 2.74915934, 0.585600019, 0.247413501, 0.161165968, 0},
 {0.07672012597, 3.71009231, 139.956955, 3287.16162, 42463.3164, 0.146437481, 4137.73193, 106420.07, 207359.031, 121932.625, 0}},
{10, 0.1405, {63.8475227, 2.6383667, 0.562000036, 0.237442613, 0.154670894, 0},
 {0.113692299, 6.26289845, 251.658844, 6162.2124, 82752.8125, 0.268081605, 8669.26953, 226513.859, 457978.469, 282961.406, 0}},
{10, 0.0412, {18.7225475, 0.773670495, 0.164800003, 0.0696272999, 0.04535545036, 0},
 {0.0005811402225, 0.08526003361, 9.8884201, 737.48053, 33899.0352, 0.408280015, 74780.4766, 2276189.5, 1982255.38, 636077.063, 0}},
{10, 0.0412, {18.7225475, 0.773670495, 0.164800003, 0.0696272999, 0.04535545036, 0},
 {0.0004769114603, 0.09527547657, 13.0436487, 1043.49854, 49382.5664, 0.488065064, 109408.219, 3403490.75, 3013125.75, 961699.813, 0}},

// Z=34,  9 orbitals
{11, 12.6578, {269.831146, 54.8513908, 25.3155994, 16.4550571, 13.2808037, 12.6704578},
 {102.904449, 2467.74854, 0, 0, 0, 2.61308599, 314.814301, 2859.95776, 9148.33203, 15902.4307, 18405.3477}},
{11, 1.6539, {35.2568169, 7.1670208, 3.30780005, 2.15005922, 1.73530328, 1.65555394},
 {9.41354561, 201.409302, 3084.02319, 29570.8711, 0, 95.526535, 5028.90283, 24670.2402, 52813.6211, 73890.2188, 80790.6172}},
{11, 1.4762, {31.4687195, 6.39697456, 2.95239997, 1.91905034, 1.54885697, 1.47767615},
 {0.593881845, 29.8914223, 1216.38818, 32970.6836, 0, 16.7941189, 3464.03857, 34490.4727, 110469.086, 192158.172, 806454.875}},
{11, 1.4358, {30.6074982, 6.22190523, 2.87160015, 1.86653066, 1.50646865, 1.43723583},
 {0.875946045, 50.3962593, 2186.65649, 61936.3945, 0, 30.8629723, 6894.74365, 70160.8828, 227644.938, 399123.906, 1695917.13}},
{10, 0.2315, {105.200722, 4.34720182, 0.925999999, 0.39123106, 0.254849195, 0},
 {1.50731444, 31.4317169, 479.496704, 5048.24072, 37444.2422, 0.671760857, 2349.33813, 42372.9219, 138259.844, 189295.625, 0}},
{10, 0.1682, {76.4352493, 3.15852857, 0.672800004, 0.284255117, 0.18516472, 0},
 {0.09510219097, 4.51131201, 165.746719, 3769.60205, 46480.418, 0.111424357, 3291.01318, 92255.5781, 204266.703, 134721.031, 0}},
{10, 0.1619, {73.5723343, 3.04022455, 0.647599995, 0.273608238, 0.178229287, 0},
 {0.139976755, 7.58119965, 297.261536, 7063.01318, 90829.4453, 0.190035895, 6817.16602, 195496.141, 450909.063, 315883.531, 0}},
{10, 0.0567, {25.7662239, 1.06473589, 0.226799995, 0.09582202882, 0.06241878495, 0},
 {0.0008087955648, 0.11716368, 13.2909241, 963.809387, 42543.8438, 0.137163624, 34927.3047, 1639086.38, 2139142.5, 459565.406, 0}},
{10, 0.0567, {25.7662239, 1.06473589, 0.226799995, 0.09582202882, 0.06241878495, 0},
 {0.0006616109749, 0.130725905, 17.5188789, 1363.39148, 62003.918, 0.154018119, 50843.1367, 2443793, 3244723.25, 694652.125, 0}},

// Z=35,  9 orbitals
{11, 13.4737, {287.223999, 58.3870163, 26.9473991, 17.5157223, 14.1368608, 13.487174},
 {115.801857, 2738.54492, 0, 0, 0, 2.46435618, 293.933472, 2669.65356, 8542.25879, 14855.8311, 17168.5137}},
{11, 1.782, {37.9875755, 7.7221303, 3.56400013, 2.3165884, 1.86970818, 1.78378201},
 {10.6241693, 223.629333, 3352.69434, 31288.7969, 0, 87.0761642, 4614.44434, 22768.4023, 48947.3555, 68661.2578, 75400.8125}},
{11, 1.596, {34.022541, 6.91611624, 3.19199991, 2.07478952, 1.67455339, 1.59759593},
 {0.717057228, 35.4425697, 1410.52295, 37236.1719, 0, 15.1906128, 3129.47437, 31291.8672, 100618.641, 174956.828, 717155.438}},
{11, 1.5499, {33.0398102, 6.71634674, 3.09980011, 2.01485991, 1.62618446, 1.55144989},
 {1.04939413, 59.4305878, 2526.896, 69876.7578, 0, 27.7121754, 6221.36328, 63677, 207570.5, 364118.344, 1505880.75}},
{10, 0.2565, {116.561485, 4.81666231, 1.02600002, 0.43348062, 0.282370687, 0},
 {1.72405887, 35.4125748, 530.380676, 5485.78857, 39851.4297, 0.576584041, 2067.87671, 38239.3828, 128469.82, 181360.719, 0}},
{10, 0.1893, {86.023735, 3.55475307, 0.757200003, 0.319913775, 0.208392873, 0},
 {0.117057204, 5.44646025, 194.888123, 4290.66504, 50516.0938, 0.09125851095, 2744.77466, 81830.6953, 199421.891, 151311.875, 0}},
{10, 0.1815, {82.4791794, 3.40828133, 0.726000011, 0.30673191, 0.199806154, 0},
 {0.171111077, 9.11055565, 348.548157, 8031.59326, 98924.9609, 0.153152615, 5720.55713, 174393.688, 441951.531, 357866.969, 0}},
{10, 0.0701, {31.8555946, 1.31636655, 0.280400008, 0.1184678, 0.0771703124, 0},
 {0.001109156408, 0.158323511, 17.5431156, 1235.59436, 52167.1211, 0.07276616246, 21845.5039, 1291753.5, 2164522, 369368.375, 0}},
{10, 0.069, {31.3557224, 1.29571033, 0.275999993, 0.116608821, 0.07595936954, 0},
 {0.0009028668865, 0.176223963, 23.0945625, 1745.56506, 75881.2969, 0.08406757563, 33285.5859, 1971108.13, 3277090.25, 564616.125, 0}},

// Z=36,  9 orbitals
{11, 14.3256, {305.384277, 62.0786438, 28.6511993, 18.6231861, 15.0306911, 14.3399258},
 {129.995651, 3029.052, 0, 0, 0, 2.32620788, 274.610352, 2494.08398, 7984.88086, 13894.3145, 16032.0732}},
{11, 1.921, {40.9506912, 8.32447338, 3.84200001, 2.49728751, 2.01554966, 1.92292106},
 {11.9420261, 247.39299, 3634.12915, 33032.5078, 0, 78.6450119, 4222.15674, 20975.0977, 45307.5313, 63711.9375, 70149.6953}},
{11, 1.7272, {36.8193817, 7.48465967, 3.45440006, 2.24534869, 1.81221104, 1.72892725},
 {0.860745907, 41.8030548, 1629.10864, 41976.5313, 0, 13.6485691, 2812.35938, 28296.9355, 91485.9141, 159344.422, 637128.125}},
{11, 1.6749, {35.704483, 7.25802231, 3.34979987, 2.17735887, 1.75733674, 1.67657483},
 {1.25032532, 69.7208557, 2908.56104, 78711.8125, 0, 24.6885414, 5580.72559, 57580.0781, 188882.234, 332067.063, 1340001.13}},
{10, 0.28833, {131.026016, 5.41437912, 1.15331995, 0.487272769, 0.317411065, 0},
 {1.9636997, 39.7445793, 584.965515, 5951.28271, 42458.3047, 0.475202024, 1761.07068, 33711.1406, 117089.836, 170235.375, 0}},
{10, 0.2227, {101.201729, 4.181952, 0.890799999, 0.376359195, 0.245161608, 0},
 {0.143166572, 6.53821802, 228.273178, 4881.62988, 55085.2852, 0.06140151992, 2028.23376, 67512.3281, 185534.875, 159878.906, 0}},
{10, 0.2138, {97.1572952, 4.01482391, 0.855199993, 0.36131835, 0.235363945, 0},
 {0.208088011, 10.8912735, 407.198456, 9133.87207, 108257.883, 0.09887553006, 4190.48828, 143476.719, 410401.406, 378941.063, 0}},
{10, 0.0889, {40.3988914, 1.66940069, 0.355599999, 0.150239483, 0.09786649048, 0},
 {0.001498913509, 0.21085985, 22.847374, 1567.29126, 63664.1602, 0.03360760957, 12333.9443, 935434.938, 2128649.75, 295130.188, 0}},
{10, 0.0889, {40.3988914, 1.66940069, 0.355599999, 0.150239483, 0.09786649048, 0},
 {0.001216916484, 0.234330356, 30.0603981, 2215.63501, 92852.1797, 0.03355485573, 17778.7051, 1389055.5, 3219008.75, 444752.625, 0}},

// Z=37,  9 orbitals
{11, 15.2, {324.017792, 65.866478, 30.3993988, 19.75951, 15.947813, 15.2148991},
 {145.150543, 3335.24902, 0, 0, 0, 2.20369673, 257.035278, 2335.36548, 7478.15186, 13017.0439, 15004.7373}},
{11, 2.0651, {44.0225258, 8.94891739, 4.13019991, 2.68461657, 2.16674209, 2.06716514},
 {13.3684654, 272.545593, 3922.27734, 34714.9336, 0, 71.6623535, 3874.4502, 19358.709, 41990.6445, 59185.3477, 64918.2656}},
{11, 1.8639, {39.7334671, 8.0770359, 3.72779989, 2.42305779, 1.95563912, 1.8657639},
 {1.02672589, 49.0009155, 1869.27454, 46924.0781, 0, 12.3343239, 2535.96387, 25624.252, 83175.4922, 145147.594, 169550.125}},
{11, 1.8044, {38.4650841, 7.81919861, 3.60879993, 2.34570813, 1.89321065, 1.80620444},
 {1.48078465, 81.2901382, 3325.66699, 87919.4688, 0, 22.133955, 5025.31738, 52153.3359, 171869.922, 302482.156, 355350.25}},
{10, 0.3221, {146.372147, 6.04852581, 1.28840005, 0.544343531, 0.354587138, 0},
 {2.22920394, 44.4258423, 642.403442, 6425.02148, 45004.0195, 0.379439354, 1512.92029, 29823.3379, 106568.219, 166562.719, 0}},
{10, 0.2474, {112.42617, 4.64577866, 0.989600003, 0.418101788, 0.272352874, 0},
 {0.173848987, 7.78626299, 264.755524, 5475.88672, 59050.4336, 0.05199510977, 1728.84705, 60143.6719, 179115.75, 170479.313, 0}},
{10, 0.2385, {108.381729, 4.47865057, 0.953999996, 0.403060913, 0.262555212, 0},
 {0.250936806, 12.9117928, 471.177307, 10247.3936, 116533.945, 0.08001378179, 3515.19214, 126855, 395169.25, 401144.719, 0}},
{10, 0.1118, {50.8053589, 2.09942627, 0.4472, 0.188940093, 0.123076193, 0},
 {0.001996008214, 0.27689755, 29.3404846, 1958.40479, 76366.8516, 0.01569856517, 6946.48096, 651206.75, 2144910.75, 288034.063, 0}},
{10, 0.1103, {50.1237106, 2.07125854, 0.441199988, 0.186405122, 0.121424899, 0},
 {0.001618464827, 0.307255805, 38.5445099, 2763.93921, 111082.711, 0.01564608514, 10406.0605, 990321.813, 3254537.25, 431920.219, 0}},

// Z=38,  12 orbitals
{11, 16.1046, {343.307892, 69.7877731, 32.2092018, 20.9358749, 16.8972511, 16.1207047},
 {161.513519, 3660.20239, 0, 0, 0, 2.0915525, 241.108521, 2189.63208, 7012.84424, 12210.6865, 14059.6348}},
{11, 2.2163, {47.2457123, 9.60412788, 4.43260002, 2.88117552, 2.3253839, 2.21851635},
 {14.91012, 299.185394, 4219.23926, 36356.2305, 0, 65.429245, 3560.5376, 17888.4688, 38957.1836, 55047.1719, 60405.582}},
{11, 2.0068, {42.7797203, 8.69628048, 4.01360035, 2.60882688, 2.1055727, 2.00880694},
 {1.2186538, 57.1402168, 2133.73535, 52165.9063, 0, 11.1423483, 2294.35181, 23263.1074, 75764.1719, 132247.688, 150195.922}},
{11, 1.9396, {41.3471947, 8.40507507, 3.87920022, 2.52146745, 2.03506517, 1.94153965},
 {1.74423289, 94.2742844, 3783.10938, 97674.4453, 0, 19.8181324, 4538.41113, 47343.8086, 156657.875, 275882.625, 315142.156}},
{10, 0.3575, {162.458984, 6.71328163, 1.42999995, 0.604168892, 0.393557578, 0},
 {2.51823068, 49.4545937, 703.053589, 6912.58838, 47542.7813, 0.320201725, 1313.70667, 26578.0039, 97467.1953, 153493.984, 0}},
{10, 0.2798, {127.149719, 5.25419903, 1.11919999, 0.472857207, 0.308020741, 0},
 {0.209908247, 9.22296047, 305.693665, 6123.24658, 63259.9023, 0.04140869156, 1399.07043, 51847.1016, 167731.375, 180220.813, 0}},
{10, 0.2691, {122.287315, 5.05327034, 1.07640004, 0.45477441, 0.296241522, 0},
 {0.30086419, 15.2228489, 542.591614, 11452.9717, 125273.719, 0.06202661991, 2845.76807, 109597.852, 370477.844, 422457.781, 0}},
{10, 0.135, {61.3481522, 2.53508544, 0.540000021, 0.2281477, 0.14861615, 0},
 {0.002632827265, 0.359631956, 37.2268562, 2414.1416, 90173.9453, 0.008880086243, 4397.25391, 476697.594, 2032991, 424226.844, 0}},
{10, 0.1331, {60.4847336, 2.49940634, 0.532400012, 0.224936724, 0.146524519, 0},
 {0.002128628781, 0.398270935, 48.8483047, 3404.98462, 131123.516, 0.008409592323, 6564.3418, 725246.188, 3091865, 635316.875, 0}},
{10, 0.0377, {17.1320381, 0.707946062, 0.150800005, 0.06371235847, 0.04150243476, 0},
 {0.378550559, 7.14977789, 99.8566971, 995.345764, 7467.83545, 21.4249725, 13344.4141, 124349.555, 205973.406, 104676.906, 0}},
{10, 0.0199, {9.04317188, 0.373690367, 0.07959999889, 0.03363065794, 0.02190712094, 0},
 {0.02590393834, 1.07259893, 34.1239853, 668.743408, 7179.04785, 32.5392952, 33407.7227, 113940.352, 2081546.5, 15789259, 0}},
{10, 0.0199, {9.04317188, 0.373690367, 0.07959999889, 0.03363065794, 0.02190712094, 0},
 {0.0369482711, 1.76024485, 60.2213287, 1243.12451, 14006.2578, 57.37817, 68820.7734, 243546.422, 3774906.5, 30677134, 0}},

// Z=39,  12 orbitals
{11, 17.0384, {363.21405, 73.8343048, 34.0767975, 22.149807, 17.8770123, 17.0554371},
 {179.14679, 4003.6084, 0, 0, 0, 1.98952651, 226.568222, 2056.57666, 6588.13281, 11473.4834, 13194.2031}},
{11, 2.3725, {50.5754852, 10.2810059, 4.74499989, 3.08423448, 2.48927188, 2.37487245},
 {16.5722656, 327.370148, 4525.74902, 38015.5039, 0, 60.0163879, 3284.59888, 16586.6406, 36256.3047, 51340.8867, 56210.4727}},
{11, 2.1555, {45.9496155, 9.34065723, 4.31099987, 2.80213594, 2.26159143, 2.15765548},
 {1.439399, 66.326355, 2426.01465, 57802.668, 0, 10.1674337, 2085.79395, 21220.9609, 69344.5781, 121139.367, 138701.953}},
{11, 2.08, {44.3401527, 9.01348495, 4.15999985, 2.70398617, 2.18237543, 2.08207989},
 {2.04443002, 108.828255, 4286.77734, 108218.703, 0, 17.909481, 4117.66357, 43179.9531, 143478.172, 252981.453, 291438.125}},
{10, 0.3936, {178.863937, 7.39118242, 1.57440007, 0.665177286, 0.433298647, 0},
 {2.83639383, 54.8693581, 767.06604, 7417.85059, 50142.582, 0.276592433, 1157.77515, 23958.1934, 89794.2656, 144308.313, 0}},
{10, 0.3124, {141.964172, 5.86637545, 1.24960005, 0.527950704, 0.343908787, 0},
 {0.251975983, 10.8622761, 351.090485, 6813.45313, 67504.2188, 0.0329615213, 1167.85828, 45512.6055, 156651.328, 183206.031, 0}},
{10, 0.3003, {136.465546, 5.63915634, 1.20120001, 0.507501841, 0.330588371, 0},
 {0.358618289, 17.8460426, 621.632385, 12742.8994, 134270.031, 0.05000425503, 2364.53101, 96150.6016, 346162.188, 429718.688, 0}},
{10, 0.1596, {72.5271454, 2.99703431, 0.638400018, 0.26972127, 0.175697312, 0},
 {0.003440195462, 0.462344706, 46.7640839, 2949.67236, 105896.805, 0.00532815326, 2960.22778, 360749.375, 1858937.13, 552808.563, 0}},
{10, 0.1574, {71.5273972, 2.95572186, 0.629599988, 0.266003311, 0.173275426, 0},
 {0.002773017157, 0.510965705, 61.2910271, 4158.1084, 153977.609, 0.004748226609, 4387.53125, 547963.625, 2829978.5, 828897.438, 0}},
{10, 0.0454, {20.631155, 0.852539837, 0.181600004, 0.07672522217, 0.04997906089, 0},
 {0.451179683, 8.43212128, 116.126633, 1141.38098, 8498.56152, 15.9779139, 11140.957, 112213.25, 209913.75, 150483, 0}},
{10, 0.0256, {11.6334267, 0.480727285, 0.102399997, 0.04326356202, 0.028182026, 0},
 {0.03378038853, 1.38149095, 42.9767265, 816.078613, 8485.08398, 19.2197819, 27772.6797, 103374.656, 1032195, 6357127.5, 0}},
{10, 0.0256, {11.6334267, 0.480727285, 0.102399997, 0.04326356202, 0.028182026, 0},
 {0.04790235683, 2.26003051, 75.7655411, 1518.69629, 16618.9922, 33.3941879, 56679.707, 234334.891, 1845420.75, 12221569, 0}},

// Z=40,  13 orbitals
{11, 17.9976, {383.661713, 77.9909134, 35.9952011, 23.3967628, 18.8834229, 18.0155983},
 {198.084564, 4365.3501, 0, 0, 0, 1.89752984, 213.369598, 1935.44495, 6200.74756, 10803.498, 12404.0029}},
{11, 2.5316, {53.967083, 10.9704514, 5.0632, 3.29106331, 2.65620255, 2.53413153},
 {18.3579235, 357.05658, 4839.54834, 39608.8945, 0, 55.0786514, 3045.3269, 15444.5264, 33866.7383, 48042.4766, 52498.1016}},
{11, 2.30671, {49.172802, 9.99586868, 4.61339998, 2.9986949, 2.42023325, 2.30900669},
 {1.69195735, 76.6252518, 2745.34692, 63772.3477, 0, 9.37018871, 1911.56104, 19491.3379, 63848.3672, 111592.367, 128467}},
{11, 2.2223, {47.3736153, 9.63012886, 4.44460011, 2.88897538, 2.33167934, 2.22452235},
 {2.38461804, 125.030273, 4834.98193, 119323.406, 0, 16.3419838, 3766.07813, 39653.8359, 132203.844, 233354.406, 270296.406}},
{10, 0.4303, {195.54155, 8.08034992, 1.72119999, 0.727199614, 0.473700225, 0},
 {3.17947149, 60.6363792, 834.236755, 7938.17578, 52774.6797, 0.243396163, 1033.27185, 21803.209, 83217.4375, 135922.281, 0}},
{10, 0.3442, {156.41507, 6.46352863, 1.37679994, 0.5816921, 0.378916144, 0},
 {0.301004767, 12.7231665, 400.982788, 7538.53076, 71666.0938, 0.0284127444, 1006.94525, 40732.1367, 147045.813, 182756.844, 0}},
{10, 0.3305, {150.189362, 6.2062645, 1.32200003, 0.558539391, 0.363834351, 0},
 {0.425387323, 20.8067722, 708.235107, 14099.4453, 143236.375, 0.04015129432, 2032.82349, 86105.8047, 325500.844, 429292.906, 0}},
{10, 0.1824, {82.8881683, 3.4251821, 0.729600012, 0.308252901, 0.200796947, 0},
 {0.004447322339, 0.588055849, 58.1128082, 3562.12866, 122820.5, 0.003797446843, 2218.65479, 291803.063, 1696030.13, 631846.563, 0}},
{10, 0.18, {81.7975388, 3.38011384, 0.720000029, 0.304196924, 0.198154882, 0},
 {0.003569497028, 0.648275793, 76.076088, 5019.27295, 178619.703, 0.003210201627, 3261.68115, 442121.875, 2581909.75, 947468.688, 0}},
{10, 0.0513, {23.3122978, 0.963332415, 0.205200002, 0.08669612557, 0.05647413805, 0},
 {0.529819191, 9.78068161, 132.75412, 1287.17285, 9516.60938, 13.6866503, 10142.7656, 105999.258, 209340.328, 171248.594, 0}},
{10, 0.0287, {13.0421619, 0.53894037, 0.114799999, 0.04850250855, 0.03159469366, 0},
 {0.04301036894, 1.72926521, 52.4891701, 965.794861, 9737.61719, 16.6749153, 26720.4961, 106705.492, 768987.25, 4248017, 0}},
{10, 0.0287, {13.0421619, 0.53894037, 0.114799999, 0.04850250855, 0.03159469366, 0},
 {0.06057272106, 2.81652451, 92.3253784, 1797.18713, 19125.3691, 28.6847382, 54410.793, 245641.844, 1375094, 8130742, 0}},
{10, 0.0040234, {1.82837915, 0.07555398345, 0.01609379798, 0.006799560972, 0.004429256078, 0},
 {0.0001564582344, 0.01751643978, 1.57011914, 90.4214554, 2764.07764, 465.709259, 26196.8945, 11198567, 13479409, 12719175, 0}},

// Z=41,  13 orbitals
{11, 18.9856, {404.723267, 82.272316, 37.971199, 24.6811543, 19.9200497, 19.0045853},
 {218.421249, 4747.1416, 0, 0, 0, 1.81405532, 201.294617, 1824.36816, 5845.46338, 10199.041, 11666.75}},
{11, 2.6977, {57.5079002, 11.6902304, 5.39540005, 3.50699234, 2.83047795, 2.70039773},
 {20.2745953, 388.375244, 5163.62695, 41282.5156, 0, 50.888401, 2828.0896, 14406.3662, 31691.5313, 45017.3555, 49136.8516}},
{11, 2.4647, {52.5409508, 10.6805468, 4.92939997, 3.20409369, 2.58600998, 2.46716475},
 {1.98029768, 88.1779251, 3097.12842, 70272.1328, 0, 8.61602688, 1756.8186, 17960.6035, 58998.5547, 103244.578, 118982.734}},
{11, 2.3705, {50.5328522, 10.2723398, 4.74100018, 3.08163452, 2.48717356, 2.37287045},
 {2.76925254, 143.079041, 5436.61084, 131534.563, 0, 14.9521971, 3454.31055, 36536.3594, 122267.82, 216200.359, 250722.047}},
{10, 0.4684, {212.855362, 8.79580784, 1.87360001, 0.791588008, 0.515643001, 0},
 {3.55402899, 66.8008041, 904.766113, 8477.38574, 55445.9492, 0.217097908, 928.643921, 19960.2207, 77480.8906, 127574.836, 0}},
{10, 0.3784, {171.956589, 7.10575056, 1.51359999, 0.639489532, 0.416565567, 0},
 {0.357288361, 14.8225965, 455.976593, 8313.5498, 75939.4063, 0.0246144142, 870.914978, 36546.6445, 137495.297, 179672.125, 0}},
{10, 0.363, {164.958359, 6.81656265, 1.45200002, 0.613463819, 0.399612308, 0},
 {0.501355648, 24.1268692, 803.454651, 15553.2676, 152661.203, 0.03376957402, 1751.86255, 77276.9609, 305101.781, 423001.219, 0}},
{10, 0.2074, {94.2489319, 3.89464211, 0.829599977, 0.350502461, 0.228318438, 0},
 {0.00569435861, 0.741096258, 71.6300507, 4274.30615, 142226.25, 0.002645154018, 1681.3822, 238178.453, 1534821.38, 582568.063, 0}},
{10, 0.2046, {92.976532, 3.84206271, 0.818399966, 0.345770508, 0.225236028, 0},
 {0.004561474547, 0.81552422, 93.6537781, 6019.17334, 206829.016, 0.002152563073, 2458.72705, 360692.031, 2338196, 869593.438, 0}},
{10, 0.0581, {26.4024258, 1.09102559, 0.2324, 0.09818800539, 0.06395998597, 0},
 {0.612750053, 11.1583605, 149.253799, 1429.06396, 10510.4922, 11.4717159, 9049.18164, 97834.2813, 196611.281, 157352.063, 0}},
{10, 0.0339, {15.4052029, 0.636588097, 0.135600001, 0.05729041994, 0.03731916845, 0},
 {0.05333347246, 2.10468006, 62.324501, 1112.88916, 10910.8672, 11.979413, 23103.1055, 102423.703, 574271.688, 3695229.25, 0}},
{10, 0.0339, {15.4052029, 0.636588097, 0.135600001, 0.05729041994, 0.03731916845, 0},
 {0.07447589189, 3.40727806, 109.204124, 2067.71436, 21464.4473, 20.2748394, 46713.0156, 237588.016, 1027884.94, 7088904.5, 0}},
{10, 0.0032, {1.45417833, 0.06009091064, 0.01279999968, 0.005407945253, 0.00352275325, 0},
 {0.0004208431346, 0.04603499547, 4.00764608, 222.856277, 6501.55127, 2256.22852, 360910.219, 21367446, 27653664, 29049492, 0}},

// Z=42,  14 orbitals
{11, 19.9995, {426.336975, 86.6659546, 39.9990005, 25.999218, 20.9838543, 20.0194988},
 {240.097275, 5144.72656, 0, 0, 0, 1.73783219, 190.097046, 1722.3855, 5518.71191, 9627.55371, 11005.0039}},
{11, 2.8655, {61.0849571, 12.4173756, 5.73099995, 3.72513127, 3.00653696, 2.86836553},
 {22.3213329, 421.095978, 5490.91113, 42420.1602, 0, 47.3239059, 2639.03882, 13488.4756, 29747.5313, 42303.9961, 46096.3047}},
{11, 2.6251, {55.9602585, 11.3756247, 5.25020027, 3.41261292, 2.75430465, 2.62772512},
 {2.30739427, 101.003838, 3476.32935, 77040.1797, 0, 8.02373505, 1624.74292, 16628.1777, 54710.8398, 95766.4297, 109727.063}},
{11, 2.5202, {53.7240639, 10.921051, 5.04040003, 3.27624345, 2.64424157, 2.5227201},
 {3.20123339, 162.969803, 6082.30273, 144061.484, 0, 13.7183266, 3188.79077, 33827.0742, 113491.219, 200842.766, 231035.063}},
{10, 0.5046, {229.305756, 9.47558594, 2.01839995, 0.852765381, 0.55549413, 0},
 {3.95829892, 73.3244247, 977.726746, 9017.90918, 57976.3359, 0.19940342, 851.258484, 18528.1855, 72779.7734, 121042.602, 0}},
{10, 0.4097, {186.180283, 7.69351435, 1.63880002, 0.692385972, 0.451022506, 0},
 {0.422186017, 17.1760159, 515.355103, 9102.08594, 79867.7734, 0.02242681757, 782.37854, 33536.2383, 130016.234, 176863.469, 0}},
{10, 0.3923, {178.273178, 7.36677027, 1.56920004, 0.662980318, 0.43186754, 0},
 {0.588181078, 27.8265152, 905.909912, 17032.8398, 161457.938, 0.03009606339, 1572.45984, 71061.6875, 289479.813, 417790.563, 0}},
{10, 0.2303, {104.655403, 4.32466793, 0.921200037, 0.389203072, 0.253528148, 0},
 {0.007225201931, 0.925237, 87.4035797, 5065.71143, 162128.094, 0.002092840848, 1372.98071, 203375.188, 1406531, 614487.563, 0}},
{10, 0.227, {103.155777, 4.26269913, 0.907999992, 0.383626133, 0.249895304, 0},
 {0.005769114941, 1.01592946, 114.13044, 7128.85303, 235719.234, 0.001656617504, 2001.51587, 308182.719, 2145403, 914660.375, 0}},
{10, 0.0618, {28.0838203, 1.16050577, 0.247199997, 0.104440942, 0.06803317368, 0},
 {0.703809321, 12.6618958, 167.027222, 1578.99634, 11535.5352, 11.180459, 8929.43945, 96923.2109, 196239.875, 165727.781, 0}},
{10, 0.0348, {15.8141899, 0.653488696, 0.139200002, 0.05881140754, 0.03830994293, 0},
 {0.0656774044, 2.54781103, 73.6351395, 1275.04834, 12151.4619, 13.2324371, 24509.7188, 107404.367, 547929.938, 3136452.5, 0}},
{10, 0.0348, {15.8141899, 0.653488696, 0.139200002, 0.05881140754, 0.03830994293, 0},
 {0.09101111442, 4.1033473, 128.648575, 2367.61401, 23964.3633, 22.259634, 49670.7539, 251517.375, 985744.375, 6027433.5, 0}},
{10, 0.0018, {0.817975342, 0.0338011384, 0.007200000342, 0.003041969379, 0.001981548732, 0},
 {0.0006148125976, 0.06714596599, 5.7374711, 308.01532, 8571.79004, 14348.623, 7814266, 25720028, 33348208, 36858940, 0}},
{10, 0.0018, {0.817975342, 0.0338011384, 0.007200000342, 0.003041969379, 0.001981548732, 0},
 {8.017745859e-05, 0.01205120422, 1.22094846, 70.6675644, 2032.77783, 3417.11523, 1867871.75, 6514205, 8738789, 9799364, 0}},

// Z=43,  14 orbitals
{11, 21.044, {448.602966, 91.1921997, 42.0879974, 27.3570614, 22.0797634, 21.0650425},
 {263.218353, 5561.27295, 0, 0, 0, 1.66740632, 179.844498, 1627.83167, 5215.59229, 9100.87891, 10391.3418}},
{11, 3.0425, {64.8581314, 13.1843891, 6.08500004, 3.95523, 3.19224858, 3.04554248},
 {24.5051975, 455.363861, 5825.05176, 0, 0, 43.9736137, 2460.91431, 12623.2471, 27913.6055, 39755.9648, 43286.3984}},
{11, 2.7932, {59.5437088, 12.1040707, 5.58640003, 3.63114166, 2.93067837, 2.79599309},
 {2.67746568, 115.233955, 3886.97217, 83721.3594, 0, 7.47870445, 1503.18005, 15398.085, 50740.5664, 88811.2891, 101943.969}},
{11, 2.6769, {57.0644989, 11.6000948, 5.35379982, 3.47995234, 2.80865407, 2.67957687},
 {3.68525243, 184.882828, 6779.14258, 157491.297, 0, 12.6491575, 2943.57031, 31316.1777, 105331.508, 186493.297, 215054.938}},
{10, 0.5476, {248.846268, 10.2830572, 2.19039989, 0.925434649, 0.602831125, 0},
 {4.39064074, 80.2249985, 1054.03064, 9577.97559, 60510.0625, 0.167934537, 766.106262, 16960.5234, 67560.5156, 114272.773, 0}},
{10, 0.4449, {202.176239, 8.35451412, 1.77960002, 0.751873374, 0.489772767, 0},
 {0.496396124, 19.8073235, 579.82373, 9923.71777, 83685.0391, 0.02020336129, 694.72644, 30511.9727, 122119.953, 171240.531, 0}},
{10, 0.425, {193.133072, 7.98082447, 1.70000005, 0.718242764, 0.467865676, 0},
 {0.686615527, 31.9365902, 1016.76294, 18575.5742, 170208.906, 0.02655065246, 1396.54407, 64829.9766, 272704.469, 405984.375, 0}},
{10, 0.2564, {116.516045, 4.81478453, 1.02559996, 0.433311641, 0.282260597, 0},
 {0.009092921391, 1.14576757, 105.810783, 5955.49951, 183483.141, 0.001629359671, 1102.35706, 171081.141, 1266380.25, 811327.688, 0}},
{10, 0.2529, {114.925537, 4.74906015, 1.01160002, 0.427396685, 0.278407604, 0},
 {0.007236727048, 1.25537086, 138.003021, 8377.70313, 266886.938, 0.001248953165, 1592.07019, 258446.125, 1929504.25, 1213104.13, 0}},
{10, 0.0684, {31.0830631, 1.28444326, 0.273600012, 0.115594834, 0.07529885322, 0},
 {0.807469964, 14.3330383, 186.406754, 1740.92493, 12638.624, 9.87278652, 8206.63672, 91428.8672, 195055.016, 189214.219, 0}},
{10, 0.0389, {17.6773567, 0.730480134, 0.155599996, 0.06574033201, 0.0428234674, 0},
 {0.08067370206, 3.07626271, 86.7981033, 1458.61951, 13531.8389, 11.223525, 22794.4453, 109566.453, 395358.594, 1882568, 0}},
{10, 0.0389, {17.6773567, 0.730480134, 0.155599996, 0.06574033201, 0.0428234674, 0},
 {0.111048825, 4.93399858, 151.375626, 2710.17578, 26790.373, 18.6609764, 46080.8359, 257800.609, 728719.875, 3596759.75, 0}},
{10, 0.007012, {3.18627834, 0.131666362, 0.02804632671, 0.01184945367, 0.007718772627, 0},
 {0.0009261008236, 0.10305395, 8.74145603, 454.142639, 12115.5635, 370.902191, 161808.516, 18166836, 12834695, 8273540, 0}},
{10, 0.006729, {3.05805564, 0.126367822, 0.02691768296, 0.01137260627, 0.0074081528, 0},
 {0.0001214610529, 0.01861779578, 1.87551463, 105.118729, 2899.7207, 98.5036316, 38843.8516, 4626662.5, 3323490.25, 2198278, 0}},

// Z=44,  14 orbitals
{11, 22.1172, {471.480804, 95.8428116, 44.2344017, 28.7522144, 23.2057858, 22.1393166},
 {287.872009, 6000.26904, 0, 0, 0, 1.59923828, 170.411652, 1540.85437, 4937.02246, 8611.95215, 9830.63184}},
{11, 3.224, {68.7272339, 13.9709015, 6.44799995, 4.1911788, 3.38268185, 3.22722387},
 {26.8350067, 491.312775, 6168.32861, 0, 0, 41.0157661, 2302.53369, 11851.9229, 26274.7012, 37438.1367, 40693.1719}},
{11, 2.9669, {63.2465363, 12.8567829, 5.93379974, 3.85695052, 3.11292768, 2.96986675},
 {3.0960288, 131.068787, 4337.2124, 97572.5078, 0, 6.99939537, 1396.5321, 14326.0068, 47300.6523, 82867.9531, 93226.5938}},
{11, 2.8379, {60.4965973, 12.2977743, 5.67580032, 3.68925142, 2.9775784, 2.84073806},
 {4.22709703, 209.09137, 7540.30762, 170757.391, 0, 11.7136602, 2729.48413, 29138.4238, 98301, 174310.5, 196049.469}},
{10, 0.585, {265.84198, 10.9853697, 2.33999991, 0.98864001, 0.644003332, 0},
 {4.85759115, 87.5167999, 1132.91003, 10143.54, 62980.7773, 0.155831888, 712.178772, 15921.9951, 64010.1133, 108310.953, 0}},
{10, 0.4828, {219.39917, 9.06621647, 1.93120003, 0.81592375, 0.531495392, 0},
 {0.581026495, 22.7495422, 650.461365, 10807.6689, 87569.0625, 0.01823580079, 617.006958, 27822.9355, 114441.383, 165762.281, 0}},
{10, 0.4606, {209.310806, 8.64933586, 1.84240007, 0.778406143, 0.507056296, 0},
 {0.797201991, 36.4975395, 1138.06311, 20246.6113, 179559.797, 0.02334276587, 1236.63257, 59170.5859, 256474.516, 394525.25, 0}},
{10, 0.2836, {128.876556, 5.32555723, 1.13440001, 0.47927916, 0.312204003, 0},
 {0.01136720739, 1.40930796, 127.387573, 6982.27832, 208328.328, 0.001303084427, 903.103516, 146945.813, 1154545.5, 683887.5, 0}},
{10, 0.2794, {126.967949, 5.24668789, 1.11759996, 0.472181231, 0.307580382, 0},
 {0.009016356431, 1.54062247, 165.916214, 9814.36816, 302962.688, 0.0009781608824, 1300.55066, 222252.859, 1762292.63, 1013434.63, 0}},
{10, 0.0749, {34.0368614, 1.40650296, 0.299600005, 0.126579717, 0.08245444298, 0},
 {0.910864234, 15.9583178, 204.765228, 1891.68176, 13675.8447, 8.82271194, 7577.24805, 85910.4922, 181337.719, 162904.641, 0}},
{10, 0.04301, {19.5859642, 0.809349477, 0.172399998, 0.07283826172, 0.04744708166, 0},
 {0.09656544775, 3.61468291, 99.5735474, 1626.64807, 14724.5879, 9.53222752, 20959.3516, 105124.797, 355984.5, 1938333.38, 0}},
{10, 0.04301, {19.5859642, 0.809349477, 0.172399998, 0.07283826172, 0.04744708166, 0},
 {0.131691754, 5.75846052, 172.905487, 3016.97754, 29205.1191, 15.6216393, 42190.2656, 248403.688, 657352.125, 3728875, 0}},
{10, 0.002, {0.908861518, 0.03755681962, 0.00800000038, 0.0033799659, 0.002201720839, 0},
 {0.001168414718, 0.126144901, 10.3602219, 519.234192, 13235.6475, 16849.252, 9683044, 18919170, 21132330, 22473160, 0}},
{10, 0.002, {0.908861518, 0.03755681962, 0.00800000038, 0.0033799659, 0.002201720839, 0},
 {0.0004559287336, 0.06772233546, 6.61125994, 357.740875, 9436.85938, 12039.8447, 6975326, 14523259, 16583439, 17588262, 0}},

// Z=45,  14 orbitals
{11, 23.2199, {494.987457, 100.621262, 46.4398003, 30.1857166, 24.3627586, 23.2431202},
 {314.038208, 6447.44287, 0, 0, 0, 1.53989196, 161.669327, 1460.09253, 4677.97705, 8160.6499, 9308.90527}},
{11, 3.4119, {72.7327728, 14.7851486, 6.82380009, 4.43544769, 3.57983017, 3.41531181},
 {29.308548, 528.739258, 6515.34619, 0, 0, 38.0843735, 2156.85938, 11137.7354, 24749.6211, 35295.7461, 38326.3281}},
{11, 3.1461, {67.0666122, 13.6333294, 6.29220009, 4.08990955, 3.30094767, 3.14924622},
 {3.56632352, 148.508102, 4819.84473, 0, 0, 6.57411432, 1300.5658, 13350.1484, 44139.0898, 77349.1563, 85781.7344}},
{11, 3.0038, {64.0331497, 13.0166855, 6.00759983, 3.9049201, 3.15164375, 3.00680375},
 {4.82996225, 235.562592, 8353.98047, 0, 0, 10.8769188, 2535.83569, 27146.8809, 91811.0781, 162951.688, 181019.75}},
{10, 0.6271, {284.973511, 11.7759409, 2.50839996, 1.05978823, 0.690349519, 0},
 {5.35817719, 95.2024002, 1214.72339, 10722.4795, 65303.3828, 0.143014431, 655.645508, 14831.9404, 60216.7891, 102613.008, 0}},
{10, 0.521, {236.758423, 9.78355122, 2.08400011, 0.880481124, 0.573548257, 0},
 {0.676429868, 26.0001163, 726.10199, 11706.2002, 91176.9219, 0.01582979597, 554.688538, 25534.0957, 107596.227, 159663.047, 0}},
{10, 0.4962, {225.488541, 9.31784725, 1.98479998, 0.838569522, 0.546246946, 0},
 {0.922076643, 41.524765, 1267.58264, 21949.5234, 188445.813, 0.02098630182, 1109.4989, 54388.3359, 241933.156, 381712.625, 0}},
{10, 0.3117, {141.646072, 5.85323048, 1.24680007, 0.526767671, 0.343138188, 0},
 {0.01410685759, 1.72059906, 152.194992, 8114.72852, 234223.766, 0.001021503587, 751.620422, 127058.289, 1049446.5, 715684.875, 0}},
{10, 0.307, {139.510239, 5.76497173, 1.22799993, 0.518824756, 0.337964118, 0},
 {0.01114092674, 1.87617385, 197.972824, 11399.3711, 340686.688, 0.0007530418225, 1075.53162, 191977.859, 1602119, 1057860.88, 0}},
{10, 0.081, {36.8088913, 1.52105117, 0.324000001, 0.136888623, 0.08916968852, 0},
 {1.02415228, 17.7333393, 224.654068, 2052.98267, 14765.3525, 8.12298489, 7134.2373, 81918.4375, 174728.828, 160096.859, 0}},
{10, 0.0479, {21.7672329, 0.899485826, 0.191599995, 0.08095017821, 0.05273121223, 0},
 {0.115625292, 4.24856138, 114.261101, 1814.72986, 16040.8086, 8.03506279, 19158.5625, 102679.789, 292163.031, 1562784.25, 0}},
{10, 0.0479, {21.7672329, 0.899485826, 0.191599995, 0.08095017821, 0.05273121223, 0},
 {0.156398177, 6.73033333, 197.786179, 3364.052, 31917.4922, 12.9800539, 38416.9727, 243174.047, 550757.75, 3011173, 0}},
{10, 0.0025, {1.13607681, 0.04694602638, 0.009999999776, 0.004224957433, 0.002752150875, 0},
 {0.001553658396, 0.166203409, 13.3680382, 647.82373, 15844.0928, 11416.7881, 6163891.5, 15592759, 15830255, 16222802, 0}},
{10, 0.0025, {1.13607681, 0.04694602638, 0.009999999776, 0.004224957433, 0.002752150875, 0},
 {0.0008063389687, 0.118710421, 11.3625679, 594.924255, 15067.2021, 10823.0381, 5871011.5, 15941303, 16474088, 16681605, 0}},

// Z=46,  14 orbitals
{11, 24.3503, {519.084656, 105.519745, 48.7005997, 31.6552315, 25.5487976, 24.374651},
 {341.820618, 6929.88525, 0, 0, 0, 1.48557293, 153.623566, 1385.51062, 4438.48926, 7752.1416, 8818.31152}},
{11, 3.6043, {76.8342361, 15.6188955, 7.20860004, 4.68556643, 3.78169966, 3.6079042},
 {31.9330921, 567.74939, 6867.7666, 0, 0, 35.6739388, 2025.88623, 10492.3643, 23366.0879, 33331.4141, 36074.6445}},
{11, 3.3303, {70.9932785, 14.4315434, 6.66060019, 4.32936811, 3.49421382, 3.33363032},
 {4.09528351, 167.752396, 5341.66504, 0, 0, 6.16704893, 1215.63281, 12484.5205, 41325.7383, 72403.0078, 74949.1016}},
{11, 3.1733, {67.6464462, 13.7511978, 6.34660006, 4.12526894, 3.32948637, 3.17647314},
 {5.5011816, 264.560547, 9230.24219, 0, 0, 10.1477365, 2365.25098, 25387.7773, 86059.5313, 152841.328, 156774.797}},
{10, 0.6699, {304.423157, 12.5796566, 2.6796, 1.13211954, 0.737466395, 0},
 {5.89325142, 103.292404, 1299.56934, 11316.4756, 67682.6172, 0.132441312, 607.180298, 13884.2705, 56833.668, 97394.2891, 0}},
{10, 0.5591, {254.072235, 10.4990091, 2.23639989, 0.944869459, 0.615491033, 0},
 {0.784810185, 29.6038456, 807.572998, 12640.6689, 94440.4453, 0.01461805869, 505.420624, 23651.623, 101696.453, 154143.234, 0}},
{10, 0.5315, {241.529938, 9.98072529, 2.12599993, 0.898225904, 0.585107327, 0},
 {1.06200707, 47.0543175, 1406.76782, 23728.873, 197113.906, 0.01815042086, 1009.27832, 50481.5547, 229668.359, 370052.688, 0}},
{10, 0.34, {154.506454, 6.38465929, 1.36000001, 0.5745942, 0.374292523, 0},
 {0.01737349667, 2.08655882, 180.792755, 9387.08984, 262496.344, 0.0008519665571, 640.333557, 111922.383, 965541.938, 597959, 0}},
{10, 0.3347, {152.097977, 6.28513384, 1.33879995, 0.56563729, 0.368457973, 0},
 {0.01368676871, 2.27084923, 234.873932, 13178.5693, 381904.469, 0.0006217189366, 911.242188, 169021.984, 1475317.5, 876121.938, 0}},
{10, 0.0864, {39.2628174, 1.62245464, 0.345600009, 0.146014526, 0.09511433542, 0},
 {1.14679289, 19.5808277, 244.713272, 2213.70801, 15841.3955, 7.48869276, 6840.04199, 78827.8672, 166242.125, 137745.578, 0}},
{10, 0.0511, {23.2214108, 0.959576786, 0.204400003, 0.08635812998, 0.05625396594, 0},
 {0.136756092, 4.93306541, 129.562637, 2002.11621, 17290.6992, 7.6103673, 18528.8125, 100141.703, 282794.25, 1683916.13, 0}},
{10, 0.0511, {23.2214108, 0.959576786, 0.204400003, 0.08635812998, 0.05625396594, 0},
 {0.183273137, 7.76306963, 223.361755, 3706.39746, 34494.4453, 12.1436052, 37082.0391, 238696.328, 533498.5, 3273070.25, 0}},
{10, 0.0054466, {2.47511697, 0.102279089, 0.02178652771, 0.009204714559, 0.005995981395, 0},
 {0.001931572333, 0.200469404, 15.6423988, 734.317932, 17329.5137, 1353.56152, 96767.9688, 11283164, 9874705, 7802603.5, 0}},
{10, 0.0050184, {2.28052115, 0.09423781186, 0.02007365227, 0.008481032215, 0.005524571985, 0},
 {0.001495557954, 0.212519825, 19.7038517, 999.829834, 24445.3828, 2400.72192, 231654.063, 17024572, 16258404, 13481004, 0}},

// Z=47,  14 orbitals
{11, 25.514, {543.891663, 110.562531, 51.0279999, 33.1680336, 26.7697716, 25.5395145},
 {371.173615, 7404.35596, 0, 0, 0, 1.43481195, 146.067505, 1315.5343, 4213.75439, 7359.54688, 8367.5293}},
{11, 3.8058, {81.1296921, 16.4920769, 7.61159992, 4.94751501, 3.99311733, 3.80960584},
 {34.7085419, 608.202698, 7222.55518, 0, 0, 33.4021225, 1901.80884, 9878.91504, 22047.4551, 31481.4609, 34134.9883}},
{11, 3.5237, {75.1160583, 15.2696238, 7.0474, 4.58078671, 3.69713283, 3.52722383},
 {4.68518305, 188.830048, 5899.13672, 0, 0, 5.80971479, 1135.11755, 11660.7373, 38641.9258, 67712.7969, 72616.8516}},
{11, 3.3511, {71.436676, 14.521678, 6.70219994, 4.35640812, 3.51603746, 3.35445118},
 {6.24195051, 296.090698, 10163.7803, 0, 0, 9.39600563, 2202.82422, 23704.5723, 80536.8672, 143151.406, 152821.203}},
{10, 0.7175, {326.054047, 13.4735088, 2.86999989, 1.2125628, 0.789867342, 0},
 {6.46324205, 111.762505, 1386.70471, 11912.9297, 69822.0469, 0.121659912, 557.751953, 12907.9512, 53366.1133, 92075.2422, 0}},
{10, 0.6024, {273.749084, 11.3121147, 2.40960002, 1.01804566, 0.663158298, 0},
 {0.906831205, 33.5687065, 894.524719, 13595.2197, 97441.9141, 0.01328355819, 453.930481, 21669.6406, 95233.3359, 146965.922, 0}},
{10, 0.5714, {259.661743, 10.7299833, 2.28559995, 0.965656221, 0.629031599, 0},
 {1.21800828, 53.098896, 1554.82153, 25551.3164, 205387.109, 0.01623871364, 905.297241, 46351.3789, 215771.313, 354873.563, 0}},
{10, 0.3728, {169.411789, 7.00059128, 1.49119997, 0.630025625, 0.410400748, 0},
 {0.02127772756, 2.51450348, 213.351471, 10782.4453, 292004.375, 0.0006910418742, 531.765259, 96376.8438, 867885.375, 780110.813, 0}},
{10, 0.3667, {166.639755, 6.88604307, 1.46679997, 0.619716763, 0.40368551, 0},
 {0.01668929122, 2.72970104, 276.806854, 15126.2559, 424861.844, 0.0005021790857, 752.546265, 145539.859, 1326624.63, 1146642.13, 0}},
{10, 0.0952, {43.2618065, 1.78770459, 0.380800009, 0.160886377, 0.104801908, 0},
 {1.27912045, 21.5915146, 266.562805, 2387.3396, 16980.1543, 6.52393007, 6185.69434, 73095.4531, 159421.859, 149676.469, 0}},
{10, 0.0626, {28.4473648, 1.17552853, 0.250400007, 0.105792932, 0.06891386211, 0},
 {0.161787838, 5.739007, 147.378464, 2218.76978, 18746.2285, 4.71469355, 14223.8457, 92560.6094, 182653.781, 980771.875, 0}},
{10, 0.0559, {25.4026775, 1.04971313, 0.2236, 0.09447004646, 0.06153809652, 0},
 {0.215594113, 8.98779106, 253.145477, 4098.16504, 37397.2734, 10.5392103, 34347.0938, 236129.094, 444850.438, 2185658, 0}},
{10, 0.0033, {1.49962151, 0.06196875498, 0.01319999993, 0.005576943979, 0.003632839303, 0},
 {0.002605415648, 0.273270607, 21.0784111, 955.892883, 21619.041, 7536.34375, 3400273.5, 11635234, 10617845, 10882367, 0}},
{10, 0.0033, {1.49962151, 0.06196875498, 0.01319999993, 0.005576943979, 0.003632839303, 0},
 {0.00201534736, 0.291224808, 26.7979927, 1314.85693, 30838.377, 10630.3838, 4795128.5, 17853854, 16372035, 16260426, 0}},

// Z=48,  14 orbitals
{11, 26.7112, {569.412842, 115.750481, 53.4224014, 34.7243843, 28.0258961, 26.7379112},
 {402.193146, 0, 0, 0, 0, 1.38744187, 139.001984, 1250.03833, 4003.46631, 6993.01221, 7938.74268}},
{11, 4.018, {85.6532364, 17.4116268, 8.03600025, 5.22337341, 4.21576166, 4.02201796},
 {37.6538239, 650.309753, 7583.02881, 0, 0, 31.2527809, 1783.91858, 9296.45605, 20793.3301, 29788.5117, 32232.9414}},
{11, 3.727, {79.4498825, 16.1506042, 7.454, 4.84507561, 3.91043901, 3.73072696},
 {5.34350967, 211.952164, 6498.77979, 0, 0, 5.47050619, 1059.40247, 10888.5313, 36130.9141, 63333.4414, 70256.5078}},
{11, 3.5375, {75.4102325, 15.3294249, 7.07499981, 4.59872675, 3.71161199, 3.54103732},
 {7.06004715, 330.427429, 11165.376, 0, 0, 8.7364912, 2050.06909, 22124.5645, 75358.7734, 134077.578, 148945.016}},
{10, 0.7702, {350.002563, 14.4631319, 3.08080006, 1.30162489, 0.847882688, 0},
 {7.07008743, 120.648285, 1476.84485, 12524.2734, 71821.1172, 0.111077316, 508.901367, 11937.0459, 49828.3125, 87278.6406, 0}},
{10, 0.6507, {295.69809, 12.2191114, 2.60279989, 1.09967184, 0.716329873, 0},
 {1.04392672, 37.9316368, 987.710999, 14587.127, 100177.531, 0.01194320992, 403.858978, 19719.7754, 88850.4609, 138626.625, 0}},
{10, 0.6165, {280.156555, 11.57689, 2.46600008, 1.04187453, 0.67868042, 0},
 {1.39177334, 59.7120972, 1713.33618, 27461.4473, 213595.25, 0.01433763001, 801.577026, 42172.6875, 201692.719, 335675.688, 0}},
{10, 0.4105, {186.543823, 7.70853758, 1.64199996, 0.693737984, 0.451903194, 0},
 {0.0259088669, 3.01359344, 250.567963, 12343.2217, 324396.969, 0.0005484857247, 432.801788, 81782.9531, 773230.188, 1046752.13, 0}},
{10, 0.4037, {183.45369, 7.58084393, 1.61479998, 0.682246089, 0.444417328, 0},
 {0.02025088482, 3.26425838, 324.690765, 17305.2656, 472220.563, 0.0003991335398, 607.127319, 123268.711, 1181358.5, 1555562.88, 0}},
{10, 0.1076, {48.8967476, 2.02055693, 0.430399984, 0.181842163, 0.118452579, 0},
 {1.42603564, 23.7744255, 289.901031, 2572.5332, 18192.9414, 5.35037184, 5335.19873, 65342.0234, 151706.297, 158377.469, 0}},
{10, 0.0669, {30.4014168, 1.25627565, 0.2676, 0.113059856, 0.07364755869, 0},
 {0.190790504, 6.65182829, 166.873779, 2444.54761, 20169.2598, 4.45543766, 13676.0293, 92448.1406, 160554.719, 692735.563, 0}},
{10, 0.0669, {30.4014168, 1.25627565, 0.2676, 0.113059856, 0.07364755869, 0},
 {0.252439588, 10.3763657, 286.513611, 4532.94678, 40641.7773, 6.85289192, 27041.0039, 218305.156, 345191.938, 1344777.13, 0}},
{10, 0.0093, {4.22620583, 0.17463921, 0.03720000014, 0.01571684144, 0.01023800205, 0},
 {0.003445164068, 0.363392204, 27.6695786, 1215.86316, 26505.5605, 395.825348, 203484.531, 14914517, 6239630, 4870765.5, 0}},
{10, 0.0093, {4.22620583, 0.17463921, 0.03720000014, 0.01571684144, 0.01023800205, 0},
 {0.00266976282, 0.388387829, 35.3342743, 1681.05505, 38035.3438, 537.642273, 312976.813, 22128590, 9579849, 7131414, 0}},

// Z=49,  14 orbitals
{11, 27.9399, {595.605469, 121.074936, 55.8797989, 36.3216858, 29.3150711, 27.9678402},
 {434.917938, 0, 0, 0, 0, 1.34354222, 132.399948, 1188.87671, 3806.95923, 6656.97412, 7536.04053}},
{11, 4.2375, {90.3324051, 18.3628082, 8.47500038, 5.50872231, 4.44606543, 4.24173737},
 {40.7497978, 693.810486, 7944.25537, 0, 0, 29.2774467, 1675.00842, 8755.65527, 19626.1855, 28158.8555, 30508.5703}},
{11, 3.938, {83.9478455, 17.0649548, 7.87599993, 5.11937428, 4.13182449, 3.94193816},
 {6.07568169, 237.183563, 7137.3623, 0, 0, 5.16041517, 989.900635, 10175.6943, 33801.3555, 59266.4219, 68483.8516}},
{11, 3.7301, {79.5159683, 16.1640396, 7.46019983, 4.84910536, 3.91369152, 3.73383021},
 {7.96010685, 367.610535, 12227.7061, 0, 0, 8.13598537, 1910.12476, 20666.8047, 70551.6797, 125639.133, 145913.063}},
{10, 0.8256, {375.178009, 15.5034552, 3.30239987, 1.39524984, 0.908870339, 0},
 {7.71465921, 129.926773, 1569.21802, 13137.7842, 73556.25, 0.101870246, 464.898895, 11044.6797, 46562.8008, 82384.8047, 0}},
{10, 0.7022, {319.101257, 13.1862001, 2.80879998, 1.18670607, 0.773024142, 0},
 {1.19739389, 42.7072029, 1086.68335, 15595.6992, 102573.328, 0.01075263787, 359.387512, 17930.0918, 82653.3672, 131364.328, 0}},
{10, 0.6643, {301.878357, 12.4744978, 2.6572001, 1.12265563, 0.731301546, 0},
 {1.58443534, 66.9061966, 1881.25659, 29413.6855, 221525.25, 0.01272562053, 710.295349, 38356.5313, 188043.344, 317844.719, 0}},
{10, 0.4508, {204.857376, 8.46530724, 1.80320001, 0.761844337, 0.496267855, 0},
 {0.03134656325, 3.59037566, 292.593079, 14046.4658, 358032.625, 0.0004392093979, 353.923401, 69448.6328, 684298.938, 1362901.38, 0}},
{10, 0.4431, {201.358261, 8.32071304, 1.77240002, 0.748831451, 0.48779124, 0},
 {0.02443429269, 3.8813448, 378.674194, 19679.1719, 521345.25, 0.0003238790378, 492.438324, 104545.422, 1045670.75, 2049506.75, 0}},
{10, 0.1219, {55.3951073, 2.28908825, 0.487599999, 0.206008926, 0.134194881, 0},
 {1.58318138, 26.0998497, 314.572235, 2766.40479, 19425.8887, 4.18845844, 4565.28613, 58088.5039, 144032.813, 158709.734, 0}},
{10, 0.0774, {35.1729393, 1.45344889, 0.309599996, 0.130804673, 0.08520659059, 0},
 {0.224245504, 7.68499517, 188.437393, 2689.43237, 21684.5605, 3.2957387, 11402.6494, 87467.1563, 132859.359, 445682.75, 0}},
{10, 0.0774, {35.1729393, 1.45344889, 0.309599996, 0.130804673, 0.08520659059, 0},
 {0.294052601, 11.9301281, 323.080231, 4995.24609, 43961.3984, 4.95171165, 22385.4727, 204827.625, 312471.688, 859850.188, 0}},
{10, 0.0162, {7.36177778, 0.304210246, 0.06480000168, 0.02737772278, 0.01783393882, 0},
 {0.004493633285, 0.474454135, 35.5743599, 1514.224, 31812.9629, 71.5568008, 237794.953, 6435903.5, 6812904.5, 3388422.75, 0}},
{10, 0.0162, {7.36177778, 0.304210246, 0.06480000168, 0.02737772278, 0.01783393882, 0},
 {0.003482840722, 0.507674456, 45.5484734, 2100.52588, 45842.3477, 93.4430008, 355699.75, 9263124, 10435674, 5075456, 0}},

// Z=50,  14 orbitals
{11, 29.2, {622.469666, 126.535896, 58.4001999, 37.959938, 30.6372986, 29.2293015},
 {469.335571, 0, 0, 0, 0, 1.30278218, 126.259201, 1131.63745, 3622.84937, 6329.84863, 7176.64746}},
{11, 4.4647, {95.1757126, 19.3473587, 8.92940044, 5.80408049, 4.68444777, 4.46916485},
 {44.0108719, 738.826843, 8308.22559, 0, 0, 27.4632511, 1574.34033, 8254.03809, 18540.6504, 26645.5977, 28882.5742}},
{11, 4.1561, {88.597168, 18.0100708, 8.31220055, 5.4029026, 4.36065912, 4.16025639},
 {6.88818979, 264.686493, 7817.49707, 0, 0, 4.87964249, 926.749756, 9525.74512, 31670.9961, 55551.707, 64946.3008}},
{11, 3.9288, {83.7517242, 17.0250874, 7.85760021, 5.10741425, 4.1221714, 3.93272877},
 {8.94866943, 407.852692, 13358.3125, 0, 0, 7.59024191, 1782.53491, 19333.6797, 66145.1328, 117902.984, 138816.156}},
{10, 0.8838, {401.625885, 16.5963593, 3.53520012, 1.49360693, 0.972940385, 0},
 {8.39794064, 139.619003, 1664.24524, 13759.5059, 75257.2188, 0.09384955466, 425.282135, 10229.7158, 43525.3945, 77673.8906, 0}},
{10, 0.7564, {343.731415, 14.203989, 3.02559996, 1.27830303, 0.832690775, 0},
 {1.36876702, 47.9281425, 1191.8927, 16629.6406, 104684.82, 0.009724338539, 320.816071, 16334.0898, 76944.7344, 124711.328, 0}},
{10, 0.7144, {324.645325, 13.4152966, 2.85759997, 1.20732379, 0.786454678, 0},
 {1.79763031, 74.7222824, 2059.40894, 31429.9531, 228986.734, 0.01139104739, 631.474609, 34957.168, 175520.609, 302964.813, 0}},
{10, 0.4933, {224.170685, 9.26338959, 1.97319996, 0.83366859, 0.543054461, 0},
 {0.03776062652, 4.25705576, 340.030151, 15916.9688, 393437, 0.0003568842076, 292.079224, 59357.5664, 608254.313, 1563263, 0}},
{10, 0.4848, {220.308029, 9.10377312, 1.93920004, 0.819303751, 0.533697128, 0},
 {0.02930581942, 4.59071922, 439.543304, 22287.4023, 573262.688, 0.0002687954111, 402.348206, 89147.5625, 928811.188, 2384795.75, 0}},
{10, 0.1365, {62.0297966, 2.56325293, 0.546000004, 0.230682671, 0.150267437, 0},
 {1.7538873, 28.5942535, 340.728271, 2970.66602, 20701.373, 3.48241735, 3979.1145, 52240.3828, 135489.859, 154490.063, 0}},
{10, 0.0886, {40.2625656, 1.6637671, 0.354400009, 0.149732485, 0.0975362286, 0},
 {0.262702942, 8.84935951, 212.11702, 2951.28784, 23259.3906, 2.44925475, 9630.53125, 82213.8906, 119546.063, 314798.563, 0}},
{10, 0.0886, {40.2625656, 1.6637671, 0.354400009, 0.149732485, 0.0975362286, 0},
 {0.342183888, 13.6839066, 363.285828, 5493.02246, 47471.6484, 3.59649491, 18782.873, 191162.406, 299461.094, 616534.125, 0}},
{10, 0.0239, {10.8608942, 0.448803991, 0.09559999406, 0.04039059207, 0.02631056309, 0},
 {0.005804231856, 0.608823895, 44.7936058, 1847.76965, 37422.2109, 21.2291412, 173031.516, 872345.125, 11045122, 2410028, 0}},
{10, 0.0239, {10.8608942, 0.448803991, 0.09559999406, 0.04039059207, 0.02631056309, 0},
 {0.004492963664, 0.651289523, 57.4127159, 2567.69458, 54067.1875, 26.5790653, 256064.547, 1202452.63, 16681656, 3680548.5, 0}},

// Z=51,  14 orbitals
{11, 30.4912, {649.992554, 132.130753, 60.982399, 39.6383591, 31.9919453, 30.5216904},
 {505.496216, 0, 0, 0, 0, 1.26502931, 120.533768, 1078.21399, 3450.94507, 6030.91211, 6829.84473}},
{11, 4.6983, {100.155449, 20.359642, 9.39659977, 6.107759, 4.9295454, 4.70299816},
 {47.439827, 785.260803, 8672.91797, 0, 0, 25.6544991, 1482.06189, 7791.75391, 17536.1699, 25232.7207, 27353.8516}},
{11, 4.3804, {93.3786545, 18.9820538, 8.76080036, 5.69449139, 4.59599876, 4.38478041},
 {7.78759003, 294.564087, 8538.69238, 0, 0, 4.62826967, 869.748779, 8935.69824, 29728.7031, 52155.4648, 61254.6172}},
{11, 4.1322, {88.0876846, 17.9065018, 8.26440048, 5.37183285, 4.33558273, 4.13633204},
 {10.0333748, 451.272095, 14553.624, 0, 0, 7.10404396, 1667.56592, 18124.2422, 62121.6719, 110940.953, 131077.984}},
{10, 0.9437, {428.846283, 17.7211857, 3.77480006, 1.59483683, 1.0388819, 0},
 {9.12128925, 149.717255, 1761.59387, 14384.0293, 76311.1875, 0.08717016876, 390.636078, 9503.69238, 40772.2461, 73333.0547, 0}},
{10, 0.8119, {368.952332, 15.246191, 3.24760008, 1.37209713, 0.893788576, 0},
 {1.55957532, 53.6119347, 1303.03162, 17677.084, 106432.773, 0.008890544064, 288.707092, 14956.6553, 71811.6797, 118304.641, 0}},
{10, 0.7656, {347.91217, 14.3767509, 3.06239986, 1.2938509, 0.842818737, 0},
 {2.03267574, 83.1816788, 2247.32642, 33490.4336, 236239.328, 0.01033644192, 565.778931, 32018.3027, 164208.984, 288257.781, 0}},
{10, 0.5369, {243.983871, 10.0821285, 2.14759994, 0.907351851, 0.591051936, 0},
 {0.04525857046, 5.02239656, 393.18457, 17949.7637, 429912.75, 0.0002794173197, 244.952682, 51304.2305, 542743, 1568719.38, 0}},
{10, 0.5275, {239.712219, 9.90561104, 2.1099999, 0.891465962, 0.580703855, 0},
 {0.03499866277, 5.40406275, 507.655334, 25119.6133, 626949.313, 0.0002189208753, 334.194427, 76887.9844, 828476.313, 2417228.5, 0}},
{10, 0.152, {69.0734787, 2.85431838, 0.60800004, 0.256877422, 0.167330787, 0},
 {1.94081104, 31.2639618, 368.240997, 3183.979, 22000.8223, 2.83087897, 3493.35181, 47218.6758, 127794.977, 150779.203, 0}},
{10, 0.0984, {44.7159843, 1.84779561, 0.393600017, 0.166294321, 0.108324662, 0},
 {0.306472033, 10.1463585, 237.725143, 3225.05713, 24828.293, 2.04246855, 8548.25391, 78516.2891, 115483.641, 241015.188, 0}},
{10, 0.0984, {44.7159843, 1.84779561, 0.393600017, 0.166294321, 0.108324662, 0},
 {0.397216707, 15.6378651, 406.766937, 6016.44824, 51029.2969, 2.94445682, 16596.709, 181993.641, 300497.594, 492359.344, 0}},
{10, 0.0314, {14.2691259, 0.589642107, 0.125599995, 0.0530654639, 0.03456701711, 0},
 {0.007412088104, 0.769712508, 55.4780731, 2218.22095, 43305.5313, 9.27482128, 126239.773, 148987.328, 18163230, 1913123.5, 0}},
{10, 0.0314, {14.2691259, 0.589642107, 0.125599995, 0.0530654639, 0.03456701711, 0},
 {0.005726999138, 0.822774887, 71.1408691, 3086.07959, 62696.125, 11.1955481, 185709.5, 206236.031, 26991814, 2937263.5, 0}},

// Z=52,  14 orbitals
{11, 31.8138, {678.18689, 137.862122, 63.6275978, 41.3577309, 33.3796425, 31.8456135},
 {543.457947, 0, 0, 0, 0, 1.23006237, 115.191147, 1028.25171, 3290.14868, 5755.02539, 6500.19238}},
{11, 4.9392, {105.29081, 21.4035606, 9.87839985, 6.42092752, 5.18230247, 4.94413948},
 {51.0387535, 833.152344, 9037.2666, 0, 0, 24.135458, 1396.82263, 7363.05762, 16601.8809, 23920.166, 25938.957}},
{11, 4.612, {98.3157654, 19.9856701, 9.22399998, 5.99556971, 4.83899784, 4.61661196},
 {8.78302097, 327.007843, 9303.80078, 0, 0, 4.37503052, 817.69342, 8395.02051, 27941.9473, 49081.9805, 57618.5156}},
{11, 4.3414, {92.5472794, 18.8130493, 8.68280029, 5.64379168, 4.55507898, 4.34574127},
 {11.2163792, 498.023865, 15818.6797, 0, 0, 6.65949011, 1562.56665, 17016.8184, 58433.0078, 104470.789, 123283.906}},
{11, 1.006, {21.4452858, 4.35940695, 2.01200008, 1.30779338, 1.05551422, 1.00700605},
 {9.88542557, 160.228302, 1861.3855, 15013.1504, 0, 268.347015, 7620.71045, 29398.875, 55816.543, 72969.3047, 78514.3203}},
{10, 0.8697, {395.218414, 16.331583, 3.47880006, 1.46977818, 0.957418263, 0},
 {1.7713927, 59.789505, 1420.43896, 18742.4453, 107931.641, 0.008181498386, 260.860443, 13730.5879, 67091.4219, 112186.773, 0}},
{10, 0.8187, {372.04245, 15.3738842, 3.27480006, 1.38358903, 0.901274383, 0},
 {2.29133701, 92.3204727, 2445.53735, 35605.1328, 243246.75, 0.00946433004, 509.012604, 29406.4902, 153818.844, 274294.938, 0}},
{10, 0.5825, {264.705902, 10.9384241, 2.32999992, 0.984415054, 0.641251206, 0},
 {0.05395657942, 5.89682484, 452.654053, 20165.2617, 467900.031, 0.0002316953905, 207.141403, 44610.1406, 485886.938, 1451067.88, 0}},
{10, 0.5721, {259.979828, 10.7431288, 2.28839993, 0.966839254, 0.629802227, 0},
 {0.0416021198, 6.33225298, 583.754639, 28203.6348, 682898.125, 0.0001897431648, 279.89798, 66722.4688, 741277.25, 2247015.25, 0}},
{10, 0.1683, {76.4806976, 3.16040635, 0.673200011, 0.284424126, 0.18527481, 0},
 {2.13859034, 34.0773811, 397.046326, 3406.03345, 23322.0332, 2.39971447, 3090.32886, 42859.1563, 120638.492, 147175.891, 0}},
{10, 0.1102, {50.07827, 2.06938076, 0.440800011, 0.186236113, 0.121314816, 0},
 {0.356053442, 11.5856295, 265.413971, 3513.79102, 26433.5625, 1.61186409, 7437.31689, 73759.3594, 112976.242, 191477.125, 0}},
{10, 0.1102, {50.07827, 2.06938076, 0.440800011, 0.186236113, 0.121314816, 0},
 {0.458347231, 17.7873383, 453.762604, 6571.93799, 54741.0703, 2.2683208, 14359.0488, 170265.156, 300900.313, 417148.813, 0}},
{10, 0.0398, {18.0863438, 0.747380733, 0.159199998, 0.06726132333, 0.04381424561, 0},
 {0.009369441308, 0.960885108, 67.7794189, 2628.51807, 49492.0039, 4.53558826, 91436.6875, 153573.922, 16884066, 1665414.5, 0}},
{10, 0.0398, {18.0863438, 0.747380733, 0.159199998, 0.06726132333, 0.04381424561, 0},
 {0.007222595625, 1.02602172, 86.9253845, 3659.91162, 71778.8203, 5.26037979, 133817.531, 240597.859, 24838258, 2550610.75, 0}},

// Z=53,  14 orbitals
{11, 33.1694, {707.084778, 143.736481, 66.3387985, 43.1200027, 34.8019638, 33.2025681},
 {583.245239, 0, 0, 0, 0, 1.19740903, 110.184814, 981.340576, 3138.84619, 5488.67529, 6199.58057}},
{11, 5.1881, {110.59671, 22.4821453, 10.3761997, 6.74449587, 5.44345284, 5.19328833},
 {54.8252258, 882.531311, 9402.8623, 0, 0, 22.7374363, 1317.67639, 6963.47266, 15728.3428, 22737.3457, 24558.4648}},
{11, 4.8521, {103.434067, 21.0261211, 9.70419979, 6.30769825, 5.0909152, 4.85695219},
 {9.87876797, 362.16748, 10115.458, 0, 0, 4.1613121, 769.584045, 7895.64453, 26293.9961, 46208.0469, 54148.6484}},
{11, 4.5571, {97.1454391, 19.7477665, 9.11419964, 5.92420006, 4.78139591, 4.56165695},
 {12.5069389, 548.316711, 17158.7988, 0, 0, 6.21549034, 1465.90515, 15995.8418, 55028.0117, 98476.6797, 116015.422}},
{11, 1.0721, {22.8543644, 4.64584494, 2.14420009, 1.39372301, 1.12486768, 1.07317209},
 {10.6919308, 171.167542, 1963.88062, 15650.5244, 0, 246.685699, 7083.80664, 27512.1523, 52495.8242, 68880.9609, 74222.1406}},
{10, 0.9305, {422.847809, 17.4733105, 3.72200012, 1.57252908, 1.02435064, 0},
 {2.00614882, 66.4919968, 1544.36499, 19827.3535, 108353.477, 0.00755343819, 236.12439, 12618.834, 62695.4531, 106305.945, 0}},
{10, 0.8746, {397.445129, 16.4235973, 3.49839997, 1.47805905, 0.962812483, 0},
 {2.57510161, 102.176842, 2654.61157, 37784.0742, 250097.375, 0.008723785169, 458.35965, 27024.2891, 144081.719, 260777.641, 0}},
{10, 0.6313, {286.882141, 11.8548098, 2.52519989, 1.06688619, 0.694973171, 0},
 {0.06408919394, 6.89599848, 519.143921, 22592.1602, 509126.281, 0.000192933745, 175.291458, 38821.3633, 435245.531, 1298463.13, 0}},
{10, 0.6194, {281.474396, 11.6313477, 2.4776001, 1.04677546, 0.681872904, 0},
 {0.04922960699, 7.38825369, 668.624268, 31571.2012, 743100.75, 0.0001664647862, 235.063843, 58034.3086, 664276, 2012410, 0}},
{10, 0.1864, {84.7058945, 3.50029564, 0.745599985, 0.315012813, 0.205200374, 0},
 {2.35337949, 37.0662956, 427.166504, 3637.14014, 24670.3906, 2.0349474, 2725.11133, 38797.4688, 113233.711, 142574.188, 0}},
{10, 0.1227, {55.7586517, 2.304111, 0.490799993, 0.207360908, 0.135075569, 0},
 {0.41252324, 13.1800241, 295.146637, 3815.27075, 28041.043, 1.32522774, 6511.53564, 69118.1094, 111560.75, 161005.781, 0}},
{10, 0.1227, {55.7586517, 2.304111, 0.490799993, 0.207360908, 0.135075569, 0},
 {0.526673079, 20.1478863, 504.201324, 7155.66553, 58536.1875, 1.81927359, 12502.0205, 159002.281, 301011.219, 377583.531, 0}},
{10, 0.0496, {22.5397644, 0.931409121, 0.198400006, 0.08382315189, 0.05460267514, 0},
 {0.01171184145, 1.18566418, 81.8601913, 3081.73657, 56009.543, 2.33227348, 65372.1094, 245734.016, 7527104.5, 1628419.25, 0}},
{10, 0.0496, {22.5397644, 0.931409121, 0.198400006, 0.08382315189, 0.05460267514, 0},
 {0.009003613144, 1.26440704, 104.971863, 4293.61768, 81361.2031, 2.58736515, 95179.1016, 385469.375, 10990779, 2475894.75, 0}},

// Z=54,  14 orbitals
{11, 34.5614, {736.758545, 149.768585, 69.1228027, 44.9295921, 36.2624741, 34.5959625},
 {624.902771, 0, 0, 0, 0, 1.16669405, 105.467674, 937.121216, 2996.29761, 5237.44922, 5917.15186}},
{11, 5.4528, {116.239418, 23.6291981, 10.9055996, 7.08860397, 5.72118092, 5.45825291},
 {58.7911682, 933.676514, 9777.10059, 0, 0, 21.3622379, 1240.57385, 6577.08008, 14887.8174, 21553.1895, 23289.6211}},
{11, 5.1037, {108.797523, 22.1164055, 10.2074003, 6.63477659, 5.35489893, 5.10880375},
 {11.0863647, 400.32373, 10981.6182, 0, 0, 3.95615411, 724.057129, 7425.8584, 24750.0664, 43523.082, 50915.9023}},
{11, 4.7822, {101.943985, 20.7232151, 9.56439972, 6.21682882, 5.01757479, 4.78698206},
 {13.9128065, 602.490906, 18586.5664, 0, 0, 5.82589436, 1374.63269, 15035.8057, 51837.3047, 92885.4141, 109281.883}},
{11, 1.1446, {24.3998737, 4.96001673, 2.28920007, 1.4879725, 1.20093596, 1.14574456},
 {11.5422506, 182.560226, 2069.69287, 16305.2402, 0, 225.554062, 6558.44336, 25666.5449, 49240.8281, 64829.4492, 69698.6953}},
{10, 0.999, {453.976318, 18.7596321, 3.99600005, 1.68829298, 1.09975958, 0},
 {2.26599121, 73.7758789, 1676.35754, 20961.6934, 125156.141, 0.006453159265, 211.35791, 11509.1846, 58277.1328, 100207.68, 0}},
{10, 0.937, {425.801605, 17.5953712, 3.74799991, 1.58351398, 1.03150618, 0},
 {2.88628817, 112.821999, 2876.83911, 40080.6445, 254936.969, 0.008000156842, 408.548584, 24666.2402, 134316.453, 246848.984, 0}},
{10, 0.6854, {311.466827, 12.8707218, 2.74160004, 1.15831435, 0.754529715, 0},
 {0.07581371069, 8.03451633, 593.645935, 25286.0742, 554710.313, 0.0001593173656, 146.911163, 33567.6992, 388215.625, 1148923, 0}},
{10, 0.6723, {305.513794, 12.6247253, 2.68919992, 1.13617551, 0.74010843, 0},
 {0.05801723897, 8.58840179, 763.666443, 35315.5352, 810876.438, 0.0001457292674, 194.779999, 50045.7188, 592061.188, 1774536.25, 0}},
{10, 0.2081, {94.5670395, 3.90778708, 0.832399964, 0.351685435, 0.229089037, 0},
 {2.58266187, 40.2186584, 458.666656, 3879.22339, 26078.457, 1.62305725, 2361.20264, 34653.8359, 104898.953, 136050.734, 0}},
{10, 0.1467, {66.6649857, 2.75479269, 0.586799979, 0.247920483, 0.161496207, 0},
 {0.475486934, 14.9392967, 327.582428, 4144.67236, 29800.2109, 0.845944464, 4936.9248, 59695.6836, 104645.664, 133242.016, 0}},
{10, 0.1467, {66.6649857, 2.75479269, 0.586799979, 0.247920483, 0.161496207, 0},
 {0.60380882, 22.7610054, 559.28772, 7799.18701, 62819.4063, 1.11512387, 9363.07715, 135641.188, 282984.938, 336481.969, 0}},
{10, 0.064, {29.0835686, 1.20181823, 0.256000012, 0.108158909, 0.07045506686, 0},
 {0.01452930365, 1.44994819, 97.9827118, 3587.86743, 63078.9258, 1.02980137, 41673.0234, 314019.625, 2160809.75, 2001500.5, 0}},
{10, 0.064, {29.0835686, 1.20181823, 0.256000012, 0.108158909, 0.07045506686, 0},
 {0.01111309417, 1.54286444, 125.607826, 5001.27832, 91777.6953, 1.07193601, 60255.125, 485718.219, 3088150, 2997721, 0}},

// Z=55,  17 orbitals
{11, 35.9846, {767.097412, 155.935883, 71.9692001, 46.7797432, 37.755722, 36.0205841},
 {668.320923, 0, 0, 0, 0, 1.13831234, 101.049385, 895.57666, 2862.36206, 5007.01465, 5643.01758}},
{11, 5.7143, {121.813911, 24.7623844, 11.4286003, 7.42855263, 5.99555159, 5.7200141},
 {62.9148407, 985.461731, 10133.5908, 0, 0, 20.2039261, 1173.67969, 6234.69531, 14131.9902, 20480.4668, 22111.2813}},
{11, 5.3594, {114.248375, 23.2244568, 10.7187996, 6.96718454, 5.62318373, 5.36475945},
 {12.4088593, 441.206024, 11877.7227, 0, 0, 3.77663493, 683.332703, 6998.48926, 23326.752, 41032.0469, 48133.7227}},
{11, 5.0119, {106.840584, 21.7185993, 10.0237999, 6.51543713, 5.25858068, 5.01691198},
 {15.4399595, 660.235291, 20066.7324, 0, 0, 5.4762845, 1291.51416, 14149.7021, 48858.8516, 87793.4297, 103287.031}},
{11, 1.2171, {25.945385, 5.274189, 2.43420005, 1.58222198, 1.27700436, 1.21831715},
 {12.4345665, 194.303421, 2176.17163, 16932.834, 0, 207.813293, 6103.1875, 24027.209, 46306.1641, 61175.3398, 66349.1484}},
{11, 1.065, {22.7030106, 4.6150775, 2.12999988, 1.38449299, 1.11741817, 1.06606495},
 {2.55166268, 81.5791855, 1811.97534, 22041.3555, 0, 132.541412, 8887.50684, 41439.9219, 78768.7188, 99256.2813, 116908.328}},
{10, 0.9976, {453.340118, 18.7333412, 3.99040008, 1.68592691, 1.09821832, 0},
 {3.22512913, 124.16246, 3105.3457, 42307.8672, 286714.625, 0.007030848414, 369.943176, 22729.1934, 125855.672, 234657.953, 0}},
{10, 0.7395, {336.051544, 13.8866339, 2.95799994, 1.24974239, 0.814086258, 0},
 {0.08929523826, 9.31651497, 674.936401, 28080.4043, 598648.063, 0.0001352251275, 125.515762, 29354.1777, 347528.063, 1013736.5, 0}},
{10, 0.7255, {329.689514, 13.6237364, 2.90199995, 1.22608256, 0.798674226, 0},
 {0.06807342172, 9.936903, 867.317322, 39209.3945, 875926.188, 0.0001313709508, 164.26033, 43590.1602, 529143.375, 1566129.13, 0}},
{10, 0.2308, {104.882614, 4.33405685, 0.923200011, 0.390048057, 0.254078567, 0},
 {2.82313156, 43.5039444, 491.15097, 4124.45654, 27439.4531, 1.35660052, 2063.27148, 31075.7227, 96953.8828, 136826, 0}},
{10, 0.1723, {78.2984161, 3.23551989, 0.689199984, 0.291184038, 0.189678237, 0},
 {0.545892358, 16.8631973, 362.008423, 4482.41504, 31513.4902, 0.584037483, 3832.93018, 51790.2969, 99812.5313, 119684.563, 0}},
{10, 0.1616, {73.4360123, 3.0345912, 0.646400034, 0.27310124, 0.177899033, 0},
 {0.688148081, 25.5704937, 616.470398, 8429.50781, 66677.3281, 0.917055428, 8224.60254, 126086.664, 282213.5, 341443.25, 0}},
{10, 0.0788, {35.8091431, 1.47973871, 0.315200001, 0.133170649, 0.08674779534, 0},
 {0.01783043146, 1.75544798, 116.101906, 4130.71094, 70102.7969, 0.542041779, 28382.6523, 333746.094, 509263.906, 33172810, 0}},
{10, 0.0765, {34.7639503, 1.43654835, 0.305999994, 0.129283696, 0.08421581984, 0},
 {0.01359604299, 1.86466908, 148.681335, 5751.55957, 101874.367, 0.606637239, 43811.2578, 516660.094, 794425, 40387076, 0}},
{10, 0.0227, {10.3155775, 0.426269919, 0.09080000222, 0.03836261109, 0.02498953044, 0},
 {0.472456068, 6.95320225, 76.9159241, 654.311829, 4590.27783, 56.5033722, 17553.1426, 113058.57, 88327.1797, 39314.4805, 0}},
{10, 0.0131, {5.95304298, 0.245997176, 0.05240000039, 0.02213877626, 0.01442127116, 0},
 {0.0731548816, 2.10896921, 43.6368027, 542.00116, 4002.73047, 117.951195, 22826.1758, 299502.938, 6245063.5, 34782560, 0}},
{10, 0.0114, {5.18051052, 0.214073882, 0.04560000077, 0.0192658063, 0.01254980825, 0},
 {0.08944954723, 3.0856297, 71.4837799, 977.464233, 8012.70508, 281.478271, 62977.6602, 667453.938, 14187204, 75207264, 0}},

// Z=56,  17 orbitals
{11, 37.4406, {798.135559, 162.245331, 74.8811951, 48.6725349, 39.2833862, 37.4780388},
 {713.582458, 0, 0, 0, 0, 1.11179888, 96.9028397, 856.469177, 2735.89893, 4783.97607, 5391.82031}},
{11, 5.9888, {127.665535, 25.9519043, 11.9776001, 7.78540087, 6.28356218, 5.99478865},
 {67.2285004, 1038.76208, 10493.1992, 0, 0, 19.0881329, 1109.38501, 5906.62012, 13409.1484, 19454.8555, 20960.9531}},
{11, 5.6236, {119.880424, 24.3693428, 11.2472, 7.3106432, 5.90038776, 5.62922382},
 {13.8606262, 485.220032, 12817.0371, 0, 0, 3.61123204, 645.642761, 6601.7207, 22001.8965, 38780.4063, 45369.6211}},
{11, 5.247, {111.852303, 22.7373829, 10.4940004, 6.82106543, 5.50525188, 5.25224686},
 {17.0906467, 721.81012, 21613.8535, 0, 0, 5.15737438, 1215.44312, 13334.3994, 46105.3984, 82941.5078, 97501.9453}},
{11, 1.2928, {27.5591106, 5.60222769, 2.5855999, 1.68063152, 1.35643029, 1.29409277},
 {13.3712473, 206.45752, 2284.67334, 17555.6973, 0, 191.704315, 5684.00342, 22503.7188, 43555.918, 57755.9063, 62266.043}},
{11, 1.1367, {24.2314682, 4.92578316, 2.27340007, 1.4777025, 1.19264722, 1.13783669},
 {2.86609626, 89.9967422, 1954.58813, 23138.7617, 0, 120.214737, 8159.1001, 38513.3047, 74012.3203, 93928.9297, 108998.781}},
{11, 1.0621, {22.643322, 4.60294437, 2.1243999, 1.38085306, 1.11448038, 1.06326222},
 {3.5944438, 136.31311, 3344.86377, 44585.9531, 0, 227.145859, 17394.1035, 87520.2578, 177928.75, 235298.797, 271842.844}},
{10, 0.7961, {361.772308, 14.9494925, 3.18440008, 1.34539545, 0.876394928, 0},
 {0.104761206, 10.7616501, 764.411133, 31069.1133, 644373.125, 0.0001156260405, 107.81546, 25759.6895, 311501.281, 900985.125, 0}},
{10, 0.7807, {354.774078, 14.660305, 3.12279987, 1.31936967, 0.859441698, 0},
 {0.07956307381, 11.4523907, 981.137512, 43359.9141, 943255.75, 0.000120954981, 139.529465, 38155.8359, 473915.219, 1389455.5, 0}},
{10, 0.253, {114.970978, 4.75093746, 1.01199996, 0.427565664, 0.278517663, 0},
 {3.08263111, 46.9716301, 524.796814, 4374.58398, 28774.8438, 1.17016554, 1838.27747, 28246.3711, 90572.1094, 129712.18, 0}},
{10, 0.1918, {87.1598129, 3.60169911, 0.767199993, 0.324138731, 0.211145014, 0},
 {0.625213265, 18.9666653, 398.201202, 4820.63867, 33107.3359, 0.47875005, 3297.2666, 47314.3828, 97749.9844, 109830.711, 0}},
{10, 0.1797, {81.6612015, 3.37448025, 0.718800008, 0.303689927, 0.197824612, 0},
 {0.781740725, 28.6364632, 677.618713, 9093.2832, 70677.25, 0.729823232, 7054.22217, 115113.43, 276223.219, 328282.125, 0}},
{10, 0.0925, {42.0348434, 1.73700297, 0.370000005, 0.156323418, 0.101829588, 0},
 {0.0217344258, 2.10992384, 136.500305, 4715.90088, 77176.7109, 0.33337456, 21233.9199, 328205.063, 191381.984, 5116668, 0}},
{10, 0.0899, {40.8533249, 1.68817914, 0.359600008, 0.151929468, 0.09896735102, 0},
 {0.01655145921, 2.23914099, 174.731979, 6567.91357, 112285.172, 0.355158508, 32598.8613, 509330.25, 286083.969, 8133637, 0}},
{10, 0.03901, {17.7682419, 0.734235823, 0.156399995, 0.06607833505, 0.04304363951, 0},
 {0.55103147, 8.08315945, 88.9186935, 753.340576, 5275.69482, 20.1813545, 8736.69629, 76697.9141, 119734.805, 46879.7344, 0}},
{10, 0.01656, {7.54355049, 0.311721623, 0.06639999896, 0.02805371769, 0.01827428304, 0},
 {0.09369242191, 2.68614078, 54.6888275, 664.928711, 4825.64844, 82.7137222, 21627.6367, 176767.813, 2802744.5, 14046967, 0}},
{10, 0.0146, {6.63468933, 0.274164796, 0.05840000138, 0.02467375249, 0.01607256196, 0},
 {0.115231708, 3.96815205, 90.7736359, 1219.17761, 9861.18945, 193.207626, 59549.1172, 412173.219, 5905087.5, 30859814, 0}},

// Z=57,  17 orbitals
{11, 38.9246, {829.770569, 168.676102, 77.8491974, 50.6017227, 40.8404274, 38.9635239},
 {760.622742, 0, 0, 0, 0, 1.08746529, 93.0395203, 819.900757, 2617.47217, 4574.8501, 5157.28516}},
{11, 6.2663, {133.5811, 27.1544247, 12.5326004, 8.14614868, 6.57472086, 6.27256632},
 {71.7344208, 1093.24158, 10849.6484, 0, 0, 18.1027164, 1051.47766, 5608.29297, 12748.25, 18545.5527, 19904.3516}},
{11, 5.8906, {125.572166, 25.5263634, 11.7812004, 7.65774155, 6.18052912, 5.89649057},
 {15.4446859, 532.367676, 13792.8945, 0, 0, 3.46828914, 612.457642, 6248.75586, 20814.959, 36688.3828, 41857.4805}},
{11, 5.4827, {116.876808, 23.7587662, 10.9653997, 7.12747383, 5.75255251, 5.48818254},
 {18.873806, 787.280396, 23222.7852, 0, 0, 4.88159323, 1148.77698, 12612.4453, 43647.6953, 78571.0156, 89963.6719}},
{11, 1.3613, {29.019352, 5.89906597, 2.72259998, 1.7696811, 1.42830181, 1.36266136},
 {14.357481, 218.96373, 2393.1001, 18141.1133, 0, 180.240738, 5369.8457, 21318.6699, 41354.8047, 55034.1289, 58846.9102}},
{11, 1.2044, {25.674654, 5.21915436, 2.40880013, 1.56571209, 1.26367927, 1.20560443},
 {3.21070743, 99.0047684, 2101.86963, 24202.1484, 0, 111.2351, 7595.96582, 36149.2617, 70033.2422, 89405.4063, 104330.344}},
{11, 1.1234, {23.9479465, 4.8681488, 2.24679995, 1.46041262, 1.17869258, 1.1245234},
 {3.99530911, 149.249466, 3592.67651, 46846.25, 0, 208.944336, 16194.5527, 82332.9766, 168956.438, 224955.078, 259557.609}},
{10, 0.8485, {385.584503, 15.9334812, 3.39400005, 1.43395054, 0.934080064, 0},
 {0.122428671, 12.3795862, 861.627136, 34171.8516, 689897.938, 0.0001033526787, 95.9333115, 23207.3984, 284373.25, 819914.25, 0}},
{10, 0.8317, {377.950043, 15.6180038, 3.32679987, 1.40555882, 0.915585577, 0},
 {0.09262146056, 13.1446447, 1104.64941, 47668.8398, 1009631, 0.0001158792584, 122.895706, 34300.375, 432385.813, 1263590.63, 0}},
{10, 0.2704, {122.878075, 5.07768202, 1.08159995, 0.456971377, 0.297672659, 0},
 {3.35369444, 50.5715675, 559.346008, 4626.12598, 30044.0469, 1.0760715, 1714.85522, 26598.2578, 86576.2031, 125810.188, 0}},
{10, 0.2058, {93.5218506, 3.86459684, 0.823199987, 0.347798496, 0.226557061, 0},
 {0.712526858, 21.2413349, 436.218292, 5161.81543, 34613.3047, 0.422868967, 3052.04541, 45007.6914, 97081.125, 109125.227, 0}},
{10, 0.1914, {86.9780426, 3.59418774, 0.765600026, 0.323462725, 0.210704684, 0},
 {0.885864735, 31.9587975, 741.741638, 9764.77148, 74519.5469, 0.670626879, 6612.56006, 110712.016, 276993.594, 337661.438, 0}},
{10, 0.0989, {44.9431992, 1.85718477, 0.395599991, 0.167139307, 0.108875088, 0},
 {0.02634496614, 2.51742959, 159.058853, 5327.98145, 83920.8281, 0.305604696, 19869.7539, 330691.094, 173071.016, 1769126.13, 0}},
{10, 0.0989, {44.9431992, 1.85718477, 0.395599991, 0.167139307, 0.108875088, 0},
 {0.01999914832, 2.66824508, 203.687515, 7435.59229, 122600.813, 0.280723274, 28343.6602, 502299.375, 256709.563, 2599748, 0}},
{10, 0.0323, {14.678113, 0.606542647, 0.129199997, 0.05458644778, 0.03555779159, 0},
 {0.634120166, 9.22574329, 100.573242, 845.609009, 5863.55518, 34.9373627, 13069.1367, 102413.617, 145775.375, 82583.2813, 0}},
{10, 0.0144, {6.54380274, 0.270409107, 0.05759999901, 0.02433575504, 0.01585238986, 0},
 {0.115328141, 3.26670742, 65.2264099, 776.051514, 5532.49072, 138.578018, 28625.2891, 268284.125, 3371859, 11701498, 0}},
{10, 0.0144, {6.54380274, 0.270409107, 0.05759999901, 0.02433575504, 0.01585238986, 0},
 {0.141208023, 4.82152081, 108.56073, 1432.34155, 11431.7119, 237.852753, 70301.5703, 473999.531, 5534420, 21534214, 0}},

// Z=58,  18 orbitals
{11, 40.443, {862.138855, 175.255951, 80.8860016, 52.575634, 42.4335632, 40.4834442},
 {809.808044, 0, 0, 0, 0, 1.06494546, 89.4196777, 785.679443, 2507.05469, 4384.17139, 4934.34863}},
{11, 6.5488, {139.603271, 28.3786106, 13.0976, 8.51339722, 6.87112522, 6.55534887},
 {76.451767, 1149.86023, 11217.5518, 0, 0, 17.2115879, 999.253052, 5340.44824, 12156.1689, 17694.2715, 18956.2617}},
{11, 6.1642, {131.404602, 26.7119827, 12.3283997, 8.01341915, 6.46759558, 6.17036438},
 {17.1871319, 583.77478, 14862.6494, 0, 0, 3.34046555, 582.965149, 5939.49902, 19788.4238, 34880.5117, 39648.7305}},
{11, 5.7234, {122.007904, 24.8018188, 11.4468002, 7.44038248, 6.00509977, 5.72912359},
 {20.8196659, 858.343506, 24986.5137, 0, 0, 4.63358974, 1089.41125, 11981.6689, 41540.082, 74921.2031, 87887.2813}},
{11, 1.4346, {30.5819168, 6.21670485, 2.86919999, 1.86497056, 1.50520957, 1.43603468},
 {15.399477, 232.209656, 2510.77319, 18812.0273, 0, 169.179718, 5079.00439, 20256.5059, 39389.9805, 52409.5625, 56607.7656}},
{11, 1.2728, {27.1327629, 5.51555967, 2.54559994, 1.65463161, 1.33544588, 1.27407277},
 {3.59319282, 108.887032, 2263.3147, 25395.0762, 0, 103.819031, 7137.40576, 34230.3945, 66648.0156, 85321.3125, 100858.266}},
{11, 1.1854, {25.2696247, 5.13681984, 2.37080002, 1.54101217, 1.24374413, 1.18658543},
 {4.43625975, 163.381516, 3865.47168, 49440.1797, 0, 193.663513, 15214.8311, 78193.3125, 161653.813, 216238.609, 250859.453}},
{11, 0.9013, {409.57843, 16.9249802, 3.60520005, 1.52318156, 0.992205441, 0},
 {0.142987847, 14.2490416, 974.597046, 37971.5156, 766256.375, 9.374778892e-05, 86.6941147, 21336.9805, 266197.406, 783598.063, 0}},
{10, 0.8833, {401.398682, 16.5869694, 3.53320003, 1.49276197, 0.972389996, 0},
 {0.107752517, 15.0949926, 1248.11316, 52964.6797, 1117025.88, 0.0001129675657, 109.798767, 31450.6113, 404560.438, 1208476.25, 0}},
{10, 0.2896, {131.603149, 5.43822765, 1.15839994, 0.489419043, 0.318809152, 0},
 {3.61078858, 53.8377266, 589.91925, 4852.89404, 31144.9297, 0.976827741, 1578.11487, 24810.5371, 80955.25, 117010.914, 0}},
{10, 0.2233, {101.47438, 4.19321871, 0.89319998, 0.377373189, 0.245822117, 0},
 {0.800607145, 23.4364147, 471.346466, 5469.02637, 35751.8672, 0.367917597, 2727.5188, 41595.8828, 90040.4922, 99440.3594, 0}},
{10, 0.2072, {94.1580505, 3.89088655, 0.828800023, 0.350164473, 0.228098273, 0},
 {0.984948337, 35.0290184, 799.215332, 10361.3623, 77732.5938, 0.551236331, 5903.72998, 102664.188, 259700.688, 312581.281, 0}},
{10, 0.101, {49.987381, 2.06562519, 0.439999998, 0.185898125, 0.121094644, 0},
 {0.03089012951, 2.9030962, 179.588837, 5875.5127, 89656, 0.22479026, 16450.4414, 302828.688, 155294.266, 2181362.25, 0}},
{10, 0.101, {49.987381, 2.06562519, 0.439999998, 0.185898125, 0.121094644, 0},
 {0.02329728007, 3.06572342, 229.529587, 8191.41748, 131030.469, 0.198385447, 23346.7324, 458748.375, 230269.875, 3252526.25, 0}},
{10, 0.0859, {39.0356026, 1.61306536, 0.343600005, 0.145169526, 0.09456390887, 0},
 {1.035557761e-05, 0.003610637737, 0.808584511, 99.477417, 7211.28125, 0.0005046080914, 1199.9967, 226161.875, 939981.813, 176164.344, 0}},
{10, 0.0378, {17.1774826, 0.709823906, 0.151199996, 0.06388135254, 0.04161252454, 0},
 {0.651443303, 9.37103844, 101.202232, 846.455566, 5818.99902, 25.0341949, 10096.7607, 82769.1406, 119174.609, 54989.6055, 0}},
{10, 0.0198, {8.99772835, 0.371812522, 0.07919999957, 0.03346166015, 0.02179703489, 0},
 {0.121410131, 3.36805582, 65.7644196, 767.27594, 5356.12988, 63.993927, 19259.2656, 134978.344, 2124584.25, 10484848, 0}},
{10, 0.0198, {8.99772835, 0.371812522, 0.07919999957, 0.03346166015, 0.02179703489, 0},
 {0.146372274, 4.9095149, 108.452263, 1408.65442, 11072.5674, 105.411148, 45629.668, 262444.625, 3349819.25, 19155670, 0}},

// Z=59,  18 orbitals
{11, 41.991, {895.129639, 181.962326, 83.9812012, 54.5875053, 44.0573349, 42.0325928},
 {860.712097, 0, 0, 0, 0, 1.04404485, 86.0257263, 753.414917, 2402.43408, 4199.05176, 4728.57568}},
{11, 6.8348, {145.700043, 29.6179657, 13.6696005, 8.88519478, 7.17120171, 6.84163475},
 {81.3395538, 1207.10864, 11565.8018, 0, 0, 16.3109131, 951.270203, 5091.45117, 11600.6182, 16877.1348, 18157.6543}},
{11, 6.4404, {137.292465, 27.9088688, 12.8808002, 8.37247753, 6.75738955, 6.44684029},
 {19.0795765, 638.302734, 15951.4932, 0, 0, 3.22954988, 556.463745, 5655.64258, 18830.7266, 33142.2188, 37849.7344}},
{11, 5.9643, {127.143257, 25.8457355, 11.9286003, 7.75355101, 6.25785637, 5.97026443},
 {22.903677, 933.039917, 26775.6172, 0, 0, 4.41599846, 1036.26831, 11404.3916, 39572.0508, 71370.6797, 83811.2344}},
{11, 1.511, {32.2105637, 6.5477767, 3.02199984, 1.96429002, 1.58536983, 1.51251101},
 {16.4826298, 245.755096, 2628.28271, 19448.873, 0, 158.747223, 4798.3833, 19215.6523, 37455.7188, 49881.4883, 54007.9102}},
{11, 1.3374, {28.5098667, 5.79549742, 2.67479992, 1.73861122, 1.40322542, 1.33873737},
 {4.00849056, 119.319763, 2425.90112, 26482.9023, 0, 98.3049698, 6764.58691, 32577.4707, 63681.0547, 81833.2656, 95781.7969}},
{11, 1.2422, {26.4804516, 5.38295746, 2.48440003, 1.61485183, 1.30333972, 1.24344218},
 {4.9123044, 178.217194, 4139.19287, 51840.7891, 0, 182.810226, 14453.498, 74755.5313, 155398.234, 208608.172, 243150.438}},
{10, 0.9511, {432.209076, 17.8601456, 3.80439997, 1.60734272, 1.0470283, 0},
 {0.166180491, 16.3103027, 1094.47742, 41736.0703, 853500.313, 8.752152644e-05, 79.9549789, 19838.3828, 249923.359, 741483.375, 0}},
{10, 0.931, {423.075012, 17.4827003, 3.72399998, 1.57337415, 1.02490103, 0},
 {0.124732487, 17.2376976, 1399.79138, 58175.6797, 1234410.5, 0.0001128136355, 100.572495, 29242.8535, 380224.063, 1145307.25, 0}},
{10, 0.3045, {138.374161, 5.71802616, 1.21800005, 0.5145998, 0.335211992, 0},
 {3.88804603, 57.3575478, 622.517395, 5085.46436, 32177.2168, 0.930876434, 1505.27185, 23761.2402, 77547.5938, 112279.977, 0}},
{10, 0.2363, {107.381989, 4.43733835, 0.945200026, 0.399342954, 0.260133296, 0},
 {0.899813354, 25.8756599, 509.420319, 5787.86377, 36860.5078, 0.347547859, 2571.41772, 39730.1016, 86690.9844, 95286.7266, 0}},
{10, 0.2176, {98.8841324, 4.08618212, 0.870400012, 0.367740303, 0.239547223, 0},
 {1.09892488, 38.4812775, 862.006897, 10987.7666, 80905.5938, 0.521490991, 5636.45605, 99302.6797, 253925.141, 306984.688, 0}},
{10, 0.1132, {51.4415588, 2.12571597, 0.452800006, 0.191306069, 0.124617398, 0},
 {0.03637861088, 3.36317515, 203.367188, 6471.13867, 95183.6875, 0.234063476, 16634.0469, 300942, 155594.25, 1859522.25, 0}},
{10, 0.1132, {51.4415588, 2.12571597, 0.452800006, 0.191306069, 0.124617398, 0},
 {0.0273201149, 3.54275942, 259.574341, 9017.19824, 139186.813, 0.203849971, 23572.1543, 456056.719, 231231.016, 2776127.5, 0}},
{10, 0.0035, {1.59050763, 0.06572443992, 0.01400000043, 0.005914940499, 0.003853011411, 0},
 {2.183549441e-05, 0.007402888034, 1.58398998, 180.476883, 11228.9365, 2134.21289, 3177413, 1800969.5, 1281234.88, 1149309.13, 0}},
{10, 0.0374, {16.9957104, 0.702312529, 0.149599999, 0.06320536137, 0.0411721766, 0},
 {0.698364556, 9.93235111, 106.226723, 882.846558, 6008.07275, 27.0433693, 10559.6523, 83490.3438, 114421.273, 52936.0195, 0}},
{10, 0.0223, {10.1338062, 0.418758541, 0.08919999748, 0.03768661991, 0.02454918623, 0},
 {0.134908944, 3.68196821, 70.4402924, 805.636169, 5511.17969, 50.8948174, 17093, 110354.102, 1795618.25, 9865037, 0}},
{10, 0.0223, {10.1338062, 0.418758541, 0.08919999748, 0.03768661991, 0.02454918623, 0},
 {0.160756379, 5.32210493, 115.589355, 1477.56909, 11442.8271, 82.3358917, 40193.3594, 223769.688, 2781508, 17979608, 0}},

// Z=60,  18 orbitals
{11, 43.5689, {928.774841, 188.801743, 87.1378021, 56.6392822, 45.7133141, 43.6124687},
 {913.447388, 0, 0, 0, 0, 1.02490735, 82.8395767, 723.043884, 2304.0708, 4028.49536, 4529.63818}},
{11, 7.126, {151.907654, 30.8798542, 14.2519999, 9.26375294, 7.47673416, 7.13312626},
 {86.4296341, 1265.45947, 11914.8076, 0, 0, 15.5730972, 906.95459, 4859.86523, 11082.6943, 16151.4941, 17348.416}},
{11, 6.7215, {143.284775, 29.1269913, 13.4429998, 8.7379055, 7.05232525, 6.72822142},
 {21.1415882, 696.520569, 17076.5527, 0, 0, 3.13180876, 532.388794, 5395.26465, 17946.8633, 31614.752, 36935.4258}},
{11, 6.2079, {132.336166, 26.9013538, 12.4158001, 8.07022953, 6.51344633, 6.21410799},
 {25.143713, 1012.18188, 28640.3652, 0, 0, 4.22006083, 987.99054, 10876.1992, 37762.1992, 68143.7031, 79710.7188}},
{11, 1.5753, {33.5812721, 6.82641506, 3.15060019, 2.0478797, 1.65283465, 1.57687533},
 {17.6077061, 259.519012, 2743.41016, 20013.6563, 0, 152.314575, 4609.48486, 18466.1133, 35998.0273, 47935.957, 51975.082}},
{11, 1.4028, {29.9040222, 6.07890224, 2.80559993, 1.82363081, 1.47184432, 1.40420282},
 {4.46138382, 130.456482, 2594.31519, 27548.584, 0, 93.4612045, 6428.92871, 31062.5059, 60916.6875, 78521.5781, 91443.4609}},
{11, 1.2974, {27.6571712, 5.62216139, 2.5948, 1.68661153, 1.36125672, 1.29869735},
 {5.4233737, 193.898621, 4421.40381, 54216.1445, 0, 174.003799, 13811.7891, 71773.0547, 149865.641, 201881.109, 238359.641}},
{10, 0.9995, {454.203522, 18.7690201, 3.99799991, 1.68913794, 1.10030997, 0},
 {0.192459345, 18.6036015, 1224.46497, 45675.3984, 10258765, 7.803834887e-05, 74.7325974, 18626.1699, 236190.359, 705053.5, 0}},
{10, 0.9777, {444.296936, 18.3596516, 3.91079998, 1.6522963, 1.07631123, 0},
 {0.143891469, 19.6166897, 1564.23059, 63645.9609, 1330188.38, 0.0001077217312, 93.2128067, 27422.7539, 359431.469, 1089675.88, 0}},
{10, 0.3152, {143.236572, 5.91895485, 1.2608, 0.532682598, 0.346991181, 0},
 {4.17606592, 60.9531631, 655.201355, 5311.15576, 33073.5, 0.876959801, 1476.41321, 23229.2422, 75376.2891, 109102.828, 0}},
{10, 0.2433, {110.562996, 4.5687871, 0.973199964, 0.411172837, 0.267839313, 0},
 {1.00769746, 28.4614315, 548.302368, 6095.6709, 37794.9883, 0.357061416, 2562.55591, 39166.2227, 84976.8047, 93437.0938, 0}},
{10, 0.2246, {102.065147, 4.21763086, 0.898400009, 0.379570156, 0.247253239, 0},
 {1.22021139, 42.1038475, 926.316162, 11606.6885, 83823.6797, 0.521504462, 5574.42334, 98074.5938, 251388.953, 305621.188, 0}},
{10, 0.1175, {53.3956108, 2.2064631, 0.469999999, 0.198572993, 0.129351094, 0},
 {0.04261606559, 3.87616181, 229.196854, 7099.17236, 100678.688, 0.234244794, 16402.457, 296047.875, 154744.047, 1601790.63, 0}},
{10, 0.1175, {53.3956108, 2.2064631, 0.469999999, 0.198572993, 0.129351094, 0},
 {0.03191594034, 4.07501125, 292.152466, 9887.37012, 147312.656, 0.200786948, 23198.957, 448626.906, 230925.766, 2393268.75, 0}},
{10, 0.003, {1.36329222, 0.05633522943, 0.0120000001, 0.005069948733, 0.003302581143, 0},
 {3.790792107e-05, 0.01291131414, 2.73084378, 303.519989, 18331.541, 6201.41406, 4246958.5, 1898921.63, 1260278, 900741.375, 0}},
{10, 0.0375, {17.041153, 0.704190373, 0.149999991, 0.06337435544, 0.04128226265, 0},
 {0.744405925, 10.4884844, 111.216232, 918.484375, 6185.98535, 28.2904549, 10784.8213, 82883.8125, 109374.461, 51805.9492, 0}},
{10, 0.0211, {9.58848858, 0.396224469, 0.08440000564, 0.03565864265, 0.02322815545, 0},
 {0.149897382, 4.01506662, 75.164299, 842.401367, 5641.87305, 62.5228653, 18274.9824, 129760.617, 2142587.5, 11493880, 0}},
{10, 0.0211, {9.58848858, 0.396224469, 0.08440000564, 0.03565864265, 0.02322815545, 0},
 {0.176585436, 5.75601721, 122.752457, 1543.74365, 11766.4434, 101.254219, 43792.4063, 248609.359, 3320073.25, 21039738, 0}},

// Z=61,  18 orbitals
{11, 45.184, {963.204529, 195.800629, 90.3680038, 58.738903, 47.4079094, 45.2291832},
 {968.105164, 0, 0, 0, 0, 1.00672913, 79.8226166, 694.223694, 2210.44873, 3862.62012, 4345.92725}},
{11, 7.4279, {158.343384, 32.1881104, 14.8557997, 9.65622139, 7.79349327, 7.43532801},
 {91.707428, 1325.11218, 12263.998, 0, 0, 14.8659601, 864.576538, 4639.04932, 10588.75, 15432.6221, 16601.2402}},
{11, 7.0128, {149.494537, 30.3893108, 14.0255995, 9.11659431, 7.35796261, 7.01981258},
 {23.3821125, 758.867065, 18264.502, 0, 0, 3.01800466, 509.440186, 5148.7085, 17113.2383, 30120.582, 35252.332}},
{11, 6.4593, {137.695358, 27.9907703, 12.9186001, 8.397048, 6.77721977, 6.46575928},
 {27.5586643, 1096.41223, 30605.2305, 0, 0, 4.0335598, 942.017212, 10375.3223, 36054.043, 65171.3828, 76039.9375}},
{11, 1.6464, {35.0990677, 7.1349535, 3.29299998, 2.14043927, 1.72753906, 1.64814651},
 {18.7907276, 273.776337, 2861.63306, 20585.3867, 0, 145.209839, 4405.80762, 17676.3828, 34478.1602, 45973.0469, 49837.1992}},
{11, 1.4714, {31.366396, 6.37617397, 2.94280005, 1.91281044, 1.54382074, 1.47287142},
 {4.95716, 142.364609, 2769.53345, 28606.2969, 0, 88.7883835, 6105.67725, 29603.0605, 58231.0273, 75172.6953, 87920.0313}},
{11, 1.3569, {28.9255543, 5.87999916, 2.71379995, 1.76396108, 1.42368519, 1.35825694},
 {5.97502613, 210.568954, 4716.79346, 56664.7109, 0, 164.718735, 13151.1162, 68740.3672, 144250.172, 195117.422, 227489.813}},
{11, 1.0515, {22.415226, 4.55657673, 2.10299993, 1.36694312, 1.10325372, 1.05255151},
 {0.222328722, 21.1643238, 1367.03345, 49949.4805, 0, 42.286438, 13450.1514, 140667.063, 436547.063, 768294.688, 9587678}},
{11, 1.0269, {21.8908195, 4.44997501, 2.05380011, 1.3349632, 1.07744288, 1.02792692},
 {0.165465444, 22.2592735, 1743.82324, 69536.2188, 0, 51.1324348, 19737.7246, 213320.188, 671213.563, 1194337.88, 14725362}},
{10, 0.3304, {150.143921, 6.20438671, 1.32159996, 0.558370352, 0.363724262, 0},
 {4.48026371, 64.6739578, 688.634399, 5541.96533, 33967.9531, 0.835677087, 1411.66357, 22271.9531, 72218.1797, 104547.602, 0}},
{10, 0.2544, {115.607178, 4.7772274, 1.01760006, 0.429931641, 0.280058891, 0},
 {1.12526691, 31.221838, 588.890503, 6411.08105, 38681.0781, 0.348930359, 2468.01147, 37817, 82121.3359, 90096.3047, 0}},
{10, 0.236, {107.245651, 4.43170452, 0.944000006, 0.398835957, 0.259803057, 0},
 {1.35085428, 45.9449043, 993.455322, 12249.7432, 86818.5234, 0.489623427, 5293.72363, 94433.5938, 244521.234, 298490.156, 0}},
{10, 0.1204, {54.7134628, 2.26092052, 0.481599987, 0.20347394, 0.132543594, 0},
 {0.04967614636, 4.4450798, 257.014374, 7749.38623, 105896.57, 0.24603942, 16661.4395, 293728.563, 154141.938, 1425006, 0}},
{10, 0.1204, {54.7134628, 2.26092052, 0.481599987, 0.20347394, 0.132543594, 0},
 {0.03703936562, 4.6613493, 327.162567, 10787.4014, 155038.797, 0.208176479, 23531.9668, 445345.063, 229929.859, 2132981.75, 0}},
{10, 0.004, {1.81772292, 0.07511363924, 0.0159999989, 0.006759931799, 0.004403441679, 0},
 {6.042137829e-05, 0.02064520679, 4.31517363, 468.574341, 27527.2949, 3282.38892, 4659896.5, 2167964, 1106787.13, 625464.75, 0}},
{10, 0.0375, {17.041153, 0.704190373, 0.149999991, 0.06337435544, 0.04128226265, 0},
 {0.793453932, 11.0569677, 116.161903, 953.135803, 6350.88232, 29.700346, 11026.4053, 82262.7031, 104472.398, 51364.2109, 0}},
{10, 0.0211, {9.58848858, 0.396224469, 0.08440000564, 0.03565864265, 0.02322815545, 0},
 {0.165404201, 4.35852671, 79.9592667, 878.540588, 5760.62988, 66.6086273, 18324.8105, 136198.859, 2281655.5, 12365242, 0}},
{10, 0.0211, {9.58848858, 0.396224469, 0.08440000564, 0.03565864265, 0.02322815545, 0},
 {0.192575037, 6.19663668, 129.966965, 1609.01404, 12074.4307, 107.336311, 44408.6367, 253856.297, 3515567, 22672972, 0}},

// Z=62,  18 orbitals
{11, 46.8342, {998.382507, 202.951614, 93.6684036, 60.8841515, 49.1393318, 46.8810349},
 {1024.60889, 0, 0, 0, 0, 0.989990175, 76.9716949, 666.947815, 2122.06958, 3709.04028, 4167.85449}},
{11, 7.7368, {164.928314, 33.5266991, 15.4736004, 10.0577888, 8.11759758, 7.74453688},
 {97.1800232, 1385.82727, 12577.5488, 0, 0, 14.209692, 824.94397, 4431.94629, 10124.3301, 14750.3789, 15886.4209}},
{11, 7.3118, {155.868423, 31.6849995, 14.6236, 9.50529194, 7.67167902, 7.31911182},
 {25.8145218, 825.375854, 19507.0449, 0, 0, 2.92831445, 488.056732, 4918.67969, 16334.6289, 28715.1699, 33631.8125}},
{11, 6.7162, {143.171799, 29.104023, 13.4323997, 8.73101616, 7.04676437, 6.72291613},
 {30.1455135, 1185.63635, 32645.8516, 0, 0, 3.85827994, 899.062134, 9907.23438, 34452.4336, 62228.0156, 73187.8359}},
{11, 1.7228, {36.7255821, 7.46559238, 3.44560003, 2.23962855, 1.8075943, 1.72452271},
 {20.0156765, 288.433502, 2982.33032, 21155.1348, 0, 137.929657, 4200.3501, 16887.6016, 32973.082, 43966.6367, 47779.8008}},
{11, 1.5407, {32.843689, 6.67647886, 3.08139992, 2.00289989, 1.61653161, 1.54224062},
 {5.49368525, 155.004822, 2950.27979, 29631.3262, 0, 84.6665878, 5814.60059, 28265.7559, 55739.707, 72101.6797, 84933.0078}},
{11, 1.4198, {30.2664165, 6.15257025, 2.83959985, 1.84573066, 1.48968101, 1.42121983},
 {6.56906414, 228.257263, 5024.90869, 59169.4727, 0, 155.505859, 12499.7314, 65756.5703, 138696.875, 188337.016, 217892.078}},
{11, 1.106, {23.5770245, 4.7927475, 2.21199989, 1.43779278, 1.16043615, 1.10710597},
 {0.255926639, 24.0063839, 1522.65051, 54543.4414, 0, 39.2380524, 12562.8447, 132371.422, 413444.719, 727641.688, 8914730}},
{11, 1.0802, {23.0270348, 4.6809454, 2.16039991, 1.40425289, 1.13336635, 1.08128023},
 {0.189811066, 25.1965961, 1940.54822, 75954.4844, 0, 46.8156319, 18364.1543, 200432.625, 635388.875, 1130759.25, 13595857}},
{10, 0.3457, {157.09671, 6.49169636, 1.38279998, 0.584227085, 0.380567431, 0},
 {4.7918148, 68.4667816, 722.490662, 5771.56299, 34794.707, 0.798738182, 1352.46814, 21380.3438, 69235.1953, 100161.945, 0}},
{10, 0.2656, {120.696808, 4.98754597, 1.06239998, 0.448859483, 0.292388529, 0},
 {1.25433755, 34.1671867, 630.850708, 6726.2085, 39463.2578, 0.34250024, 2380.81567, 36525.8398, 79284.4063, 86797.0078, 0}},
{10, 0.2474, {112.42617, 4.64577866, 0.989600003, 0.418101788, 0.272352874, 0},
 {1.49108446, 49.9956284, 1062.76575, 12899.3506, 89694.4844, 0.462454557, 5043.36279, 91066.2344, 237872.828, 291451.938, 0}},
{10, 0.129, {58.6215668, 2.42241502, 0.515999973, 0.218007803, 0.142010987, 0},
 {0.05766981095, 5.07888269, 287.474701, 8456.4209, 111497.336, 0.214876041, 15080.7578, 279950.5, 151400.031, 1231856.88, 0}},
{10, 0.129, {58.6215668, 2.42241502, 0.515999973, 0.218007803, 0.142010987, 0},
 {0.04280338809, 5.3121953, 365.430267, 11766.2344, 163378.766, 0.176703438, 21222.5664, 423828.625, 228382.328, 1842798.25, 0}},
{10, 0.0055, {2.49936914, 0.103281252, 0.02199999988, 0.0092949057, 0.006054732017, 0},
 {9.115662397e-05, 0.03120273724, 6.44168377, 684.032349, 39105.0742, 1407.56836, 4632811, 2616724.75, 1052927.88, 517781.719, 0}},
{10, 0.0374, {16.9957104, 0.702312529, 0.149599999, 0.06320536137, 0.0411721766, 0},
 {0.841388226, 11.6187687, 121.064171, 986.825439, 6502.87549, 31.2840271, 11289.3701, 81614.6094, 99689.8125, 51566.5273, 0}},
{10, 0.0213, {9.67937469, 0.399980128, 0.08519999683, 0.03599663451, 0.02344832569, 0},
 {0.182548285, 4.72200823, 84.8162308, 913.745117, 5864.5293, 69.1399307, 18138.4902, 140176.719, 2382270.25, 13121841, 0}},
{10, 0.0213, {9.67937469, 0.399980128, 0.08519999683, 0.03599663451, 0.02344832569, 0},
 {0.210070804, 6.6577239, 137.215744, 1672.80249, 12359.4609, 110.741966, 44411.7617, 255105.328, 3646091.75, 24088756, 0}},

// Z=63,  18 orbitals
{11, 48.519, {1034.29797, 210.252533, 97.038002, 63.0743828, 50.9070549, 48.5675201},
 {1082.95947, 0, 0, 0, 0, 0.97417146, 74.2786179, 641.095825, 2037.95776, 3560.11499, 4002.5083}},
{11, 8.052, {171.647552, 34.8925858, 16.1040001, 10.4675474, 8.44831085, 8.06005192},
 {102.862061, 1447.51978, 12906.5879, 0, 0, 13.6051102, 788.036255, 4237.81445, 9688.19727, 14137.7637, 15188.7998}},
{11, 7.6171, {162.376633, 33.007988, 15.2342005, 9.90217972, 7.99200583, 7.62471724},
 {28.4541492, 896.136658, 20760.3008, 0, 0, 2.84694123, 468.369171, 4705.22266, 15609.0039, 27462.1641, 32092.1113}},
{11, 6.9769, {148.729233, 30.2337418, 13.9538002, 9.06992435, 7.32029581, 6.98387671},
 {32.9148712, 1279.87524, 34769.1055, 0, 0, 3.67583942, 859.4646, 9473.74414, 32965.2734, 59528.4609, 70212.0781}},
{11, 1.8, {38.3712883, 7.8001318, 3.60000014, 2.33998823, 1.88859415, 1.80180001},
 {21.2896595, 303.460571, 3104.04297, 21702.4473, 0, 131.338531, 4011.77026, 16156.6387, 31568.166, 42096.6328, 45800.4844}},
{11, 1.6139, {34.4041214, 6.99368429, 3.22779989, 2.09805942, 1.69333446, 1.61551392},
 {6.07634449, 168.462387, 3137.86792, 30639.4648, 0, 80.575058, 5529.34521, 26960.1426, 53302.7578, 69107.0547, 80889.2344}},
{11, 1.4806, {31.5625134, 6.41604137, 2.9612, 1.92477024, 1.55347359, 1.48208058},
 {7.20967674, 246.91098, 5340.70264, 61623.875, 0, 148.134735, 11953.7559, 63179.25, 133790.188, 182081.109, 211428.422}},
{11, 1.1606, {24.7409534, 5.02935123, 2.32119989, 1.50877237, 1.21772349, 1.16176057},
 {0.29370603, 27.1489658, 1690.91504, 59372.9727, 0, 36.6632385, 11795.0303, 125044.891, 392674.344, 691058.563, 8393959}},
{11, 1.1309, {24.1078262, 4.90064907, 2.26180005, 1.47016263, 1.1865617, 1.13203084},
 {0.216936782, 28.422842, 2151.36255, 82561.6563, 0, 43.5729256, 17289.3379, 189921.781, 605157.125, 1076519.5, 12751725}},
{10, 0.3602, {163.685959, 6.76398325, 1.44079995, 0.608731866, 0.396529913, 0},
 {5.11582518, 72.3587494, 756.797119, 5998.93994, 35548.1523, 0.771124959, 1304.55896, 20625.25, 66597.2422, 96260.0313, 0}},
{10, 0.2839, {129.012894, 5.33119059, 1.13559997, 0.479786158, 0.312534273, 0},
 {1.39358258, 37.3071594, 675.029053, 7056.30273, 40208.6094, 0.312348515, 2182.68188, 34230.7852, 75077.6719, 81696.9531, 0}},
{10, 0.2566, {116.606926, 4.8185401, 1.02639997, 0.4336496, 0.282480776, 0},
 {1.64342415, 54.2822533, 1134.02893, 13547.2734, 92375.8125, 0.452580839, 4910.6084, 88969.875, 233275.406, 287084.031, 0}},
{10, 0.1332, {60.5301743, 2.50128436, 0.532800019, 0.225105733, 0.146634609, 0},
 {0.06667943299, 5.77828264, 319.998474, 9173.61035, 116564.227, 0.217281744, 14959.1025, 275113.344, 149905.344, 1111816.63, 0}},
{10, 0.1332, {60.5301743, 2.50128436, 0.532800019, 0.225105733, 0.146634609, 0},
 {0.04926908761, 6.02834845, 406.214783, 12758.124, 170930.344, 0.176095545, 21010.75, 416563.594, 226317.938, 1665856.63, 0}},
{10, 0.0029115, {1.32307839, 0.05467347428, 0.01164602861, 0.004920397419, 0.003205162939, 0},
 {0.0001146181894, 0.03901197389, 7.91907215, 820.722168, 45399.4453, 17581.6465, 5085654.5, 1741765.38, 1076830.88, 847386.25, 0}},
{10, 0.0318, {14.4508972, 0.597153425, 0.127199993, 0.05374145508, 0.03500736132, 0},
 {0.893049419, 12.199091, 125.938431, 1018.71301, 6625.91211, 46.2834549, 14583.2295, 92570.0938, 99155.5078, 59714.5078, 0}},
{10, 0.022, {9.99747658, 0.413125038, 0.08799999952, 0.03717962652, 0.02421892807, 0},
 {0.200391889, 5.09999228, 89.8084869, 948.916504, 5960.54932, 67.7624512, 17511.6875, 136995.969, 2374781.5, 13462045, 0}},
{10, 0.022, {9.99747658, 0.413125038, 0.08799999952, 0.03717962652, 0.02421892807, 0},
 {0.228655919, 7.13864422, 144.625183, 1736.83936, 12636.8389, 107.579514, 43169.0313, 246647.641, 3599109.5, 24708374, 0}},

// Z=64,  19 orbitals
{11, 50.2391, {1070.96606, 217.706436, 100.478195, 65.3105011, 52.7118149, 50.2893372},
 {1143.10596, 0, 0, 0, 0, 0.959365785, 71.7253799, 616.497253, 1957.80847, 3417.84131, 3844.55103}},
{11, 8.3765, {178.545853, 36.2948761, 16.7511997, 10.8882246, 8.78783798, 8.38397598},
 {108.705818, 1509.59424, 13262.7119, 0, 0, 13.0326872, 752.893372, 4052.26807, 9269.20313, 13526.1328, 14527.7451}},
{11, 7.9302, {169.053223, 34.3652115, 15.8606005, 10.3093376, 8.32062054, 7.93823051},
 {31.297184, 970.748169, 22036.9688, 0, 0, 2.77000737, 449.687836, 4501.18213, 14910.6494, 26207.998, 29811.4863}},
{11, 7.2428, {154.397522, 31.3859959, 14.4855995, 9.41559219, 7.59928274, 7.25004292},
 {35.8704872, 1378.49341, 36935.5586, 0, 0, 3.52496028, 821.969116, 9058.11621, 31526.7383, 57033.2656, 67089.6094}},
{11, 1.8808, {40.0937309, 8.15027046, 3.76160002, 2.44502759, 1.97337091, 1.88268077},
 {22.6053371, 318.704803, 3224.01489, 22194.4258, 0, 124.233932, 3824.28784, 15422.8926, 30159.1445, 40230.457, 43631.0859}},
{11, 1.6883, {35.9901352, 7.31609011, 3.37660003, 2.19477892, 1.7713964, 1.68998826},
 {6.703969, 182.570953, 3325.80371, 31544.627, 0, 76.8523788, 5260.08838, 25697.5762, 50940.0703, 66192.6016, 77765.4844}},
{11, 1.544, {32.9140358, 6.69077969, 3.08800006, 2.00718999, 1.61999404, 1.54554403},
 {7.88950205, 266.344055, 5659.05615, 63945.3555, 0, 140.800232, 11401.4141, 60519.6875, 128700.688, 175722.438, 205292.328}},
{11, 1.2172, {25.9475174, 5.27462196, 2.43440008, 1.58235204, 1.27710927, 1.21841717},
 {0.335776091, 30.5740318, 1868.08398, 64148.168, 0, 34.2140961, 11034.293, 117430.922, 370179, 640741.938, 7457943}},
{11, 1.1852, {25.2653599, 5.13595295, 2.37039995, 1.54075217, 1.24353433, 1.18638515},
 {0.247009426, 31.9362679, 2374.0166, 89178.0625, 0, 40.2313499, 16143.1318, 178306.984, 570633.938, 998727.5, 11790371}},
{10, 0.3758, {170.77507, 7.05692673, 1.50319993, 0.635095596, 0.413703322, 0},
 {5.46481276, 76.5115356, 792.899414, 6229.29053, 36272.9961, 0.742980003, 1255.48206, 19839.9688, 64191.543, 93504.5156, 0}},
{10, 0.2885, {131.103271, 5.41757154, 1.15400004, 0.487560064, 0.317598224, 0},
 {1.54827595, 40.700161, 720.488281, 7362.80664, 40852.5234, 0.332672507, 2226.52393, 34164.9609, 74950.7344, 83071.2188, 0}},
{10, 0.2709, {123.105293, 5.08707142, 1.08360004, 0.457816362, 0.298223078, 0},
 {1.81181812, 58.9971085, 1211.78772, 14245.8799, 95347.7031, 0.417872757, 4602.69824, 84913.4297, 227768.609, 286891.344, 0}},
{10, 0.1405, {63.8475227, 2.6383667, 0.562000036, 0.237442613, 0.154670894, 0},
 {0.07735446095, 6.59424448, 357.122192, 9967.38086, 122043.969, 0.202335522, 14063.834, 268268.563, 165319.609, 712462.688, 0}},
{10, 0.1405, {63.8475227, 2.6383667, 0.562000036, 0.237442613, 0.154670894, 0},
 {0.05692562088, 6.86587, 452.948425, 13862.2871, 179168.156, 0.160692036, 19700.252, 405765.781, 255306.219, 1038431.19, 0}},
{10, 0.0092794, {4.21684551, 0.174252421, 0.03711760789, 0.01568203233, 0.01021532621, 0},
 {0.0001382994815, 0.05050205067, 10.4638147, 1061.43884, 56876.5391, 267.74054, 3218998.75, 2688464.25, 642599.563, 712302.25, 0}},
{10, 0.0085242, {3.87365603, 0.160070822, 0.03409677744, 0.01440574322, 0.00938394852, 0},
 {2.152928209e-05, 0.007652925793, 1.61883366, 167.511948, 9102.13379, 59.6805649, 582469.25, 432664.156, 113483.703, 131450.891, 0}},
{10, 0.0361, {16.4049511, 0.677900612, 0.144400001, 0.06100838631, 0.03974106163, 0},
 {0.981478751, 13.3112621, 136.460098, 1096.66003, 7067.83691, 38.4084663, 12845.1797, 87106.9688, 106088.938, 71148.1016, 0}},
{10, 0.0203, {9.22494411, 0.381201744, 0.08120000362, 0.03430665284, 0.02234746702, 0},
 {0.233047232, 5.84594774, 100.983673, 1045.93884, 6441.26855, 92.7021561, 20351.8477, 175719.719, 2570084.75, 10158516, 0}},
{10, 0.0203, {9.22494411, 0.381201744, 0.08120000362, 0.03430665284, 0.02234746702, 0},
 {0.263996065, 8.17296124, 163.252853, 1930.42529, 13845.0762, 149.189301, 51857.5273, 306965.938, 3910144, 18497008, 0}},

// Z=65,  19 orbitals
{11, 51.9957, {1108.41211, 225.318497, 103.991402, 67.5940704, 54.5548744, 52.0476952},
 {1205.34631, 0, 0, 0, 0, 0.945949912, 69.3186493, 593.315186, 1882.57104, 3287.10913, 3693.98096}},
{11, 8.708, {185.63176, 37.735302, 17.4160004, 11.320343, 9.13659859, 8.71670818},
 {114.814537, 1573.87476, 13375.7695, 0, 0, 12.5001888, 720.184631, 3880.36597, 8883.49219, 12978.498, 13936.375}},
{11, 8.2516, {175.902496, 35.757534, 16.5032005, 10.727026, 8.65773487, 8.25985146},
 {34.3973389, 1051.35217, 23490.6621, 0, 0, 2.69980812, 432.70813, 4317.91797, 14290.9492, 25129.9688, 29369.0313}},
{11, 7.514, {160.178802, 32.5612144, 15.0279999, 9.76815033, 7.88383102, 7.52151394},
 {39.047718, 1484.50305, 39273.8164, 0, 0, 3.38509846, 787.80304, 8687.35645, 30263.5215, 54709.1133, 64491.0703}},
{11, 1.9675, {41.9419479, 8.52597713, 3.93499994, 2.55773711, 2.06433821, 1.96946752},
 {23.9997387, 334.799286, 3353.47583, 22755.9551, 0, 117.800514, 3644.35376, 14739.792, 28850.6484, 38510.1875, 41981.3789}},
{11, 1.7677, {37.6827354, 7.66016245, 3.53539991, 2.29799843, 1.85470438, 1.76946771},
 {7.39434862, 197.90657, 3531.16821, 32558.916, 0, 73.1972122, 5008.93066, 24554.0254, 48755.4336, 63337.918, 75171.7266}},
{11, 1.6113, {34.3486977, 6.98241758, 3.22259998, 2.09467959, 1.69060647, 1.61291134},
 {8.62680721, 287.360413, 6009.1626, 66647.3047, 0, 133.595505, 10891.2568, 58180.1133, 124256.758, 170151.328, 197383.625}},
{11, 1.275, {27.1796608, 5.52509308, 2.54999995, 1.65749156, 1.33775413, 1.27627504},
 {0.383780777, 34.4529114, 2070.8125, 69956.3516, 0, 32.1261711, 10430.3506, 111923.531, 355110.25, 624042.125, 7105357}},
{11, 1.2412, {26.4591331, 5.37862396, 2.48239994, 1.61355186, 1.30229056, 1.24244118},
 {0.281034321, 35.9007416, 2629.06689, 97290.8906, 0, 37.2868767, 15211.0215, 169810.875, 547588.938, 974825.438, 10944593}},
{10, 0.3979, {180.817993, 7.47192955, 1.59160006, 0.672444224, 0.438032359, 0},
 {5.80675125, 80.4825516, 827.44812, 6460.78662, 36959.6289, 0.686059952, 1169.97119, 18663.2559, 60345.0586, 87095.1875, 0}},
{10, 0.3102, {140.964417, 5.82506275, 1.24080002, 0.524232686, 0.341486901, 0},
 {1.70720196, 44.126194, 766.399048, 7688.83008, 41268.0859, 0.283821434, 2007.77905, 31646.252, 69303.9453, 75000.9297, 0}},
{10, 0.285, {129.512772, 5.35184717, 1.13999999, 0.481645137, 0.313745201, 0},
 {1.97987509, 63.5598145, 1285.20654, 14906.1006, 97783.3438, 0.387636513, 4321.58154, 81055.0938, 217233.656, 268992.719, 0}},
{10, 0.147, {66.8013229, 2.76042628, 0.588, 0.248427495, 0.161826476, 0},
 {0.08829211444, 7.40487146, 392.962372, 10726.9199, 126635.656, 0.185695708, 13399.7881, 256613.031, 146052.031, 902967.5, 0}},
{10, 0.147, {66.8013229, 2.76042628, 0.588, 0.248427495, 0.161826476, 0},
 {0.06464228779, 7.68600273, 497.457062, 14905.9932, 186049.25, 0.144532889, 18709.584, 388000.563, 222527.609, 1354789.88, 0}},
{10, 0.0094, {4.27164888, 0.176517054, 0.03759999946, 0.01588583924, 0.01034808718, 0},
 {0.00017212292, 0.06180451438, 12.5331964, 1244.15955, 64718.2148, 299.456726, 3195598.25, 2391837.75, 624617.5, 773388.25, 0}},
{10, 0.0086, {3.90810442, 0.16149433, 0.03440000117, 0.01453385316, 0.009467399679, 0},
 {5.317702744e-05, 0.01868223213, 3.86977792, 392.014709, 20689.9512, 135.322144, 1153528.25, 769299.875, 221514.781, 285723.438, 0}},
{10, 0.039, {17.7227993, 0.732357979, 0.156000003, 0.06590933353, 0.04293355718, 0},
 {1.03394198, 13.9011078, 141.431335, 1129.07935, 7199.43555, 33.8753967, 11612.2939, 80027.2891, 98025.5156, 66108.4219, 0}},
{10, 0.0254, {11.5425415, 0.476971626, 0.101599999, 0.04292556643, 0.02796185389, 0},
 {0.2540856, 6.27169275, 106.286377, 1080.55054, 6513.52344, 56.7002983, 16049.0508, 111103.094, 1789577.25, 8258508, 0}},
{10, 0.0254, {11.5425415, 0.476971626, 0.101599999, 0.04292556643, 0.02796185389, 0},
 {0.285247445, 8.7033062, 171.062729, 1995.0603, 14106.6416, 88.2457733, 39786.7695, 217099.766, 2625967.25, 14889155, 0}},

// Z=66,  19 orbitals
{11, 53.7885, {1146.63, 233.087433, 107.577003, 69.9246979, 56.4359131, 53.842289},
 {1269.03821, 0, 0, 0, 0, 0.933116853, 67.0328217, 571.199646, 1810.45398, 3159.20166, 3552.20166}},
{11, 9.0458, {192.832764, 39.1991272, 18.0916004, 11.7594805, 9.49102497, 9.05484581},
 {121.073395, 1638.1886, 0, 0, 0, 12.0055704, 689.449402, 3717.6167, 8515.11621, 12436.9521, 13368.8477}},
{11, 8.5806, {182.915924, 37.1832275, 17.1611996, 11.1547232, 9.00292778, 8.58918095},
 {37.7270584, 1136.07471, 24637.5625, 0, 0, 2.63352537, 416.574951, 4142.22021, 13691.4355, 24044.9551, 28127.0898}},
{11, 7.7901, {166.064529, 33.7576675, 15.5802002, 10.127079, 8.17352009, 7.79789019},
 {42.4298096, 1595.1582, 41594.8398, 0, 0, 3.25432801, 755.406189, 8330.66699, 29036.6445, 52571.1172, 61865.2969}},
{11, 2.0468, {43.6324158, 8.86961651, 4.0935998, 2.66082668, 2.14754128, 2.04884672},
 {25.4214058, 350.885254, 3476.92969, 23219.5254, 0, 112.960999, 3499.95288, 14161.792, 27715.084, 36980.9453, 40356.6797}},
{11, 1.8418, {39.262352, 7.98126793, 3.68360019, 2.39432788, 1.93245149, 1.84364188},
 {8.1309967, 213.814957, 3732.61304, 33409.2852, 0, 70.5572205, 4805.42578, 23562.7539, 46824.7617, 60916.7031, 71674.3516}},
{11, 1.6756, {35.7194023, 7.26105547, 3.35119987, 2.17826891, 1.7580713, 1.67727554},
 {9.41305542, 309.148773, 6356.02832, 69104.2813, 0, 127.893814, 10452.666, 56052.543, 120100.625, 164736.438, 191838.219}},
{11, 1.3325, {28.4054108, 5.77426386, 2.66499996, 1.73224127, 1.39808428, 1.3338325},
 {0.436785549, 38.6510544, 2282.50684, 75612.8828, 0, 30.3219795, 9870.32422, 106393.242, 338903.469, 594702.563, 6478023}},
{11, 1.2949, {27.603878, 5.61132765, 2.58980012, 1.68336153, 1.35863364, 1.29619491},
 {0.318683326, 40.182621, 2893.49243, 105063.195, 0, 34.9681435, 14408.7461, 161738.984, 523696.688, 931776.188, 10292234}},
{10, 0.4163, {189.17952, 7.81745195, 1.6652, 0.703539908, 0.458288163, 0},
 {6.16890669, 84.6834488, 863.450195, 6687.70117, 37538.8594, 0.654098988, 1115.98035, 17845.0586, 57640.7539, 83085.8359, 0}},
{10, 0.3318, {150.780121, 6.23067665, 1.32720006, 0.560736358, 0.365265489, 0},
 {1.88586521, 47.8831367, 815.259094, 8020.146, 41676.5938, 0.256207049, 1830.93079, 29529.2188, 65308.7188, 70211.6172, 0}},
{10, 0.2929, {133.102768, 5.50019646, 1.17159998, 0.494996011, 0.322442025, 0},
 {2.16415858, 68.5139236, 1362.90686, 15563.0889, 100088.992, 0.388950288, 4276.51904, 79927.7734, 214024.359, 265998.406, 0}},
{10, 0.1542, {70.0732193, 2.89563084, 0.61680001, 0.260595351, 0.169752672, 0},
 {0.101031192, 8.34041023, 433.510498, 11552.5342, 131406.703, 0.174231455, 12695.7656, 247717.438, 144066.406, 825162.25, 0}},
{10, 0.1542, {70.0732193, 2.89563084, 0.61680001, 0.260595351, 0.169752672, 0},
 {0.07362566143, 8.63463211, 548.028015, 16047.1104, 193261.594, 0.132819816, 17672.1348, 374285.719, 220294.266, 1239739.75, 0}},
{10, 0.0042, {1.90860915, 0.07886932045, 0.0167999994, 0.007097928319, 0.004623613786, 0},
 {0.0002185936755, 0.07233473659, 13.9175034, 1352.66003, 68482.1563, 7500.48291, 4132833.5, 1385385.25, 828736.375, 883879.313, 0}},
{10, 0.0042, {1.90860915, 0.07886932045, 0.0167999994, 0.007097928319, 0.004623613786, 0},
 {0.0001387346565, 0.04390750453, 8.51859379, 845.185059, 43485.875, 4717.44727, 2783733, 1026051.56, 612730.688, 501245.656, 0}},
{10, 0.0629, {28.5836945, 1.181162, 0.251599997, 0.106299929, 0.06924411654, 0},
 {1.05230796, 13.9913273, 141.300461, 1124.23755, 7125.84473, 12.0562201, 5477.91113, 47303.6328, 71032.5, 36958.8359, 0}},
{10, 0.0263, {11.9515285, 0.493872195, 0.1052, 0.04444655031, 0.02895262837, 0},
 {0.263115197, 6.35300446, 105.325546, 1050.53577, 6179.10303, 51.8145599, 14367.9707, 105505.625, 1987611.88, 12897179, 0}},
{10, 0.0263, {11.9515285, 0.493872195, 0.1052, 0.04444655031, 0.02895262837, 0},
 {0.289452046, 8.66673756, 167.291809, 1923.84509, 13376.96, 79.1424026, 35618.4531, 194356.781, 2880928.5, 23535914, 0}},

// Z=67,  19 orbitals
{11, 55.6177, {1185.62378, 241.014099, 111.235397, 72.3026428, 58.3551445, 55.673317},
 {1334.60571, 0, 0, 0, 0, 0.92162478, 64.8667908, 550.195801, 1742.11914, 3040.18921, 3414.98804}},
{11, 9.3942, {200.25975, 40.7088852, 18.7884007, 12.2123985, 9.85657215, 9.40359402},
 {127.54055, 1703.4729, 0, 0, 0, 11.5333118, 660.07843, 3562.11353, 8163.54395, 11921.418, 12838.9736}},
{11, 8.9178, {190.104141, 38.6444511, 17.8355999, 11.5930815, 9.35672474, 8.92671776},
 {41.3182983, 1225.84668, 0, 0, 0, 2.57155919, 401.451569, 3977.12207, 13127.624, 23027.7676, 27011.9238}},
{11, 8.0711, {172.054718, 34.9753571, 16.1422005, 10.4923773, 8.46835136, 8.07917118},
 {46.0263824, 1711.56433, 44107.8242, 0, 0, 3.13175416, 725.036011, 7996.13916, 27882.2988, 50455.6523, 59582.1641}},
{11, 2.1283, {45.3697853, 9.22278881, 4.2565999, 2.76677608, 2.23305273, 2.13042831},
 {26.8941078, 367.326843, 3601.06226, 23651.1621, 0, 108.356148, 3362.00195, 13608.3203, 26627.5039, 35508.25, 38797.9258}},
{11, 1.9228, {40.9890594, 8.33227348, 3.84559989, 2.49962735, 2.01743817, 1.92472279},
 {8.92685795, 230.693222, 3941.90771, 34243.8906, 0, 67.2341232, 4589.64648, 22539.3301, 44850.4766, 58392.7813, 69366.3203}},
{11, 1.7412, {37.1178246, 7.54532671, 3.48239994, 2.26354861, 1.82690001, 1.74294114},
 {10.2482252, 331.990784, 6713.10693, 71558.0859, 0, 122.531265, 10040.0723, 54037.5586, 116145.539, 159645.594, 188161.109}},
{11, 1.3915, {29.6631374, 6.02993488, 2.78299999, 1.80894089, 1.45998824, 1.39289153},
 {0.495855212, 43.2557602, 2510.42969, 81572.2656, 0, 28.6561489, 9351.12891, 101240.547, 323746.344, 568443.188, 6167648.5}},
{11, 1.3514, {28.8083096, 5.85616541, 2.70280004, 1.75681114, 1.41791451, 1.35275137},
 {0.360295892, 44.8689651, 3179.04492, 113355.961, 0, 32.6780357, 13619.6924, 153859.719, 500477.813, 889989.313, 9361720}},
{10, 0.4357, {197.995483, 8.18175316, 1.7428, 0.736325562, 0.479644865, 0},
 {6.54965067, 89.0065689, 899.93219, 6912.58545, 38045.2539, 0.62371546, 1062.71436, 17037.9434, 54990.2344, 79177.1328, 0}},
{10, 0.3435, {156.096954, 6.45038366, 1.37399995, 0.580509126, 0.378145546, 0},
 {2.0760541, 51.7956772, 864.182861, 8324.95703, 41912.6055, 0.255674392, 1787.67529, 28664.5137, 63006.1758, 67616.25, 0}},
{10, 0.3066, {139.328461, 5.75746059, 1.22640002, 0.51814878, 0.337523788, 0},
 {2.3610158, 73.7348785, 1444.09924, 16252.5742, 102489.891, 0.350760937, 4059.06152, 76798.9766, 207234.531, 258288.594, 0}},
{10, 0.161, {73.163353, 3.02332401, 0.643999994, 0.272087246, 0.177238524, 0},
 {0.115379266, 9.36775875, 476.764099, 12403.8252, 135893.094, 0.166401088, 12151.7646, 239899.516, 141879.063, 761918.75, 0}},
{10, 0.161, {73.163353, 3.02332401, 0.643999994, 0.272087246, 0.177238524, 0},
 {0.08369409293, 9.67334175, 601.871643, 17223.3164, 200078.641, 0.124392033, 16865.3789, 362281.5, 217464.953, 1146450.5, 0}},
{10, 0.0037, {1.68139374, 0.06948012114, 0.0148, 0.006252937019, 0.004073183518, 0},
 {0.0002674403368, 0.08751519024, 16.528429, 1573.17749, 77197.3984, 13586.1748, 3802449.25, 1208605.63, 892395.938, 1131814.25, 0}},
{10, 0.0037, {1.68139374, 0.06948012114, 0.0148, 0.006252937019, 0.004073183518, 0},
 {0.0002103789157, 0.06618701667, 12.6178923, 1226.53137, 61204.2852, 10686.0869, 3228999, 1132930.5, 786884.563, 737857.188, 0}},
{10, 0.0512, {23.2668552, 0.96145463, 0.204799995, 0.08652712405, 0.056364052, 0},
 {1.10694396, 14.5887871, 146.121933, 1151.68323, 7187.39893, 19.6703682, 7636.67969, 57470.4766, 72151.2578, 45606.9375, 0}},
{10, 0.0203, {9.22494411, 0.381201744, 0.08120000362, 0.03430665284, 0.02234746702, 0},
 {0.286340177, 6.8015027, 110.562202, 1080.74817, 6214.21094, 101.692322, 18338.1328, 212050.547, 3549127.75, 20366612, 0}},
{10, 0.0203, {9.22494411, 0.381201744, 0.08120000362, 0.03430665284, 0.02234746702, 0},
 {0.311033279, 9.19705486, 174.79863, 1979.67224, 13542.6328, 159.988159, 48520.7188, 316384.688, 5354897.5, 37786016, 0}},

// Z=68,  19 orbitals
{11, 57.4855, {1225.44031, 249.108032, 114.971001, 74.7307739, 60.3148766, 57.542984},
 {1402.23413, 0, 0, 0, 0, 0.910493791, 62.8085632, 530.175903, 1676.71313, 2924.07813, 3286.56689}},
{11, 9.7513, {207.872177, 42.2563438, 19.5025997, 12.6766262, 10.2312489, 9.76105118},
 {134.224869, 1769.51392, 0, 0, 0, 11.0918627, 632.328796, 3414.5769, 7830.02783, 11448.7598, 12303.9238}},
{11, 9.2643, {197.490616, 40.145977, 18.5286007, 12.0435295, 9.72027874, 9.27356434},
 {45.1934395, 1320.92285, 0, 0, 0, 2.51335192, 387.175659, 3820.93506, 12594.4482, 22104.3848, 25883.5938}},
{11, 8.3579, {178.168549, 36.2181778, 16.7157993, 10.8652153, 8.76926708, 8.36625767},
 {49.8604622, 1834.03943, 46768.0977, 0, 0, 3.01659632, 696.403503, 7680.22266, 26794.0332, 48556.0273, 57260.2188}},
{11, 2.2065, {47.0368042, 9.56166077, 4.41300011, 2.86843562, 2.31510162, 2.20870662},
 {28.4245224, 384.056274, 3723.58569, 24031.4414, 0, 104.604179, 3244.11865, 13121.5859, 25649.2266, 34204.8359, 37387.1719}},
{11, 2.0058, {42.7584038, 8.69194698, 4.01160002, 2.60752678, 2.10452342, 2.00780582},
 {9.78788471, 248.512024, 4155.96094, 35022.5781, 0, 64.4948883, 4388.03076, 21574.4277, 42972.6523, 55936.0938, 66895.3828}},
{11, 1.8118, {38.6228333, 7.85126591, 3.62360001, 2.35532808, 1.90097499, 1.81361187},
 {11.1441288, 356.091522, 7084.56201, 74073.6016, 0, 116.850319, 9610.54688, 51969.5469, 112100.977, 154343.984, 183117.734}},
{11, 1.4533, {30.9805508, 6.29773951, 2.9066, 1.88928056, 1.52482998, 1.4547534},
 {0.561815262, 48.3105659, 2756.40894, 87908.3125, 0, 27.031208, 8847.34668, 96264.0625, 309125.344, 542704.563, 5805503}},
{11, 1.4093, {30.0425854, 6.10706949, 2.81859994, 1.83208072, 1.47866428, 1.41070926},
 {0.406345814, 49.9846649, 3485.59351, 122086.5, 0, 30.5725346, 12886.4717, 146502.688, 478717.344, 852600.375, 8976748}},
{10, 0.4491, {204.084839, 8.43338394, 1.79639995, 0.758971334, 0.494396389, 0},
 {6.93704367, 93.3750305, 936.191528, 7123.01416, 38406.8633, 0.617027283, 1041.06995, 16609.8438, 53253.5117, 76497.2344, 0}},
{10, 0.3662, {166.412537, 6.87665367, 1.4648, 0.618871748, 0.403135061, 0},
 {2.28188133, 55.9661674, 915.904053, 8649.78418, 42045.2266, 0.232938945, 1638.24182, 26802.6348, 59393.1172, 63320.1914, 0}},
{10, 0.32, {145.417831, 6.00909138, 1.27999997, 0.540794551, 0.352275312, 0},
 {2.57315612, 79.2279968, 1527.52917, 16944.2441, 104755.18, 0.332286477, 3873.14941, 74026.1875, 201022.984, 251265.359, 0}},
{10, 0.1767, {80.2979126, 3.31814504, 0.706799984, 0.298619986, 0.194522023, 0},
 {0.131258249, 10.4975939, 524.155212, 13348.2412, 140832.594, 0.129286721, 10204.1797, 221111.578, 141987.141, 671699.75, 0}},
{10, 0.1676, {76.16259, 3.14726162, 0.670400023, 0.283241153, 0.184504196, 0},
 {0.09472762048, 10.8004751, 659.165894, 18436.4121, 206537.156, 0.117730275, 16194.6572, 351368.469, 214190.594, 1068542.88, 0}},
{10, 0.0043, {1.95405221, 0.08074716479, 0.01720000058, 0.007266926579, 0.00473369984, 0},
 {0.0003274603223, 0.105456732, 19.5129871, 1820.59167, 86677.6875, 9084.09766, 3613573.25, 1161608.38, 818143.25, 1083411.63, 0}},
{10, 0.0043, {1.95405221, 0.08074716479, 0.01720000058, 0.007266926579, 0.00473369984, 0},
 {0.0003070689563, 0.09543409199, 17.8338051, 1700.11145, 82362.3672, 8540.36621, 3669606, 1304625.5, 859613.5, 804831.188, 0}},
{10, 0.0598, {27.1749592, 1.12294888, 0.239199996, 0.101060979, 0.06583145261, 0},
 {1.16627645, 15.2216101, 151.309143, 1184.74243, 7322.17041, 14.6506004, 6117.94434, 48891.0039, 66023.3203, 41461.5586, 0}},
{10, 0.0294, {13.3602638, 0.55208528, 0.117600001, 0.04968549684, 0.03236529604, 0},
 {0.312212288, 7.28530025, 116.16153, 1114.94849, 6270.96338, 43.9072113, 12625.0703, 90044.5938, 1784281.63, 12572056, 0}},
{10, 0.0294, {13.3602638, 0.55208528, 0.117600001, 0.04968549684, 0.03236529604, 0},
 {0.334918499, 9.76585865, 182.850647, 2044.66467, 13802.0625, 65.282608, 31433.3828, 168209.703, 2505131.25, 22829432, 0}},

// Z=69,  19 orbitals
{11, 59.3896, {1266.03076, 257.359253, 118.779198, 77.2060928, 62.3126945, 59.4489899},
 {1471.39294, 0, 0, 0, 0, 0.900793612, 60.857933, 511.148865, 1614.70337, 2815.89136, 3162.20386}},
{11, 10.1157, {215.640228, 43.8354378, 20.2313995, 13.1503439, 10.6135836, 10.1258154},
 {141.087921, 1836.06555, 0, 0, 0, 10.6769648, 606.147705, 3275.28369, 7513.65723, 10981.3125, 11814.0508}},
{11, 9.6169, {205.007126, 41.6739349, 19.233799, 12.5019064, 10.0902338, 9.6265173},
 {49.3521461, 1421.06287, 0, 0, 0, 2.46011782, 373.952698, 3675.54248, 12095.373, 21194.6816, 24834.6289}},
{11, 8.648, {184.352707, 37.475296, 17.2959995, 11.2423429, 9.07364559, 8.65664768},
 {53.9229431, 1962.2594, 48693.1875, 0, 0, 2.91011405, 669.807983, 7385.81055, 25773.7207, 46685.1289, 55121.957}},
{11, 2.3068, {49.1749344, 9.99630165, 4.61359978, 2.99882483, 2.42033815, 2.30910683},
 {30.0097198, 401.448883, 3854.00562, 24460.3086, 0, 98.967041, 3083.91357, 12506.667, 24474.5977, 32640.4004, 35710.4648}},
{11, 2.0898, {44.5490646, 9.05595303, 4.17960024, 2.7167263, 2.19265771, 2.09188986},
 {10.7098856, 267.221588, 4373.77295, 35729.5391, 0, 62.0311852, 4203.39844, 20678.1426, 41215.582, 53704.4063, 63454.3359}},
{11, 1.8845, {40.1726074, 8.16630459, 3.76900005, 2.44983768, 1.9772532, 1.88638449},
 {12.093379, 381.320831, 7467.22461, 76588.0703, 0, 110.840065, 9201.0498, 49991.0078, 108227.203, 149431.625, 174449.719}},
{11, 1.5146, {32.2873077, 6.56337738, 3.02920008, 1.96897018, 1.58914709, 1.51611471},
 {0.634685993, 53.8141022, 3018.67407, 94442.3594, 0, 25.6605186, 8411.94238, 91863.3594, 295954.406, 518979.188, 5293580}},
{11, 1.4677, {31.2875195, 6.36014032, 2.93540001, 1.90800035, 1.53993857, 1.46916771},
 {0.457368761, 55.5603256, 3813.18188, 131185.469, 0, 28.7137737, 12225.6748, 139787.984, 458623.906, 816952, 8491173}},
{10, 0.4717, {214.35498, 8.85777569, 1.88679993, 0.797164917, 0.519275844, 0},
 {7.33887577, 97.8807297, 973.763184, 7345.04736, 38792.1875, 0.582199454, 982.922791, 15753.3369, 50548.5898, 72527.1875, 0}},
{10, 0.3859, {175.364822, 7.24658871, 1.54359996, 0.6521644, 0.424822032, 0},
 {2.50546837, 60.3660126, 968.579041, 8961.88574, 42031.0703, 0.220133245, 1537.44055, 25404.2891, 56425.6523, 59866.5938, 0}},
{10, 0.3366, {152.961395, 6.3208127, 1.34640002, 0.568848252, 0.370549619, 0},
 {2.79648042, 84.9697342, 1614.07434, 17656.377, 106991.719, 0.30604738, 3628.43652, 70536.375, 193484.969, 242422.828, 0}},
{10, 0.1796, {81.6157608, 3.37260246, 0.718400002, 0.303520918, 0.197714522, 0},
 {0.148810402, 11.7134838, 572.790588, 14228.7041, 144489.688, 0.137195289, 10440.6719, 219468.938, 138375.516, 642768.188, 0}},
{10, 0.1796, {81.6157608, 3.37260246, 0.718400002, 0.303520918, 0.197714522, 0},
 {0.106935367, 12.0330038, 721.105347, 19745.8281, 213306.766, 0.09786464274, 14379.4043, 330647.188, 213969, 969475.813, 0}},
{10, 0.0053, {2.40848303, 0.09952557087, 0.02119999938, 0.00895690918, 0.00583455991, 0},
 {0.0003954572894, 0.125986248, 22.9034939, 2096.21631, 96838.7969, 4822.57129, 3442956.75, 1159065.38, 719061.125, 957708.813, 0}},
{10, 0.0053, {2.40848303, 0.09952557087, 0.02119999938, 0.00895690918, 0.00583455991, 0},
 {0.0004371200339, 0.133366182, 24.3661098, 2279.20117, 107210.68, 5262.50977, 4053068, 1512070.63, 890228, 796651.313, 0}},
{10, 0.0532, {24.1757164, 0.999011397, 0.212799996, 0.08990709484, 0.05856577307, 0},
 {1.22701144, 15.8550014, 156.277832, 1212.51257, 7389.15381, 19.642437, 7400.30957, 53884.4375, 65175.7578, 49021.0898, 0}},
{10, 0.0323, {14.678113, 0.606542647, 0.129199997, 0.05458644778, 0.03555779159, 0},
 {0.338816017, 7.78060484, 121.763123, 1146.57324, 6304.62207, 36.6902924, 11354.2598, 76200.6406, 1554979.25, 11641636, 0}},
{10, 0.0323, {14.678113, 0.606542647, 0.129199997, 0.05458644778, 0.03555779159, 0},
 {0.359959543, 10.3489361, 190.831161, 2104.99976, 14006.3213, 53.486084, 28089.1348, 148433.328, 2130197.75, 21017072, 0}},

// Z=70,  19 orbitals
{11, 61.3323, {1307.44409, 265.777771, 122.664604, 79.7315903, 64.3510132, 61.3936348},
 {1541.71777, 0, 0, 0, 0, 0.891275764, 59.0038948, 493.002045, 1555.27869, 2710.34204, 3045.1167}},
{11, 10.4864, {223.542587, 45.4418335, 20.9727993, 13.6322508, 11.0025291, 10.4968863},
 {148.15921, 1903.04565, 0, 0, 0, 10.2957468, 581.685791, 3144.11011, 7215.60938, 10556.7441, 11334.2373}},
{11, 9.9782, {212.709091, 43.2395935, 19.9563999, 12.9715948, 10.4693165, 9.98817825},
 {53.827343, 1526.73938, 0, 0, 0, 2.41081572, 361.532532, 3538.25586, 11624.1348, 20375.9707, 23828.7031}},
{11, 8.9436, {190.654129, 38.7562523, 17.8871994, 11.6266212, 9.38379478, 8.95254326},
 {58.2434082, 2096.82056, 0, 0, 0, 2.81023669, 644.727661, 7107.52979, 24810.9688, 44999.6055, 53052.9805}},
{11, 2.3981, {51.121212, 10.391942, 4.7961998, 3.11751413, 2.51613188, 2.40049815},
 {31.6497993, 419.014099, 3980.2395, 24817.1445, 0, 94.8272552, 2958.55615, 12004.4395, 23485.4336, 31333.7148, 34275.3516}},
{11, 2.173, {46.32267, 9.41649151, 4.34599972, 2.82488561, 2.27995276, 2.17517304},
 {11.6992607, 286.825867, 4593.76123, 36350.0625, 0, 59.9591522, 4040.46191, 19866.1914, 39596.6797, 51595.8047, 61729.0781}},
{11, 1.9498, {41.5646286, 8.44927597, 3.89960003, 2.5347271, 2.04576707, 1.9517498},
 {13.0980949, 407.446991, 7849.08105, 78921.0469, 0, 107.2062, 8900.72363, 48445.7031, 105081.813, 145282.313, 172125.234}},
{11, 1.5763, {33.6025887, 6.83074808, 3.15259981, 2.04917955, 1.65388381, 1.57787621},
 {0.715343714, 59.8061752, 3298.4585, 101215.008, 0, 24.4502983, 8020.99121, 87857.6094, 283850.594, 498111.625, 5105320.5}},
{11, 1.5278, {32.5686951, 6.62057781, 3.05559993, 1.98612988, 1.60299659, 1.52932775},
 {0.513391793, 61.618515, 4163.95898, 140750.281, 0, 26.9674721, 11601.7119, 133430.359, 439541.156, 782701.625, 7711613.5}},
{10, 0.4872, {221.398666, 9.14884186, 1.94879997, 0.823359668, 0.536339164, 0},
 {7.75942469, 102.477119, 1010.9447, 7549.69922, 39027.3984, 0.573737562, 957.574097, 15289.0342, 48781.6914, 69840.4219, 0}},
{10, 0.3967, {180.272675, 7.44939518, 1.58679998, 0.670416236, 0.436711311, 0},
 {2.74265718, 64.9363937, 1021.21985, 9246.31445, 41915.9375, 0.224003002, 1520.10571, 24818.6543, 54561.5313, 57819.707, 0}},
{10, 0.3435, {156.096954, 6.45038366, 1.37399995, 0.580509126, 0.378145546, 0},
 {3.03556943, 90.9325562, 1700.32141, 18316.3105, 108793.031, 0.312239051, 3632.25, 70013.7891, 191272.922, 240422.688, 0}},
{10, 0.1981, {90.0227356, 3.72000313, 0.792400002, 0.33478561, 0.218080446, 0},
 {0.168596536, 13.065465, 627.022766, 15258.4873, 148995.359, 0.104630895, 8614.63477, 200358.922, 139436.656, 566868.563, 0}},
{10, 0.1849, {84.0242462, 3.47212815, 0.739600003, 0.312477857, 0.203549087, 0},
 {0.120508559, 13.3664236, 785.539612, 21021.0391, 218893, 0.09704419225, 14176.207, 323827.875, 209291.875, 922033.5, 0}},
{10, 0.0063, {2.86291385, 0.118303984, 0.02520000003, 0.01064689271, 0.006935420446, 0},
 {0.0004783113254, 0.150039479, 26.7413387, 2401.42017, 107613.094, 2872.09058, 3231043.25, 1160468.25, 645744.625, 861925.125, 0}},
{10, 0.0063, {2.86291385, 0.118303984, 0.02520000003, 0.01064689271, 0.006935420446, 0},
 {0.0006005421747, 0.180947632, 32.4296608, 2977.81274, 135966.766, 3564.33423, 4325047.5, 1725410.88, 922834.375, 786038.563, 0}},
{10, 0.0541, {24.5847034, 1.01591194, 0.216399997, 0.09142807871, 0.05955654755, 0},
 {1.28677082, 16.4926815, 161.412033, 1242.0907, 7478.63379, 19.6980019, 7301.84863, 52230.3672, 62098.707, 51460.8164, 0}},
{10, 0.0234, {10.6336794, 0.439414799, 0.09360000491, 0.03954559937, 0.02576013282, 0},
 {0.368140638, 8.30016899, 127.264908, 1173.84985, 6308.03711, 84.3702698, 15436.2383, 179142.203, 3209781.5, 20650992, 0}},
{10, 0.0234, {10.6336794, 0.439414799, 0.09360000491, 0.03954559937, 0.02576013282, 0},
 {0.384861618, 10.9296017, 198.542496, 2156.73022, 14119.7676, 128.185013, 41605.4023, 259133.484, 4670217.5, 38034944, 0}},

// Z=71,  19 orbitals
{11, 63.3138, {1349.68445, 274.36441, 126.627602, 82.3075256, 66.4300385, 63.3771133},
 {1613.48828, 0, 0, 0, 0, 0.883233249, 57.2371292, 475.654175, 1498.60583, 2611.30737, 2930.99756}},
{11, 10.8704, {231.728455, 47.1058617, 21.7408009, 14.1314487, 11.4054298, 10.8812704},
 {155.393066, 1970.14307, 0, 0, 0, 9.9206028, 557.767273, 3016.21631, 6924.03613, 10125.4639, 10887.1211}},
{11, 10.3486, {220.605057, 44.8446884, 20.6972008, 13.4531116, 10.8579473, 10.3589487},
 {58.6059151, 1637.07654, 0, 0, 0, 2.36310601, 349.590149, 3405.74805, 11166.1563, 19539.6133, 22904.3906}},
{11, 9.2441, {197.059998, 40.0584412, 18.4881992, 12.0172691, 9.69908524, 9.25334454},
 {62.7940331, 2236.1731, 0, 0, 0, 2.71535683, 620.746887, 6838.91699, 23871.2461, 43283.5977, 51212.6484}},
{11, 2.4912, {53.1058617, 10.7953815, 4.98239994, 3.23854375, 2.61381435, 2.49369121},
 {33.3201714, 436.623932, 4102.1499, 25088.7871, 0, 90.3469238, 2837.81519, 11514.1992, 22520.4492, 30028.6621, 32739.8477}},
{11, 2.2634, {48.2518921, 9.80866528, 4.52699995, 2.94253516, 2.37490702, 2.26576352},
 {12.7611723, 307.310638, 4814.39746, 36885.3945, 0, 57.6274033, 3863.48804, 19000.7676, 37902.2305, 49390.6875, 59108.7813}},
{11, 2.0236, {43.1378517, 8.76908112, 4.0472002, 2.63066673, 2.12319946, 2.02562356},
 {14.1667366, 434.669891, 8237.31543, 81203.1641, 0, 102.608055, 8537.13574, 46617.7578, 101414.391, 140450.109, 166391.203}},
{11, 1.6394, {34.9477158, 7.10418606, 3.27880001, 2.13120914, 1.72008955, 1.64103937},
 {0.804120004, 66.2429428, 3587.74219, 107704.898, 0, 23.3029346, 7631.8667, 83671.4688, 270691.625, 470973.875, 4747013}},
{11, 1.5885, {33.8626595, 6.88361549, 3.17699981, 2.0650394, 1.66668427, 1.59008849},
 {0.574538291, 68.0975571, 4525.4375, 149873.734, 0, 25.3744278, 10999.21, 126918.438, 419047.344, 740054.438, 7348300.5}},
{10, 0.5062, {230.032837, 9.50563145, 2.02480006, 0.855469346, 0.557255507, 0},
 {8.19799042, 107.265495, 1049.34241, 7751.48779, 39244.7852, 0.527745187, 922.143555, 14694.8594, 46854.3125, 67806.1094, 0}},
{10, 0.4101, {186.362045, 7.70102596, 1.64040005, 0.693062007, 0.451462835, 0},
 {3.00205064, 69.8310776, 1076.05896, 9526.97656, 41866.3672, 0.224305376, 1484.56787, 24069.7949, 53115.8008, 57030.2344, 0}},
{10, 0.3593, {163.276962, 6.74708271, 1.43719995, 0.607210875, 0.395539135, 0},
 {3.29421878, 97.3660736, 1793.27148, 19027.498, 110932.484, 0.293062896, 3441.92285, 67151.6953, 186272.109, 238990.172, 0}},
{10, 0.2048, {93.067421, 3.84581852, 0.819200039, 0.346108496, 0.225456208, 0},
 {0.190909237, 14.5593395, 684.131104, 16224.4375, 152658.563, 0.09987676889, 8416.99707, 196324.063, 150398.484, 419206, 0}},
{10, 0.195, {88.6139984, 3.66178989, 0.780000031, 0.32954666, 0.214667782, 0},
 {0.135900751, 14.8690519, 857.071838, 22396.6934, 225081.625, 0.08654536307, 13099.7598, 310839.906, 230788.641, 645989.625, 0}},
{10, 0.0069, {3.1355722, 0.129571036, 0.02759999968, 0.01166088227, 0.007595936768, 0},
 {0.0005493086064, 0.180792093, 32.5430222, 2858.23975, 123095.383, 2413.55591, 3257706, 1017187.19, 744482.438, 1443660.5, 0}},
{10, 0.0069, {3.1355722, 0.129571036, 0.02759999968, 0.01166088227, 0.007595936768, 0},
 {0.0006554692518, 0.215257287, 39.6647453, 3566.21997, 156586.266, 3008.99121, 4377753, 1460495, 957145.563, 1859926.5, 0}},
{10, 0.0568, {25.8116665, 1.06661367, 0.227200001, 0.09599103034, 0.062528871, 0},
 {1.39114165, 17.7119598, 172.230255, 1314.03467, 7828.31934, 19.0485363, 7102.07129, 51647.7773, 66721.8984, 55492.9336, 0}},
{10, 0.028, {12.724061, 0.52579546, 0.112000003, 0.04731952026, 0.03082409129, 0},
 {0.419142008, 9.31658077, 140.336517, 1269.64856, 6687.05322, 60.994091, 13675.252, 119009.672, 2060831.88, 9758692, 0}},
{10, 0.028, {12.724061, 0.52579546, 0.112000003, 0.04731952026, 0.03082409129, 0},
 {0.437312812, 12.2985239, 220.535339, 2363.42725, 15287.5957, 90.7493744, 36299.0938, 200419.875, 2854342.5, 17617800, 0}},

// Z=72,  20 orbitals
{11, 65.3508, {1393.10791, 283.191559, 130.701599, 84.9556122, 68.5672989, 65.416153},
 {1688.20093, 0, 0, 0, 0, 0.874796391, 55.5281868, 458.902313, 1443.75891, 2514.02637, 2823.03589}},
{11, 11.2707, {240.26181, 48.8405228, 22.5414009, 14.6518364, 11.8254318, 11.281971},
 {162.86203, 2038.14709, 0, 0, 0, 9.50245762, 534.485107, 2891.82593, 6642.30713, 9725.18066, 10436.9785}},
{11, 10.7394, {228.935883, 46.5381851, 21.4787998, 13.9611492, 11.2679815, 10.7501392},
 {63.7537384, 1754.60303, 0, 0, 0, 2.31085491, 337.440704, 3274.01147, 10717.6094, 18738.5156, 21985.8203}},
{11, 9.5607, {203.809082, 41.430397, 19.1214008, 12.4288473, 10.0312672, 9.570261},
 {67.6438751, 2383.94409, 0, 0, 0, 2.61657453, 596.337463, 6570.35596, 22947.1758, 41677.6367, 49231.2305}},
{11, 2.6009, {55.4443779, 11.2707567, 5.20179987, 3.38115287, 2.72891355, 2.60350084},
 {35.0507317, 454.802521, 4229.03564, 25362.293, 0, 85.5042877, 2697.13721, 10965.4121, 21466.5176, 28631.2539, 31063.2051}},
{11, 2.3654, {50.4241333, 10.2502394, 4.73079967, 3.07500434, 2.48182249, 2.36776543},
 {13.8976984, 328.929138, 5043.80518, 37395.7422, 0, 54.8440781, 3668.76709, 18083.6582, 36142.5859, 47197.5508, 55615.4844}},
{11, 2.1076, {44.9285126, 9.13308716, 4.21519995, 2.73986626, 2.21133375, 2.10970759},
 {15.2976475, 463.293823, 8645.29688, 83623.2344, 0, 97.0691605, 8123.25684, 44606.0742, 97459.6406, 135486.656, 157065.313}},
{11, 1.7164, {36.5891533, 7.43785858, 3.43279982, 2.2313087, 1.80087936, 1.7181164},
 {0.90207845, 73.312912, 3907.9895, 115234.406, 0, 21.6249485, 7123.05469, 78621.7813, 255715.156, 441644.375, 4246291.5}},
{11, 1.6617, {35.4230919, 7.20082092, 3.32339978, 2.16019893, 1.743487, 1.66336167},
 {0.642104566, 75.2015991, 4923.74219, 160371.875, 0, 23.258791, 10237.4199, 119206.297, 395983.813, 693805.25, 6792580}},
{10, 0.5381, {244.52919, 10.1046629, 2.15240002, 0.90937984, 0.592372954, 0},
 {8.65830803, 112.285439, 1090.12671, 7975.20752, 39552.8906, 0.480294138, 849.728149, 13673.6367, 44007.043, 64164.0156, 0}},
{10, 0.437, {198.586227, 8.20616531, 1.74800003, 0.73852253, 0.481075972, 0},
 {3.28429103, 75.095047, 1134.99402, 9836.72656, 41694.0547, 0.205113158, 1359.13391, 22479.2051, 50541.5391, 54532.5078, 0}},
{10, 0.3804, {172.865463, 7.14330721, 1.52160001, 0.642869532, 0.418767303, 0},
 {3.57295585, 104.230095, 1892.04309, 19789.1621, 113293.516, 0.264104277, 3174.73242, 63355.3164, 179580.281, 234113.688, 0}},
{10, 0.2238, {101.701599, 4.20260811, 0.895199955, 0.378218174, 0.246372551, 0},
 {0.216101766, 16.2323036, 748.213989, 17346.4609, 157133.984, 0.07853221148, 7099.69336, 181097.531, 164954.906, 298310.938, 0}},
{10, 0.2137, {97.1118469, 4.01294613, 0.854799986, 0.361149341, 0.235253856, 0},
 {0.153188035, 16.5431213, 936.57666, 23953.7383, 232304.688, 0.06261973083, 10912.792, 285928.219, 257060.188, 446283.844, 0}},
{10, 0.0171, {7.7707653, 0.321110815, 0.06839999557, 0.02889870666, 0.01882471144, 0},
 {0.000667571032, 0.220822141, 39.3831596, 3398.25854, 141960.484, 71.2154617, 1770791.25, 1856734.88, 476388.219, 709763.125, 0}},
{10, 0.0171, {7.7707653, 0.321110815, 0.06839999557, 0.02889870666, 0.01882471144, 0},
 {0.0007900626515, 0.262668312, 48.090107, 4250.57373, 181161.719, 87.1911545, 2318386, 2598049.25, 641885.063, 975640.875, 0}},
{10, 0.0649, {29.4925556, 1.21871877, 0.259599984, 0.109679893, 0.07144583762, 0},
 {1.51112092, 19.0874825, 184.359863, 1395.62964, 8246.8623, 15.4164753, 6106.85547, 47049.9883, 67087.8125, 55035.6289, 0}},
{10, 0.0381, {17.3138123, 0.715457439, 0.152400002, 0.06438834965, 0.0419427827, 0},
 {0.477280259, 10.4594488, 154.85556, 1375.17896, 7105.65918, 31.8034935, 10495.0957, 64203.2539, 1095132.38, 5265648.5, 0}},
{10, 0.0306, {13.9055805, 0.574619353, 0.122400001, 0.05171347782, 0.03368632868, 0},
 {0.495673567, 13.8010616, 244.304199, 2582.97314, 16523.7656, 80.3886871, 35074.4531, 189362.813, 2306188, 12463341, 0}},
{10, 0.005, {2.27215385, 0.09389205277, 0.01999999955, 0.008449914865, 0.005504302215, 0},
 {0.005636430345, 0.354511261, 14.7646332, 333.031616, 3140.31787, 648.280334, 102469.688, 7361422, 11662020, 12384611, 0}},

// Z=73,  20 orbitals
{11, 67.4164, {1437.14111, 292.142639, 134.832794, 87.6408768, 70.7345657, 67.4838181},
 {1764.08289, 0, 0, 0, 0, 0.868327558, 53.9224358, 443.057892, 1391.93054, 2423.3313, 2718.72656}},
{11, 11.6815, {249.018982, 50.6206856, 23.362999, 15.185873, 12.2564507, 11.693181},
 {170.5103, 2106.43604, 0, 0, 0, 9.15231609, 511.630402, 2773.23804, 6372.74121, 9331.01563, 10000.7559}},
{11, 11.1361, {237.392487, 48.2572441, 22.2721996, 14.4768572, 11.684207, 11.1472359},
 {69.2644882, 1877.56543, 0, 0, 0, 2.26493645, 326.306915, 3151.5332, 10298.1045, 18016.4746, 20370.0176}},
{11, 9.8811, {210.639175, 42.818821, 19.7621994, 12.8453655, 10.3674374, 9.89098072},
 {72.7586212, 2537.89771, 0, 0, 0, 2.52462101, 573.537903, 6318.8374, 22075.7539, 40093.8438, 47404.6367}},
{11, 2.708, {57.7274666, 11.7348642, 5.41599989, 3.52038217, 2.84128499, 2.7107079},
 {36.8399849, 473.191589, 4352.81982, 25634.0977, 0, 81.4038849, 2574.1189, 10475.6426, 20512.1055, 27388.6699, 29541.9727}},
{11, 2.4687, {52.6262169, 10.6978798, 4.93739986, 3.2092936, 2.59020662, 2.47116852},
 {15.1124964, 351.51358, 5274.74268, 37790.0078, 0, 52.0738792, 3491.09937, 17234.5039, 34495.3164, 45095.1875, 53710.6758}},
{11, 2.194, {46.7703362, 9.50749397, 4.38800001, 2.85218573, 2.30198646, 2.19619393},
 {16.5006599, 493.18277, 9062.79297, 86018.9922, 0, 91.9110031, 7732.41016, 42691.3672, 93661.6953, 130546.828, 151169.859}},
{11, 1.7932, {38.2263298, 7.77066422, 3.58640003, 2.33114839, 1.88145947, 1.79499328},
 {1.00968874, 80.9500122, 4246.04883, 122891.891, 0, 20.2029152, 6680.4082, 74131.8359, 242187.578, 416389.344, 3998763.75}},
{11, 1.7351, {36.9877892, 7.51889324, 3.47019982, 2.25561857, 1.82049978, 1.736835},
 {0.71572715, 82.8500214, 5344.44971, 171114.109, 0, 21.4328136, 9566.49414, 112277.258, 374941.938, 652197.063, 6060169}},
{10, 0.5655, {256.980591, 10.6191912, 2.26200008, 0.955685377, 0.62253654, 0},
 {9.13684845, 117.435211, 1131.14722, 8187.0127, 39774.8008, 0.45049879, 800.082458, 12929.4141, 41821.4141, 61330.8359, 0}},
{10, 0.4648, {211.219406, 8.72820473, 1.8592, 0.785504103, 0.511679888, 0},
 {3.59210086, 80.6877594, 1195.90381, 10142.7871, 41449.8906, 0.189176232, 1248.69189, 21034.209, 48140.8828, 52246.6875, 0}},
{10, 0.4045, {183.817245, 7.59586716, 1.61800003, 0.683598101, 0.445298016, 0},
 {3.87172508, 111.500801, 1995.66016, 20583.1133, 115724.797, 0.234615475, 2898.20264, 59392.9688, 172377.781, 228286.563, 0}},
{10, 0.2413, {109.654144, 4.53123045, 0.965200007, 0.407792896, 0.265637606, 0},
 {0.244223863, 18.0670052, 816.785461, 18500.4902, 161459.875, 0.06520989537, 6196.16357, 169041.797, 178054.969, 224480.016, 0}},
{10, 0.2293, {104.200966, 4.30588913, 0.917199969, 0.387513071, 0.25242728, 0},
 {0.17240946, 18.3736744, 1021.24396, 25537.3867, 239193.469, 0.05150463432, 9607.10059, 268493.25, 279922.344, 331131.531, 0}},
{10, 0.025, {11.3607683, 0.469460249, 0.100000001, 0.04224957153, 0.02752150968, 0},
 {0.0008157273987, 0.268452168, 47.1837082, 3992.97632, 161293.719, 15.8366928, 1023417.75, 2180539.5, 449506.844, 567391.875, 0}},
{10, 0.025, {11.3607683, 0.469460249, 0.100000001, 0.04224957153, 0.02752150968, 0},
 {0.0009591804119, 0.318981498, 57.6663742, 5002.06787, 206289.688, 19.2783432, 1330570.88, 3008531.25, 611872.125, 797208.625, 0}},
{10, 0.0711, {32.3100243, 1.335145, 0.284399986, 0.120157786, 0.07827117294, 0},
 {1.64040196, 20.5572128, 197.213211, 1480.7583, 8680.72852, 13.6645327, 5590.10547, 44457.9922, 67152.5859, 56622.4805, 0}},
{10, 0.0449, {20.4039402, 0.843150616, 0.1796, 0.07588023692, 0.04942863062, 0},
 {0.541975379, 11.7062254, 170.3069, 1483.9187, 7528.1333, 23.390873, 9218.46289, 50637.6055, 783713.75, 3727575.25, 0}},
{10, 0.0364, {16.5412788, 0.683534145, 0.145600006, 0.0615153797, 0.04007131979, 0},
 {0.561229527, 15.4432154, 269.733032, 2815.31812, 17838.4727, 56.7247391, 30256.5352, 162221.516, 1595024.88, 8489136, 0}},
{10, 0.0057, {2.59025526, 0.107036933, 0.02280000038, 0.00963290222, 0.006274904124, 0},
 {0.01150703989, 0.728107095, 30.0322151, 661.442322, 6055.97852, 944.121033, 119037.039, 11622477, 14882909, 12524732, 0}},

// Z=74,  20 orbitals
{11, 69.525, {1482.09094, 301.28006, 139.050003, 90.3820419, 72.9469452, 69.5945282},
 {1839.5907, 0, 0, 0, 0, 0.86160326, 52.3897018, 427.891815, 1342.07275, 2334.80713, 2620.14185}},
{11, 12.0998, {257.936066, 52.4333496, 24.1996002, 15.729661, 12.6953392, 12.1119003},
 {178.335876, 2174.81152, 0, 0, 0, 8.82645988, 490.719055, 2661.26294, 6117.26758, 8952.00098, 9617.07813}},
{11, 11.544, {246.087845, 50.0248413, 23.0879993, 15.0071239, 12.1121836, 11.5555439},
 {75.1551666, 2006.74841, 0, 0, 0, 2.20830035, 315.302155, 3034.927, 9898.46484, 17287.3848, 19598.9844}},
{11, 10.2068, {217.582245, 44.2302132, 20.413599, 13.2687731, 10.7091675, 10.2170067},
 {78.1690216, 2698.55029, 0, 0, 0, 2.43909431, 552.122559, 6081.44873, 21254.2305, 38657.418, 45660.5977}},
{11, 2.8196, {60.1064911, 12.2184725, 5.63920021, 3.66546154, 2.95837784, 2.82241964},
 {38.6690903, 491.844788, 4476.14502, 25778.6523, 0, 77.4271851, 2455.24756, 10003.251, 19595.5137, 26147.5293, 28204.7148}},
{11, 2.5749, {54.8901253, 11.1580877, 5.14979982, 3.34735298, 2.70163393, 2.57747483},
 {16.4158726, 375.142456, 5507.75635, 38107.6367, 0, 49.7714119, 3324.67725, 16432.8613, 32928.4063, 43064.8789, 51348.9023}},
{11, 2.281, {48.6249466, 9.88449955, 4.5619998, 2.96528506, 2.39326835, 2.28328085},
 {17.7657871, 524.185852, 9487.11426, 88367.6172, 0, 87.2588196, 7375.69922, 40920.1836, 90119.0391, 125967.289, 147796.5}},
{11, 1.8716, {39.8976097, 8.11040306, 3.74320006, 2.4330678, 1.96371818, 1.87347162},
 {1.1282568, 89.2119217, 4604.43408, 130792.68, 0, 18.8366451, 6276.40381, 69987.8516, 229586.891, 393425.563, 3693187}},
{11, 1.8092, {38.5674057, 7.83999872, 3.6184001, 2.35194802, 1.89824688, 1.81100917},
 {0.796105862, 91.0866699, 5788.82764, 182139.047, 0, 19.8286781, 8965.3457, 105974.055, 355593.156, 616299.875, 5715916.5}},
{10, 0.595, {270.386292, 11.1731539, 2.37999988, 1.00553989, 0.655011952, 0},
 {9.64179802, 122.77285, 1173.07715, 8399.04297, 39935.5742, 0.421601743, 751.282166, 12201.376, 39692.8438, 58542.0313, 0}},
{10, 0.4916, {223.398148, 9.23146629, 1.96640003, 0.830795586, 0.541182935, 0},
 {3.92075706, 86.5611801, 1258.35046, 10439.9082, 41147.5898, 0.177314118, 1160.6355, 19814.1895, 46024.0625, 50295.0273, 0}},
{10, 0.4253, {193.269394, 7.98645782, 1.70120001, 0.718749762, 0.468195915, 0},
 {4.19067049, 119.129013, 2102.08813, 21368.9277, 118118.719, 0.216527209, 2712.24292, 56548.5117, 167150.609, 224852.141, 0}},
{10, 0.2588, {117.606674, 4.85985279, 1.0352, 0.437367588, 0.284902662, 0},
 {0.275548726, 20.0754814, 890.190186, 19696.584, 165691.563, 0.05526931211, 5477.47314, 158389.719, 189602.141, 177491.109, 0}},
{10, 0.2454, {111.517303, 4.60822201, 0.981599987, 0.414721817, 0.270151138, 0},
 {0.193728566, 20.3751984, 1111.99805, 27188.4883, 246114.906, 0.04290794209, 8504.03711, 252341.953, 300490.063, 259114.969, 0}},
{10, 0.0365, {16.5867214, 0.68541199, 0.145999998, 0.06168437749, 0.04018140584, 0},
 {0.0009907740168, 0.324058443, 56.1426735, 4666.74756, 182839.422, 3.27545214, 496492.813, 2512080, 442645.656, 433333.844, 0}},
{10, 0.0336, {15.2688732, 0.630954564, 0.134399995, 0.05678342655, 0.03698891029, 0},
 {0.001164219226, 0.385088027, 68.5707245, 5836.61279, 232993.234, 5.84086895, 776562.438, 3326166.5, 600367, 668055.625, 0}},
{10, 0.0771, {35.0366096, 1.44781542, 0.308400005, 0.130297676, 0.08487633616, 0},
 {1.77789497, 22.1060219, 210.645462, 1568.55933, 9127.75977, 12.3787432, 5189.83691, 42301.6406, 66690.7344, 58239.9375, 0}},
{10, 0.0468, {21.2673588, 0.878829598, 0.187199995, 0.07909119874, 0.05152026564, 0},
 {0.614715755, 13.061203, 186.514069, 1593.65063, 7943.89209, 23.3992462, 9255.58008, 50394.6445, 732290.313, 3191617, 0}},
{10, 0.0356, {16.1777344, 0.668511391, 0.142399997, 0.06016339362, 0.03919063136, 0},
 {0.630771518, 17.1803799, 296.236694, 3050.06738, 19139.8848, 66.5594025, 33376.7227, 178937.891, 1679722.5, 7773797, 0}},
{10, 0.0061, {2.77202749, 0.114548303, 0.02439999953, 0.01030889619, 0.006715248339, 0},
 {0.01981321163, 1.25190413, 50.960144, 1095.7583, 9751.22949, 1327.13818, 150278.109, 16371733, 17352948, 12707123, 0}},

// Z=75,  21 orbitals
{11, 71.6764, {1527.95313, 310.602966, 143.352798, 93.1788483, 75.2042389, 71.7480774},
 {1915.94141, 0, 0, 0, 0, 0.856772244, 50.9274673, 413.388062, 1294.55408, 2251.55054, 2524.54565}},
{11, 12.5267, {267.036438, 54.2832794, 25.0534, 16.2846279, 13.1432505, 12.5392265},
 {186.36264, 2243.05322, 0, 0, 0, 8.52171612, 470.999451, 2555.01782, 5874.97314, 8606.56836, 9233.93164}},
{11, 11.9587, {254.928162, 51.821907, 23.9174004, 15.5462313, 12.5472946, 11.9706583},
 {81.4502411, 2141.64185, 0, 0, 0, 2.16742635, 305.453949, 2925.90161, 9523.15723, 16638.9668, 18821.6328}},
{11, 10.5353, {224.585007, 45.6537361, 21.0706005, 13.6958208, 11.0538359, 10.5458355},
 {83.8583908, 2865.32886, 0, 0, 0, 2.3597424, 532.16394, 5859.44531, 20479.0078, 37230.543, 42373.4063}},
{11, 2.9317, {62.4961662, 12.7042475, 5.86339998, 3.81119061, 3.07599521, 2.93463159},
 {40.547554, 510.697052, 4597.0752, 25755.3281, 0, 73.8408203, 2346.29224, 9565.46484, 18739.3359, 25000.2461, 26851.5547}},
{11, 2.6816, {57.1646881, 11.6204624, 5.36319971, 3.48606229, 2.81358552, 2.68428159},
 {17.7996998, 399.693878, 5740.29785, 38368.5313, 0, 47.7427826, 3174.26465, 15695.8252, 31477.2227, 41229.4531, 48126.5547}},
{11, 2.3673, {50.4646378, 10.2584724, 4.73460007, 3.07747436, 2.48381591, 2.36966729},
 {19.0995522, 556.300598, 9915.43457, 90657.8906, 0, 83.2025681, 7056.9043, 39308.3945, 86865.1719, 121818.766, 139126.891}},
{11, 1.9489, {41.5454445, 8.44537544, 3.89779997, 2.53355718, 2.04482269, 1.95084894},
 {1.25757134, 98.0825653, 4979.79834, 138712.344, 0, 17.7671642, 5929.13525, 66329.9375, 218231.391, 373204.563, 3301167.25}},
{11, 1.8829, {40.1384964, 8.15937042, 3.7658, 2.44775772, 1.97557437, 1.88478291},
 {0.884033144, 99.9366913, 6255.14063, 193281.297, 0, 18.3719311, 8438.23535, 100324.594, 337942.75, 584597.063, 5269126}},
{10, 0.625, {284.019226, 11.7365065, 2.5, 1.05623937, 0.688037753, 0},
 {10.1583014, 128.224472, 1215.58923, 8605.82031, 40085.8594, 0.396186143, 707.321289, 11537.1494, 37729.2148, 55959.25, 0}},
{10, 0.5179, {235.349686, 9.72533894, 2.07159996, 0.875242174, 0.570135593, 0},
 {4.27405453, 92.7415237, 1322.24011, 10726.3467, 40762.2188, 0.159107268, 1087.4425, 18750.2637, 44108.4883, 48576.1055, 0}},
{10, 0.4444, {201.94902, 8.3451252, 1.77759993, 0.751028419, 0.489222348, 0},
 {4.53424215, 127.161201, 2211.54761, 22153.6484, 120493.781, 0.204284385, 2572.76563, 54298.7188, 162975.359, 222607.797, 0}},
{10, 0.2737, {124.377693, 5.13965082, 1.0948, 0.462548316, 0.301305473, 0},
 {0.310283005, 22.2605629, 967.831177, 20901.084, 169629.563, 0.04961069301, 5023.96338, 150751.188, 199682.219, 149001.297, 0}},
{10, 0.2602, {118.242882, 4.88614225, 1.04079998, 0.439733565, 0.286443889, 0},
 {0.217295468, 22.5520267, 1208.34241, 28870.9023, 252725.297, 0.03730181977, 7712.63477, 239449.125, 318651.625, 215458.031, 0}},
{10, 0.0406, {18.4498882, 0.762403429, 0.162399992, 0.06861330569, 0.04469493032, 0},
 {0.001201500068, 0.388672352, 66.1321869, 5375.47461, 202675.484, 2.35555959, 417513.969, 2530010.25, 432043.594, 420179.375, 0}},
{10, 0.0406, {18.4498882, 0.762403429, 0.162399992, 0.06861330569, 0.04469493032, 0},
 {0.001404164592, 0.461420894, 80.8667755, 6745.74121, 260033.359, 2.82714105, 538840.125, 3440587.5, 593333.813, 597202.438, 0}},
{10, 0.0828, {37.6268654, 1.55485237, 0.331200004, 0.139930591, 0.09115123749, 0},
 {1.92073214, 23.7256927, 224.696335, 1659.15942, 9583.94531, 11.4298239, 4883.94971, 40571.7852, 66197.8438, 60112.1328, 0}},
{10, 0.0456, {20.7220421, 0.856295526, 0.182400003, 0.07706321776, 0.050199233, 0},
 {0.693705618, 14.5190458, 203.630386, 1705.74011, 8358.34766, 27.6798763, 10018.3906, 56245.9688, 783515.875, 2965132.75, 0}},
{10, 0.0346, {15.7233038, 0.649733007, 0.138400003, 0.05847340822, 0.03808977082, 0},
 {0.708791614, 19.0751152, 324.471436, 3295.98877, 20495.2559, 79.1133957, 37051.2227, 198682.766, 1777104.75, 7157214, 0}},
{10, 0.006063, {2.75506401, 0.113847315, 0.02425068244, 0.01024581026, 0.006674154196, 0},
 {0.02498664521, 1.57436454, 63.2084389, 1326.85498, 11485.3457, 1627.0022, 192282.688, 17979350, 15546440, 10769062, 0}},
{10, 0.005209, {2.36718655, 0.09781908989, 0.02083650045, 0.008803332224, 0.005734519567, 0},
 {0.002776123583, 0.249567151, 12.2760305, 284.824127, 2630.48193, 502.536102, 73569.9609, 4687253.5, 4525270, 3534136, 0}},

// Z=76,  21 orbitals
{11, 73.8708, {1574.73206, 320.112183, 147.741608, 96.0315552, 77.5066452, 73.9446716},
 {1998.90881, 0, 0, 0, 0, 0.851439536, 49.5324211, 399.505341, 1248.81079, 2170.23877, 2434.48438}},
{11, 12.968, {276.443817, 56.1956139, 25.9360008, 16.8583145, 13.6062717, 12.9809685},
 {194.57341, 2311.51489, 0, 0, 0, 8.22620964, 451.903687, 2452.52637, 5641.01025, 8263.15332, 8874.19434}},
{11, 12.385, {264.015778, 53.669239, 24.7700005, 16.1004181, 12.9945765, 12.3973846},
 {88.1680374, 2283.05469, 0, 0, 0, 2.12791586, 296.024048, 2821.88159, 9165.03418, 15993.6143, 18093.2109}},
{11, 10.8709, {231.73912, 47.1080284, 21.7418003, 14.1320982, 11.4059544, 10.8817711},
 {89.8695908, 3039.3894, 0, 0, 0, 2.28465271, 513.147217, 5647.55273, 19742.5625, 35937.0938, 40865.7891}},
{11, 3.0485, {64.9860382, 13.2103891, 6.09700012, 3.96302986, 3.19854403, 3.05154848},
 {42.490757, 529.875488, 4717.12939, 0, 0, 70.3948135, 2240.75293, 9141.83301, 17910.0566, 23923.6348, 25565.959}},
{11, 2.7922, {59.5223923, 12.0997372, 5.58440018, 3.62984157, 2.92962909, 2.79499221},
 {19.2818851, 425.341064, 5974.59668, 38379.8125, 0, 45.8187141, 3030.62378, 14990.4873, 30079.4863, 39400.7031, 45775.0156}},
{11, 2.4572, {52.3810692, 10.6480455, 4.9144001, 3.19434381, 2.57814074, 2.45965719},
 {20.5141945, 589.783813, 10354.5322, 92906.1797, 0, 78.8293304, 6746.3877, 37738.2813, 83675.9063, 117593.523, 133297.078}},
{11, 2.0308, {43.2913399, 8.80028152, 4.06160021, 2.64002681, 2.13075399, 2.03283095},
 {1.39927042, 107.673134, 5380.69873, 147113.125, 0, 16.6950417, 5585.86182, 62750.9883, 207209.391, 354913.469, 3120916.5}},
{11, 1.9601, {41.7841988, 8.49390984, 3.92019987, 2.54811716, 2.05657411, 1.96206009},
 {0.979503036, 109.458542, 6751.12744, 205034.984, 0, 17.0489063, 7928.55518, 94875.0156, 320961.156, 555184.313, 4692289.5}},
{10, 0.6543, {297.334045, 12.2867136, 2.61720014, 1.10575581, 0.720292985, 0},
 {10.693861, 133.808701, 1258.50085, 8806.44629, 40180.1016, 0.37563315, 669.973206, 10958.3008, 35978.082, 53634.3945, 0}},
{10, 0.5465, {248.346405, 10.2624006, 2.18600011, 0.92357564, 0.601620197, 0},
 {4.65388107, 99.2522964, 1388.0127, 11008.4053, 40285.1289, 0.149091095, 1014.79846, 17698.9414, 42182.9766, 46797.1055, 0}},
{10, 0.4682, {212.764481, 8.79205132, 1.87279999, 0.79124999, 0.515422821, 0},
 {4.89679003, 135.598221, 2326.23145, 22975.7363, 122932.898, 0.187397838, 2393.85498, 51509.3242, 157549.375, 218334.688, 0}},
{10, 0.2894, {131.512253, 5.43447161, 1.15759993, 0.489081055, 0.318589002, 0},
 {0.348783582, 24.6409035, 1050.63794, 22148.418, 173515.281, 0.04452895001, 4606.27441, 143338.641, 208318.391, 130930.875, 0}},
{10, 0.2728, {123.968712, 5.12275028, 1.09119999, 0.461027354, 0.300314724, 0},
 {0.243262097, 24.91152, 1310.11157, 30572.0625, 259095.313, 0.03403215483, 7209.48877, 230274.516, 334488.406, 191418.828, 0}},
{10, 0.0463, {21.040144, 0.869440377, 0.185200006, 0.0782462135, 0.05096983537, 0},
 {0.001447967952, 0.463260889, 77.4304123, 6157.79639, 223774.094, 1.48723567, 326497.031, 2556644.5, 422422.719, 388640.281, 0}},
{10, 0.0463, {21.040144, 0.869440377, 0.185200006, 0.0782462135, 0.05096983537, 0},
 {0.001687170239, 0.549519956, 94.6720352, 7731.37061, 287446.625, 1.77697241, 420702.313, 3466842, 580206.188, 554103, 0}},
{10, 0.0837, {38.0358543, 1.57175291, 0.334800005, 0.141451567, 0.0921420157, 0},
 {2.07420516, 25.4199944, 239.049011, 1749.07959, 10019.4287, 11.989706, 5035.51123, 41312.3555, 67346.7422, 65126.1133, 0}},
{10, 0.058, {26.3569832, 1.08914781, 0.231999993, 0.09801901132, 0.06384990364, 0},
 {0.781921804, 16.1034527, 221.873444, 1824.38684, 8787.2832, 16.6444149, 7928.81055, 40503.4844, 488539.688, 2061220.63, 0}},
{10, 0.0454, {20.631155, 0.852539837, 0.181600004, 0.07672522217, 0.04997906089, 0},
 {0.793199241, 21.1034431, 354.554565, 3559.82104, 21988.5039, 42.5161095, 27586.7891, 154677.563, 1046575.56, 4693892.5, 0}},
{10, 0.0070526, {3.20493889, 0.132437468, 0.02821058221, 0.01191885024, 0.007763978094, 0},
 {0.03070637211, 1.92730153, 76.2970963, 1563.94263, 13180.4521, 1332.16736, 118385.555, 17022298, 13249806, 8565134, 0}},
{10, 0.0060279, {2.73928213, 0.113195166, 0.02411176823, 0.01018711925, 0.006635922473, 0},
 {0.006823690142, 0.613249362, 29.7935276, 675.614624, 6084.06689, 842.825684, 94287.0703, 9001080, 7622129.5, 5465839.5, 0}},

// Z=77,  21 orbitals
{11, 76.111, {1622.48718, 329.819885, 152.222, 98.9438019, 79.8571014, 76.1871109},
 {2084.31763, 0, 0, 0, 0, 0.848372638, 48.1983681, 386.203674, 1205.151, 2093.6853, 2346.68823}},
{11, 13.4185, {286.047272, 58.1478119, 26.8369999, 17.4439621, 14.0789442, 13.4319181},
 {202.978378, 2379.82813, 0, 0, 0, 7.94961882, 433.875183, 2355.21118, 5418.98389, 7944.38232, 8525.54395}},
{11, 12.8241, {273.376221, 55.5720367, 25.6481991, 16.6712456, 13.4552889, 12.8369246},
 {95.3508377, 2431.2627, 0, 0, 0, 2.08959174, 286.99173, 2722.27539, 8823.24609, 15401.6016, 17388.1426}},
{11, 11.2152, {239.07869, 48.6000175, 22.4304008, 14.5796862, 11.7672005, 11.2264156},
 {96.2003555, 3221.25073, 0, 0, 0, 2.21084285, 494.703766, 5443.54102, 19033.6289, 34653.4336, 39425.0234}},
{11, 3.1737, {67.6549683, 13.7529316, 6.34739971, 4.12578917, 3.32990599, 3.17687368},
 {44.4766579, 549.36792, 4837.60547, 0, 0, 66.8703003, 2134.5354, 8720.20117, 17093.248, 22808.8457, 24449.2773}},
{11, 2.9087, {62.005867, 12.604579, 5.81739998, 3.78129101, 3.05186319, 2.9116087},
 {20.8563576, 452.088379, 6211.59619, 38778.5977, 0, 43.8913994, 2889.7666, 14303.1133, 28721.6406, 37636.582, 44253.2461}},
{11, 2.5507, {54.3742447, 11.0532198, 5.1013999, 3.31589317, 2.67624283, 2.55325079},
 {21.9989281, 624.555847, 10804.3447, 95120.3594, 0, 74.9710388, 6445.19922, 36214.6797, 80581.5391, 113507.164, 130227.563}},
{11, 2.1161, {45.1097107, 9.16992092, 4.23220015, 2.75091624, 2.22025228, 2.11821628},
 {1.5547353, 118.027016, 5806.80859, 155937.141, 0, 15.675581, 5258.79541, 59341.3984, 196701.406, 337881.219, 2871178.25}},
{11, 2.0404, {43.4959831, 8.84188175, 4.08080006, 2.65250659, 2.14082623, 2.04244041},
 {1.08376312, 119.709694, 7277.5708, 217373.359, 0, 15.8038664, 7443.31494, 89683.9844, 304795.938, 528760.125, 4303920}},
{10, 0.6901, {313.602661, 12.9589806, 2.76040006, 1.16625726, 0.759703755, 0},
 {11.2494116, 139.565872, 1302.66797, 9012.4375, 40174.4141, 0.349638999, 625.133118, 10294.0283, 34029.5703, 51012.5156, 0}},
{10, 0.5771, {262.251984, 10.8370209, 2.30839992, 0.975289166, 0.635306537, 0},
 {5.06145191, 106.100014, 1455.52429, 11283.7832, 39719.5938, 0.139374569, 945.073547, 16682.4863, 40277.2813, 44989.4844, 0}},
{10, 0.4943, {224.625122, 9.28216839, 1.97720003, 0.83535856, 0.5441553, 0},
 {5.28274965, 144.459274, 2445.39209, 23822.7559, 125401.656, 0.170843989, 2215.44385, 48694.0586, 151871.141, 213374.094, 0}},
{10, 0.3114, {141.509735, 5.84759665, 1.24559999, 0.526260674, 0.342807919, 0},
 {0.391450971, 27.2448635, 1140.29614, 23494.5469, 177507.766, 0.03553931415, 4019.15039, 132922.703, 214696.141, 117752.625, 0}},
{10, 0.2949, {134.011627, 5.53775311, 1.1796, 0.498375952, 0.324643731, 0},
 {0.27195853, 27.4967785, 1421.43127, 32464.7871, 266095.906, 0.02719595842, 6175.17725, 212257.125, 346230.438, 172286.266, 0}},
{10, 0.0634, {28.8109093, 1.19055116, 0.253600001, 0.107144915, 0.06979455054, 0},
 {0.001736157807, 0.549801707, 90.500061, 7081.02197, 250507.734, 0.377159357, 152774.266, 2498325.5, 461861.563, 300963.813, 0}},
{10, 0.0605, {27.4930611, 1.13609385, 0.241999999, 0.102243967, 0.06660205126, 0},
 {0.002016902668, 0.651284337, 110.502411, 8871.02246, 320563.281, 0.563207269, 223356.625, 3402945.75, 615073.625, 444890, 0}},
{10, 0.0952, {43.2618065, 1.78770459, 0.380800009, 0.160886377, 0.104801908, 0},
 {2.23526239, 27.1868801, 254.041412, 1844.41895, 10506.6924, 9.46362782, 4318.91797, 37110.8359, 63932.7344, 62294.8242, 0}},
{10, 0.063, {28.629137, 1.1830399, 0.252000004, 0.106468923, 0.06935420632, 0},
 {0.877922356, 17.7858124, 240.620728, 1940.79102, 9193.33105, 14.8904295, 7501.68311, 37936.2227, 423219.875, 1749258.5, 0}},
{10, 0.0505, {22.9487514, 0.94830972, 0.201999992, 0.08534413576, 0.05559344962, 0},
 {0.88378793, 23.2416325, 385.564301, 3824.12793, 23457.9824, 35.0851936, 25319.3652, 146989.172, 863663.25, 3837076.75, 0}},
{10, 0.0080627, {3.66395903, 0.151405513, 0.03225097805, 0.01362590026, 0.008875955828, 0},
 {0.0372838527, 2.32002163, 90.3582153, 1808.81775, 14857.0703, 1107.3114, 77534.5313, 15999634, 11557034, 7199740.5, 0}},
{10, 0.0068546, {3.11492109, 0.128717676, 0.027418226, 0.01158408355, 0.007545909379, 0},
 {0.01240959018, 1.10907686, 53.0882607, 1176.70337, 10343.2783, 1072.80054, 94213.7109, 12927625, 9863556, 6704388, 0}},

// Z=78,  21 orbitals
{11, 78.3948, {1671.17188, 339.716522, 156.789597, 101.912727, 82.2533112, 78.4731979},
 {2152.86523, 0, 0, 0, 0, 0.846432447, 46.9265976, 373.479858, 1163.34302, 2020.09045, 2262.84082}},
{11, 13.8799, {295.883118, 60.1472473, 27.7598, 18.0437794, 14.5630541, 13.8937798},
 {211.555557, 2448.26978, 0, 0, 0, 7.68703556, 416.657501, 2262.49536, 5206.62012, 7630.89746, 8209.95313}},
{11, 13.2726, {282.937073, 57.5155678, 26.5452003, 17.2542934, 13.9258633, 13.2858725},
 {103.003242, 2586.15039, 0, 0, 0, 2.05388427, 278.498779, 2628.65894, 8500.87891, 14814.5088, 16786.3652}},
{11, 11.5637, {246.507797, 50.1102104, 23.1273994, 15.0327339, 12.1328535, 11.575264},
 {102.874779, 3410.6814, 0, 0, 0, 2.13092494, 476.970398, 5252.69678, 18371.7715, 33486.5664, 38127.0547}},
{11, 3.296, {70.262085, 14.2829065, 6.59200001, 4.28477812, 3.45822549, 3.2992959},
 {46.5231895, 569.037903, 4954.47266, 0, 0, 63.4835701, 2041.33167, 8343.52637, 16352.0498, 21835.4512, 23311.2852}},
{11, 3.0265, {64.5170593, 13.1150541, 6.05299997, 3.93443012, 3.17546129, 3.02952647},
 {22.5318508, 479.917175, 6448.80908, 0, 0, 42.172699, 2761.56201, 13670.0547, 27461.3164, 36029.4453, 40899.0625}},
{11, 2.6454, {56.3929977, 11.4635925, 5.29080009, 3.43900251, 2.77560377, 2.6480453},
 {23.5708256, 660.746826, 11264.8525, 97432.0156, 0, 71.4618454, 6167.74268, 34803.2539, 77696.25, 109723.523, 123099.289}},
{11, 2.2019, {46.9387398, 9.54172707, 4.40380001, 2.86245537, 2.31027508, 2.2041018},
 {1.72388852, 129.156601, 6258.42529, 165190.234, 0, 14.7782784, 4969.396, 56307.0273, 187319.625, 322146.5, 2506648.5}},
{11, 2.1216, {45.2269554, 9.19375515, 4.2432003, 2.75806618, 2.22602296, 2.1237216},
 {1.19665027, 130.708649, 7836.30566, 230389.578, 0, 14.6959858, 7008.56201, 85020.4922, 290270.938, 504087, 4003906.75}},
{10, 0.722, {328.098999, 13.558012, 2.88800001, 1.22016764, 0.794821203, 0},
 {11.8233175, 145.43959, 1346.93054, 9208.60059, 40177.9961, 0.332287967, 592.279236, 9783.34766, 32463.5078, 48631.1563, 0}},
{10, 0.6092, {276.839203, 11.4398079, 2.4368, 1.02953756, 0.670644164, 0},
 {5.50269032, 113.329018, 1525.03955, 11554.332, 38976.4805, 0.130480543, 880.564514, 15729.3545, 38386.2813, 43067.9727, 0}},
{10, 0.519, {235.849548, 9.74599457, 2.07599998, 0.877101123, 0.571346521, 0},
 {5.69251537, 153.742264, 2568.58374, 24683.3379, 127952.664, 0.149947762, 2075.37939, 46410.0625, 147089.547, 209243.25, 0}},
{10, 0.3308, {150.325684, 6.21189785, 1.32319999, 0.559046328, 0.364164621, 0},
 {0.438516498, 30.0672398, 1235.32422, 24872.5625, 181224.297, 0.03104903921, 3631.53516, 125298, 218161.641, 112476.07, 0}},
{10, 0.3133, {142.373154, 5.88327599, 1.25320005, 0.529471636, 0.344899565, 0},
 {0.30346337, 30.2908669, 1539.00562, 34389.4453, 272764.5, 0.02241119184, 5549.0332, 200042.172, 352990.563, 165764, 0}},
{10, 0.0743, {33.7642059, 1.3952359, 0.297199994, 0.125565737, 0.08179392666, 0},
 {0.002073757816, 0.648876309, 105.026894, 8065.94189, 276766.594, 0.201935664, 105967.828, 2356522, 450271.25, 288861.844, 0}},
{10, 0.0711, {32.3100243, 1.335145, 0.284399986, 0.120157786, 0.07827117294, 0},
 {0.00240224693, 0.767768264, 128.175995, 10105.0068, 354410.719, 0.297191739, 154104.531, 3225300.75, 597038.5, 432386.906, 0}},
{10, 0.1017, {46.2156067, 1.90976429, 0.406800002, 0.17187126, 0.111957498, 0},
 {2.39787269, 28.9441452, 268.689026, 1934.86853, 10956.9707, 8.76206303, 4065.82104, 35385.5, 60787.6797, 60415.582, 0}},
{10, 0.0653, {29.6743279, 1.22623014, 0.261200011, 0.110355884, 0.07188618183, 0},
 {0.977953315, 19.4883652, 258.886353, 2048.1499, 9545.51953, 14.9105349, 7464.91455, 37688.8164, 426000.906, 1793904.38, 0}},
{10, 0.0517, {23.4940701, 0.970843792, 0.206799999, 0.08737211674, 0.05691448227, 0},
 {0.97381717, 25.3129902, 414.666168, 4062.57349, 24760.0371, 35.8129692, 25837.0449, 149480.656, 885650.375, 4116500.5, 0}},
{10, 0.00744, {3.38092208, 0.139709607, 0.02975962311, 0.0125733139, 0.008190297522, 0},
 {0.04277142882, 2.60824323, 99.3904114, 1943.15613, 15571.6396, 1458.41064, 130757.008, 15573292, 10632751, 7138561, 0}},
{10, 0.006125, {2.78356099, 0.115024894, 0.02450151928, 0.01035178732, 0.006743188016, 0},
 {0.02346534841, 2.04996371, 95.8907242, 2077.51929, 17845.752, 2485.0271, 306832.75, 19785610, 15738859, 11531171, 0}},

// Z=79,  21 orbitals
{10, 80.7249, {372.712921, 168.043701, 114.162247, 92.0403137, 82.6876373, 0},
 {0, 0, 0, 0, 0, 38.6293297, 324.666565, 902.221741, 1572.68774, 2059.21899, 0}},
{11, 14.3528, {305.964111, 62.1965141, 28.7056007, 18.6585464, 15.0592299, 14.3671532},
 {220.632462, 2516.19824, 0, 0, 0, 7.43844557, 400.267578, 2173.69653, 5003.65381, 7338.01123, 7891.47705}},
{11, 13.7336, {292.764404, 59.5132713, 27.4672012, 17.85359, 14.4095535, 13.7473335},
 {111.160133, 2747.67578, 0, 0, 0, 2.01955223, 270.382141, 2538.76636, 8191.92285, 14277.0967, 16157.168}},
{11, 11.9187, {254.07547, 51.648571, 23.8374004, 15.4942312, 12.5053253, 11.9306183},
 {109.872185, 3607.06885, 0, 0, 0, 2.06537437, 460.459686, 5069.52979, 17732.0488, 32306.1172, 36908.1055}},
{11, 3.4249, {73.009903, 14.8414831, 6.84980011, 4.45234728, 3.59346986, 3.42832494},
 {48.6085091, 588.897339, 5069.56836, 0, 0, 60.4823265, 1949.15015, 7972.59766, 15626.1211, 20858.4121, 22135.5469}},
{11, 3.1478, {67.1028519, 13.6406965, 6.29559994, 4.09211922, 3.30273151, 3.15094781},
 {24.3186417, 508.801605, 6683.98535, 0, 0, 40.5689774, 2639.74976, 13064.5391, 26246.9785, 34417.5859, 38574.1055}},
{11, 2.743, {58.4735756, 11.8865337, 5.48600006, 3.56588197, 2.87800741, 2.74574304},
 {25.2148151, 698.127991, 11731.1084, 99664.2188, 0, 68.0884933, 5900.4624, 33435.5117, 74900.0781, 106048.859, 113809.922}},
{11, 2.2911, {48.8402519, 9.92826748, 4.58220005, 2.97841501, 2.40386558, 2.29339123},
 {1.9087745, 141.108276, 6733.83203, 174732, 0, 13.9222813, 4690.4751, 53360.8867, 178147.047, 308140.906, 2278065.75}},
{11, 2.2057, {47.0197487, 9.55819416, 4.41139984, 2.86739564, 2.31426215, 2.20790577},
 {1.3189851, 142.470169, 8423.35449, 243802.313, 0, 13.6534481, 6592.4873, 80502.7813, 276057.313, 481926.281, 3517298.75}},
{10, 0.7588, {344.822052, 14.2490578, 3.03519988, 1.282359, 0.83533287, 0},
 {12.4245834, 151.485703, 1391.80408, 9401.99512, 40092.7461, 0.312762827, 555.852295, 9229.88086, 30793.0605, 46336.4688, 0}},
{10, 0.6437, {292.517059, 12.0876627, 2.57480001, 1.08784199, 0.708623827, 0},
 {5.97069407, 120.86618, 1595.74219, 11811.167, 38153.6992, 0.121744417, 818.142944, 14798.9648, 36558.0859, 41252.3125, 0}},
{10, 0.5454, {247.846527, 10.241745, 2.18160009, 0.92171669, 0.600409269, 0},
 {6.13133955, 163.471375, 2695.16089, 25549.3164, 130381.32, 0.137875795, 1938.36426, 44127.1914, 142208.25, 204849.344, 0}},
{10, 0.352, {159.959625, 6.61000013, 1.40799999, 0.594873965, 0.387502849, 0},
 {0.490304232, 33.1184692, 1335.7301, 26275.9258, 184691.813, 0.02689767443, 3259.02222, 117527.594, 221379.563, 109208.75, 0}},
{10, 0.3339, {151.734421, 6.27011108, 1.33560002, 0.564285278, 0.367577285, 0},
 {0.337982684, 33.3066254, 1663.35889, 36360.0391, 279083.813, 0.0189363528, 4930.04053, 187187.797, 359200.031, 162359.984, 0}},
{10, 0.0864, {39.2628174, 1.62245464, 0.345600009, 0.146014526, 0.09511433542, 0},
 {0.002464715159, 0.761895299, 121.243187, 9130.99805, 303588.688, 0.108959161, 73797.3906, 2183817.25, 473625.219, 254614.594, 0}},
{10, 0.0828, {37.6268654, 1.55485237, 0.331200004, 0.139930591, 0.09115123749, 0},
 {0.002847113181, 0.900383472, 147.877197, 11437.7539, 388956.375, 0.162117675, 107021.711, 3008845.5, 623508.438, 379320.5, 0}},
{10, 0.1078, {48.9876328, 2.02431273, 0.431199998, 0.182180166, 0.118672751, 0},
 {2.57219863, 30.8129902, 284.109406, 2028.68372, 11406.4199, 8.25963116, 3872.29126, 34036.7461, 59422.1289, 61084.4102, 0}},
{10, 0.0717, {32.5826836, 1.34641194, 0.286799997, 0.121171772, 0.07893168926, 0},
 {1.09061289, 21.3785706, 278.904144, 2164.76709, 9921.41602, 12.929822, 6920.77734, 34738.8086, 363105.688, 1527340.13, 0}},
{10, 0.0537, {24.4029312, 1.00840068, 0.2148, 0.09075208008, 0.05911620334, 0},
 {1.0768342, 27.6668377, 447.594238, 4332.26416, 26230.8809, 35.2958069, 25916.666, 151241.641, 843603, 3729802.5, 0}},
{10, 0.008308, {3.77558589, 0.156018272, 0.033233542, 0.01404102985, 0.009146372788, 0},
 {0.05085494742, 3.07416463, 115.277191, 2201.76416, 17229.4004, 1262.50232, 93408.2422, 14953335, 9368696, 6231522, 0}},
{10, 0.00679, {3.08572841, 0.127511337, 0.02716126479, 0.01147551835, 0.00747519033, 0},
 {0.03338646144, 2.90079951, 133.737671, 2833.18213, 23805.293, 2647.1228, 271471.438, 22768304, 16474063, 11680687, 0}},

// Z=80,  22 orbitals
{10, 83.1023, {383.689545, 172.992691, 117.524399, 94.7509613, 85.1228409, 0},
 {0, 0, 0, 0, 0, 37.6708527, 314.210602, 871.379089, 1518.18823, 1988.59009, 0}},
{11, 14.8393, {316.335022, 64.304718, 29.6786003, 19.2909927, 15.5696745, 14.8541393},
 {229.523453, 2583.53491, 0, 0, 0, 7.19898653, 384.036194, 2088.10229, 4807.28418, 7049.54932, 7582.37207}},
{11, 14.2087, {302.892303, 61.5720711, 28.4174004, 18.4712181, 14.9080381, 14.222909},
 {119.987572, 2916.03564, 0, 0, 0, 1.98558009, 262.480316, 2451.86548, 7892.88232, 13737.6699, 15523.6602}},
{11, 12.2839, {261.860565, 53.2311287, 24.5678005, 15.9689894, 12.8885002, 12.2961836},
 {117.239082, 3811.51001, 0, 0, 0, 2.0010128, 444.345276, 4891.01221, 17112.7383, 31212.5625, 35622.8555}},
{11, 3.5616, {75.9239883, 15.4338598, 7.12319994, 4.63005686, 3.73689818, 3.56516171},
 {50.7448845, 608.994934, 5182.5498, 0, 0, 57.488224, 1857.45996, 7605.57227, 14912.4912, 19887.8281, 21237.7188}},
{11, 3.2785, {69.8890381, 14.2070723, 6.55700016, 4.26202822, 3.4398644, 3.28177857},
 {26.2121353, 538.837524, 6920.41309, 0, 0, 38.6270218, 2514.95776, 12453.917, 25036.1055, 32819.918, 37971.5625}},
{11, 2.8471, {60.6927147, 12.3376408, 5.69420004, 3.70121121, 2.98723125, 2.84994698},
 {26.9522266, 736.975159, 12208.249, 101485.32, 0, 64.6448517, 5628.21045, 32045.2227, 72048.1563, 102182.398, 110843}},
{11, 2.3849, {50.8398209, 10.3347406, 4.76979971, 3.10035419, 2.50228214, 2.38728476},
 {2.10924292, 153.90596, 7234.72949, 184603.578, 0, 13.075983, 4416.37061, 50452.8555, 169043.188, 294712.563, 2243732.75}},
{11, 2.2949, {48.921257, 9.94473362, 4.58979988, 2.98335481, 2.40785241, 2.29719496},
 {1.45195472, 155.072311, 9043.05273, 257820.844, 0, 12.6301603, 6177.84326, 75985.2266, 261800.734, 461422.781, 3343575}},
{10, 0.8003, {363.680939, 15.0283613, 3.20120001, 1.35249329, 0.881018579, 0},
 {13.0368004, 157.633972, 1437.08508, 9591.41797, 39842.2656, 0.291843653, 517.6521, 8656.07617, 29067.1582, 44198.2227, 0}},
{10, 0.6769, {307.604156, 12.7111053, 2.70759988, 1.14394939, 0.745172381, 0},
 {6.46967506, 128.708359, 1666.59021, 12044.709, 37353.2266, 0.1152713, 767.519165, 13999.4492, 34956.625, 39751.6523, 0}},
{10, 0.571, {259.47995, 10.7224722, 2.28399992, 0.964980245, 0.628591299, 0},
 {6.59015846, 173.532974, 2823.66895, 26396.5586, 132724.922, 0.128319412, 1824.18433, 42133.4844, 137869.641, 200981.234, 0}},
{10, 0.3783, {171.911148, 7.10387278, 1.51320004, 0.639320552, 0.416455477, 0},
 {0.547235131, 36.4216881, 1442.55212, 27736.9629, 187947.438, 0.02231793851, 2837.85498, 108309.156, 223219.719, 109609.313, 0}},
{10, 0.3598, {163.504181, 6.75647211, 1.43920004, 0.60805589, 0.396089584, 0},
 {0.375783116, 36.5666656, 1795.7998, 38421.8203, 285280.281, 0.01529175881, 4230.52148, 171749.438, 362657.75, 165932.516, 0}},
{10, 0.1022, {46.4428215, 1.91915357, 0.408800006, 0.17271626, 0.112507932, 0},
 {0.002916152822, 0.890594184, 139.378189, 10294.0381, 331971.719, 0.05432848632, 47886.6445, 1922911.63, 565484.563, 208792.016, 0}},
{10, 0.0985, {44.7614288, 1.84967339, 0.393999994, 0.166463315, 0.108434752, 0},
 {0.003359529423, 1.05118167, 169.904404, 12895.752, 425758.281, 0.07615496218, 68313.9297, 2655413, 744125.375, 302197.031, 0}},
{10, 0.1203, {54.6680183, 2.25904274, 0.48120001, 0.203304946, 0.132433504, 0},
 {2.75942302, 32.8090019, 300.530212, 2128.75781, 11891.0273, 6.72155762, 3397.78882, 30975.9766, 57560.1719, 60507.2617, 0}},
{10, 0.0805, {36.5816765, 1.51166201, 0.321999997, 0.136043623, 0.08861926198, 0},
 {1.21736455, 23.4722557, 300.726715, 2289.89429, 10319.0693, 10.6393175, 6229.20117, 31394.5254, 282175.094, 1138393.13, 0}},
{10, 0.0576, {26.175211, 1.08163643, 0.230399996, 0.09734302014, 0.06340955943, 0},
 {1.19494247, 30.3488846, 484.944244, 4637.80176, 27901.2949, 32.02351, 24897.125, 149795.922, 723183.5, 2809717.75, 0}},
{10, 0.0064, {2.90835667, 0.120181821, 0.02559999935, 0.01081589051, 0.0070455065, 0},
 {0.06201901287, 3.75049901, 138.964645, 2591.22046, 19821.168, 2782.53613, 354706.813, 17363448, 9453465, 8431507, 0}},
{10, 0.0064, {2.90835667, 0.120181821, 0.02559999935, 0.01081589051, 0.0070455065, 0},
 {0.04085951298, 3.58066297, 163.793716, 3392.00171, 27899.5762, 3650.28955, 417583.906, 25729812, 15345441, 12314197, 0}},
{10, 0.007714, {3.50529981, 0.144849256, 0.03085442446, 0.01303586271, 0.008491603658, 0},
 {0.243561655, 2.62276173, 22.9374752, 163.816376, 958.838562, 125.570839, 11494.9297, 45395.9727, 1154198, 5944232.5, 0}},

// Z=81,  22 orbitals
{10, 85.5304, {394.900269, 178.047226, 120.958252, 97.5194168, 87.6099777, 0},
 {0, 0, 0, 0, 0, 36.7505112, 304.159332, 841.720154, 1465.02539, 1917.29431, 0}},
{11, 15.3467, {327.151459, 66.5034866, 30.6934013, 19.9506092, 16.1020489, 15.3620472},
 {238.65564, 2650.93481, 0, 0, 0, 6.96272993, 368.510254, 2004.36279, 4616.48633, 6773.05127, 7278.79541}},
{11, 14.6979, {313.32074, 63.6919708, 29.3957996, 19.107172, 15.4213142, 14.7125969},
 {129.22554, 3091.14038, 0, 0, 0, 1.95263731, 254.904739, 2368.2207, 7605.98242, 13237.04, 14920.7422}},
{11, 12.6575, {269.824768, 54.85009, 25.3150005, 16.4546661, 13.280489, 12.6701574},
 {124.958359, 4023.65405, 0, 0, 0, 1.93868458, 428.747192, 4718.60156, 16512.6172, 30123.8984, 34298.8086}},
{11, 3.7041, {78.9617157, 16.0513706, 7.40819979, 4.81530571, 3.88641191, 3.7078042},
 {52.944416, 629.299194, 5292.59961, 0, 0, 54.6170578, 1768.81311, 7250.07129, 14218.6465, 19002.5254, 20241.8477}},
{11, 3.4157, {72.8137817, 14.8016167, 6.83139992, 4.44038773, 3.58381724, 3.41911578},
 {28.2292805, 570.029846, 7155.31543, 0, 0, 36.9420433, 2393.39551, 11859.7744, 23856.9121, 31273.2285, 36988.2617}},
{11, 2.9566, {63.0269699, 12.812149, 5.9131999, 3.84356046, 3.10212064, 2.95955658},
 {28.7684555, 777.151306, 12694.6758, 107799.883, 0, 61.2213783, 5359.0459, 30669.0098, 69222.7969, 98373.4688, 113081.57}},
{11, 2.4851, {52.9758263, 10.7689486, 4.97020006, 3.23061371, 2.60741401, 2.48758507},
 {2.32726169, 167.645569, 7766.12061, 195036.063, 0, 12.1542692, 4141.6377, 47547.7813, 159969.766, 281337.531, 2133384.75}},
{11, 2.3893, {50.9336205, 10.3538074, 4.77860022, 3.10607433, 2.50689888, 2.3916893},
 {1.5952847, 168.536713, 9697.68848, 272517.844, 0, 11.6307917, 5770.26563, 71525.0859, 247680.75, 440288.219, 3370416}},
{10, 0.8455, {384.221191, 15.8771458, 3.38199997, 1.42888057, 0.93077749, 0},
 {13.6683912, 163.903152, 1482.60254, 9774.77051, 39569.7383, 0.271508664, 480.142029, 8090.28613, 27381.377, 41939.8477, 0}},
{10, 0.7213, {327.780884, 13.5448675, 2.88520002, 1.21898472, 0.794050574, 0},
 {7.00394964, 136.961182, 1740.10291, 12279.1162, 36269.2188, 0.104858741, 698.989746, 12999.334, 33023.7305, 37786.5898, 0}},
{10, 0.609, {276.748322, 11.4360514, 2.43600011, 1.0291996, 0.670423985, 0},
 {7.07736444, 184.14679, 2959.82471, 27318.2383, 135199.766, 0.112245992, 1642.63049, 39087.8398, 131048.93, 192620.031, 0}},
{10, 0.4066, {184.771545, 7.63530159, 1.62639999, 0.687147021, 0.447609842, 0},
 {0.609680235, 39.9801559, 1555.03601, 29221.6465, 190981.688, 0.0184787605, 2464.54248, 99482.8906, 224271.359, 117338.148, 0}},
{10, 0.3862, {175.50116, 7.25222206, 1.54480004, 0.652671397, 0.425152272, 0},
 {0.41701743, 40.0667915, 1934.58154, 40495.5039, 291232.219, 0.01258266717, 3661.52075, 157913.375, 365015.281, 181976.344, 0}},
{10, 0.1228, {55.8040962, 2.30598879, 0.4912, 0.207529902, 0.135185659, 0},
 {0.003436008003, 1.03688824, 159.680298, 11575.1328, 362794.156, 0.02389349788, 28986.375, 1593842.88, 749133.063, 184318.234, 0}},
{10, 0.1185, {53.8500443, 2.22524166, 0.474000007, 0.200262979, 0.130451962, 0},
 {0.003947643563, 1.22219718, 194.503113, 14495.2129, 465411.188, 0.03316638619, 41200.2734, 2209708.25, 983727.5, 257940.484, 0}},
{10, 0.1363, {61.9389153, 2.55949736, 0.54520005, 0.230344683, 0.150047272, 0},
 {2.95544767, 34.8801651, 317.410248, 2230.55225, 12376.1113, 5.45351887, 2901.15405, 27592.9336, 54876.1563, 56690.043, 0}},
{10, 0.0996, {45.2613029, 1.87032962, 0.398400009, 0.168322295, 0.109645694, 0},
 {1.35342801, 25.7043285, 323.76413, 2419.9436, 10712.0381, 6.59764719, 4895.47607, 25824.5352, 174385.094, 777126.375, 0}},
{10, 0.0754, {34.2640762, 1.41589212, 0.301600009, 0.127424717, 0.08300486952, 0},
 {1.32109165, 33.2294579, 525.348633, 4973.021, 29784.1953, 16.8682785, 17707.2637, 122690.359, 433154.469, 1640877.13, 0}},
{10, 0.0153, {6.95279026, 0.287309676, 0.06120000035, 0.02585673891, 0.01684316434, 0},
 {0.07489055395, 4.53407192, 166.225403, 3031.78906, 22667.2891, 343.807739, 39589.9375, 6899228, 9519483, 4303452, 0}},
{10, 0.0131, {5.95304298, 0.245997176, 0.05240000039, 0.02213877626, 0.01442127116, 0},
 {0.04969504476, 4.37703419, 198.447693, 4021.89771, 32400.9922, 667.043518, 60946.2852, 13976118, 14531094, 7702536.5, 0}},
{10, 0.009665, {4.39199686, 0.181490168, 0.03865932673, 0.0163334012, 0.01063963026, 0},
 {0.322027743, 3.49257398, 30.5639877, 216.844971, 1261.86975, 112.20826, 11709.5225, 38995.6016, 972106.063, 5652065, 0}},

// Z=82,  23 orbitals
{10, 88.005, {406.323364, 183.19751, 124.457153, 100.340317, 90.1442337, 0},
 {0, 0, 0, 0, 0, 35.8716545, 294.5242, 813.227112, 1414.63635, 1851.9552, 0}},
{11, 15.8608, {338.110718, 68.7312927, 31.7215996, 20.6189365, 16.6414528, 15.8766613},
 {247.877075, 2717.17554, 0, 0, 0, 6.74366903, 353.930481, 1925.35803, 4434.95605, 6505.57031, 6997.19141}},
{11, 15.2, {324.0242, 65.867775, 30.4000015, 19.759901, 15.9481287, 15.2152004},
 {139.015854, 3272.97607, 0, 0, 0, 1.92084634, 247.397278, 2288.31763, 7330.9502, 12735.9219, 14381.0098}},
{11, 13.0352, {277.876343, 56.4868164, 26.0704002, 16.9456749, 13.6767788, 13.0482349},
 {133.052155, 4242.77148, 0, 0, 0, 1.88079166, 414.137512, 4556.13574, 15947.2148, 29116.4727, 33094.918}},
{11, 3.8507, {82.0868378, 16.6866474, 7.7013998, 5.00588465, 4.04022741, 3.8545506},
 {55.1786766, 649.716736, 5399.47363, 0, 0, 51.9043045, 1684.87061, 6912.60889, 13559.9385, 18136.0488, 19197.5273}},
{11, 3.5542, {75.7662354, 15.4017925, 7.10839987, 4.62043667, 3.72913408, 3.55775428},
 {30.3572769, 602.17627, 7385.01611, 0, 0, 35.4397774, 2282.71826, 11311.4248, 22763.2051, 29838.457, 35104.1133}},
{11, 3.0664, {65.3676147, 13.2879572, 6.1328001, 3.98629975, 3.21732497, 3.06946635},
 {30.6800423, 818.620605, 13184.6084, 0, 0, 58.1819344, 5114.12451, 29399.6211, 66591.6953, 94860.8672, 109804.336}},
{11, 2.5856, {55.1182213, 11.2044554, 5.1711998, 3.36126304, 2.71286035, 2.58818555},
 {2.56441355, 182.313431, 8320.59863, 205728.031, 0, 11.4095411, 3897.81689, 44929.0664, 151687.359, 268430.313, 2024232.25}},
{11, 2.484, {52.9523735, 10.7641811, 4.96799994, 3.22918367, 2.60625982, 2.48648405},
 {1.75049591, 182.878403, 10380.8555, 287412.031, 0, 10.704443, 5407.70557, 67492.0313, 234746.563, 419983.875, 3163863.5}},
{10, 0.8936, {406.079315, 16.7803879, 3.57439995, 1.51016879, 0.983728826, 0},
 {14.3185396, 170.298172, 1528.43176, 9952.65332, 39227.0586, 0.25274545, 444.717194, 7553.37305, 25753.1699, 39676.5117, 0}},
{10, 0.7639, {347.139648, 14.3448277, 3.05559993, 1.29097795, 0.84094727, 0},
 {7.57276058, 145.542236, 1813.84351, 12491.1357, 35186.3164, 0.09720915556, 644.845276, 12162.3096, 31334.0801, 36176.0547, 0}},
{10, 0.6445, {292.880615, 12.102685, 2.57800007, 1.08919406, 0.709504545, 0},
 {7.59156179, 195.160721, 3098.26685, 28227.4707, 137553, 0.100781851, 1504.72974, 36656.7031, 125312.609, 186195.719, 0}},
{10, 0.4352, {197.768265, 8.17236423, 1.74080002, 0.735480547, 0.479094446, 0},
 {0.678053319, 43.8103218, 1673.45789, 30730.8574, 193726.781, 0.01556147821, 2162.63745, 91739.6797, 223297.859, 131120.797, 0}},
{10, 0.4129, {187.63446, 7.75360537, 1.6516, 0.697793961, 0.454545259, 0},
 {0.461993575, 43.8263206, 2080.62988, 42607.7734, 296820.406, 0.01058416534, 3199.59058, 145721.188, 364428.625, 208738.016, 0}},
{10, 0.1429, {64.9381561, 2.68343496, 0.57160002, 0.24149856, 0.157312959, 0},
 {0.004032428842, 1.20229006, 182.168381, 12949.874, 394023.594, 0.01262279321, 19181.6973, 1324029.13, 962867.563, 169274.266, 0}},
{10, 0.1381, {62.7568893, 2.59329844, 0.552399993, 0.233386651, 0.152028829, 0},
 {0.004620685242, 1.41520584, 221.727631, 16211.8027, 505706.313, 0.01737721264, 27119.9004, 1836849.5, 1265559, 233725.141, 0}},
{10, 0.1473, {66.937645, 2.76605964, 0.58919996, 0.248934478, 0.162156731, 0},
 {3.16178107, 37.0364761, 334.694977, 2332.00024, 12834.0713, 4.73887587, 2667.75488, 25855.5645, 53445.3672, 55956.082, 0}},
{10, 0.1048, {47.6243439, 1.9679774, 0.419200003, 0.17711021, 0.115370169, 0},
 {1.50196934, 28.0811501, 347.392487, 2546.10205, 11081.9209, 6.37304354, 4764.15967, 25354.5469, 153902.828, 663843.5, 0}},
{10, 0.086, {39.0810432, 1.61494327, 0.344000012, 0.145338535, 0.0946739912, 0},
 {1.45888615, 36.328846, 567.993652, 5318.38916, 31683.584, 12.5284662, 15330.2109, 114308.805, 348896.344, 1225272, 0}},
{10, 0.0218, {9.90659046, 0.409369349, 0.08720000088, 0.03684162721, 0.02399875596, 0},
 {0.08954252303, 5.39927149, 195.322311, 3483.39063, 25483.1211, 141.210983, 48489.7266, 2526930, 13926979, 3251424.25, 0}},
{10, 0.0192, {8.72507, 0.360545486, 0.07680000365, 0.03244767338, 0.02113652043, 0},
 {0.05954287946, 5.24107456, 234.869537, 4658.67432, 36794.3047, 249.896179, 77629.7344, 4999401, 19581046, 5958861, 0}},
{10, 0.0117, {5.31246948, 0.219526812, 0.04676153511, 0.01975654811, 0.01286948007, 0},
 {0.393189073, 4.27342463, 37.3253899, 262.932922, 1523.90234, 97.195488, 11322.2471, 38387.7773, 706199, 4520768, 0}},
{10, 0.004912, {2.23201036, 0.0922332108, 0.01964665018, 0.008300625719, 0.005407054443, 0},
 {0.09778127819, 1.60369289, 18.6227684, 137.44902, 629.868591, 216.5383, 37120.3281, 2332257, 14477358, 33478308, 0}},

// Z=83,  23 orbitals
{10, 90.526, {417.964874, 188.446274, 128.022949, 103.215149, 92.7269363, 0},
 {0, 0, 0, 0, 0, 35.0313034, 285.287354, 785.885315, 1365.58289, 1786.26929, 0}},
{11, 16.3875, {349.338593, 71.0136948, 32.7750015, 21.3036423, 17.1940765, 16.4038887},
 {257.295105, 2782.74634, 0, 0, 0, 6.53682518, 340.036987, 1849.79492, 4261.91357, 6253.20752, 6720.55078}},
{11, 15.7111, {334.919525, 68.0825806, 31.4222012, 20.4243279, 16.4843845, 15.7268114},
 {149.416214, 3460.82422, 0, 0, 0, 1.89219773, 240.637894, 2213.10327, 7071.67627, 12280.5059, 13831.9316}},
{11, 13.4186, {286.049408, 58.1482468, 26.8372002, 17.4440918, 14.0790491, 13.4320183},
 {141.511261, 4469.10352, 0, 0, 0, 1.82588029, 400.200531, 4401.35645, 15404.8818, 28119.4688, 32014.1934}},
{11, 3.9991, {85.2503357, 17.3297253, 7.99819994, 5.19880342, 4.19593143, 4.00309896},
 {57.4725113, 670.225647, 5501.37598, 0, 0, 49.4341507, 1606.95117, 6596.7876, 12940.4873, 17344.4629, 18283.0898}},
{11, 3.6963, {78.795433, 16.0175686, 7.39260006, 4.80516577, 3.87822795, 3.69999623},
 {32.609642, 635.363953, 7610.20703, 0, 0, 34.043457, 2178.88477, 10792.9971, 21720.832, 28461.8203, 34004.9414}},
{11, 3.1769, {67.7231903, 13.766799, 6.35379982, 4.12994909, 3.33326364, 3.18007684},
 {32.6707001, 861.232056, 13676.707, 0, 0, 55.4122543, 4888.97021, 28219.4629, 64132.8477, 91479.2813, 105628.164}},
{11, 2.6876, {57.2925911, 11.6464624, 5.37519979, 3.49386215, 2.81988072, 2.69028759},
 {2.82040119, 197.929398, 8899.62695, 216495, 0, 10.7359819, 3675.45996, 42511.4453, 143958.656, 255534.563, 1912356.63}},
{11, 2.5796, {54.9903145, 11.1784544, 5.15919971, 3.35346293, 2.70656514, 2.58217955},
 {1.91708291, 198.100357, 11092.5488, 302747.5, 0, 9.93102837, 5079.45801, 63789.8672, 222771.313, 401488, 2956409}},
{10, 0.9382, {426.346924, 17.6179047, 3.75279999, 1.58554196, 1.03282726, 0},
 {14.9950428, 176.798233, 1573.41577, 10112.4609, 38650.5938, 0.239857987, 417.083588, 7116.51025, 24382.7129, 37838.4063, 0}},
{10, 0.8053, {365.953094, 15.1222534, 3.22119999, 1.3609432, 0.886522889, 0},
 {8.18303871, 154.478928, 1887.6261, 12679.8613, 34116.4102, 0.0915286243, 600.691833, 11441.7998, 29811.3867, 34709.4336, 0}},
{10, 0.6789, {308.513031, 12.7486629, 2.71560001, 1.14732945, 0.747374117, 0},
 {8.13323975, 206.5867, 3239.14526, 29127.998, 139812.109, 0.09195916355, 1392.10449, 34588.3672, 120337.891, 181400.359, 0}},
{10, 0.4636, {210.674103, 8.70567131, 1.85440004, 0.783476114, 0.51035887, 0},
 {0.752776563, 47.9184341, 1797.39246, 32245.8066, 196220.234, 0.01335039455, 1920.67053, 85025.8672, 221361.969, 148493.578, 0}},
{10, 0.434, {199.949524, 8.26250076, 1.75999999, 0.743592501, 0.484378576, 0},
 {0.510924935, 47.8553734, 2233.79736, 44745.0469, 302076.063, 0.00904890988, 2817.76147, 134831.344, 361423.063, 242470.063, 0}},
{10, 0.1619, {73.5723343, 3.04022455, 0.647599995, 0.273608238, 0.178229287, 0},
 {0.004714301322, 1.38842094, 206.898804, 14403.4854, 424836.25, 0.00736968359, 13740.0723, 1117142.5, 1183275.63, 157517.953, 0}},
{10, 0.1574, {71.5273972, 2.95572186, 0.629599988, 0.266003311, 0.173275426, 0},
 {0.005388054997, 1.63211393, 251.696182, 18037.5547, 546186.188, 0.009792809375, 19018.0117, 1538130.63, 1575194.88, 216121.422, 0}},
{10, 0.1593, {72.3908234, 2.99140072, 0.637199998, 0.269214272, 0.175367057, 0},
 {3.37275028, 39.250576, 352.373291, 2434.26758, 13285.7627, 4.22969723, 2445.92578, 24154.916, 52025.4648, 55566.9219, 0}},
{10, 0.1168, {53.0775108, 2.19331837, 0.467200011, 0.197390005, 0.128580496, 0},
 {1.66625845, 30.6352291, 372.13208, 2675.41235, 11441.5107, 5.13349295, 4271.2417, 23504.8164, 121862.406, 547337.938, 0}},
{10, 0.0928, {42.1711731, 1.74263644, 0.371199995, 0.156830415, 0.102159843, 0},
 {1.60768282, 39.6253166, 612.516968, 5670.96289, 33584.3477, 11.0835199, 14409.6475, 111979.516, 313722, 1009973.5, 0}},
{10, 0.0265, {12.0424147, 0.497627884, 0.105999999, 0.04478454962, 0.02917280048, 0},
 {0.106654033, 6.368258, 226.64003, 3950.73438, 28295.7363, 89.7414627, 51561.1094, 1270817.88, 20336782, 2788147.75, 0}},
{10, 0.0244, {11.08811, 0.458193213, 0.09759999812, 0.04123558104, 0.02686099336, 0},
 {0.07075495273, 6.19856358, 273.989136, 5317.54688, 41195.0039, 136.017487, 83653.1406, 2191223.75, 29711264, 5008162.5, 0}},
{10, 0.01423, {6.46809006, 0.26728043, 0.05693355948, 0.02405418642, 0.01566897519, 0},
 {0.468857676, 5.10219765, 44.4555626, 311.038757, 1796.65747, 80.6937408, 10730.5605, 40150.2227, 452448.813, 3054768.25, 0}},
{10, 0.006179, {2.80379581, 0.115861058, 0.02467963099, 0.01042703912, 0.006792207249, 0},
 {0.144043684, 2.36591268, 27.2262955, 197.456253, 895.054871, 219.57428, 28690.0664, 2224832.25, 15599741, 32534522, 0}},

// Z=84,  24 orbitals
{10, 93.105, {429.872772, 193.81514, 131.670349, 106.155762, 95.3687439, 0},
 {0, 0, 0, 0, 0, 34.2208443, 276.37619, 759.477966, 1318.86035, 1725.66687, 0}},
{11, 16.9393, {361.101532, 73.4048691, 33.8786011, 22.0209789, 17.773035, 16.9562397},
 {266.872803, 2848.6875, 0, 0, 0, 6.32812405, 326.251343, 1775.42444, 4091.75171, 6004.19678, 6452.10645}},
{11, 16.2443, {346.28595, 70.3931503, 32.4886017, 21.1174831, 17.0438271, 16.2605438},
 {160.466888, 3657.99561, 0, 0, 0, 1.86150336, 233.861206, 2139.08374, 6817.99902, 11823.7256, 13289.1875}},
{11, 13.8138, {294.47406, 59.8608093, 27.6276016, 17.9578495, 14.493701, 13.8276148},
 {150.405594, 4705.5127, 0, 0, 0, 1.77159297, 386.573853, 4250.64844, 14882.252, 27186.2402, 30899.5527}},
{11, 4.1494, {88.4543457, 17.9810371, 8.29880047, 5.3941927, 4.35362911, 4.15354967},
 {59.7953682, 690.732544, 5598.08887, 0, 0, 47.1483383, 1534.4502, 6301.30176, 12359.1094, 16543.8066, 17596.0977}},
{11, 3.8541, {82.159317, 16.7013817, 7.70819998, 5.01030445, 4.04379463, 3.85795403},
 {35.0146523, 670.142517, 7840.27539, 0, 0, 32.4779968, 2068.58521, 10257.8301, 20659.041, 27085.3965, 32381.3809}},
{11, 3.3019, {70.3878632, 14.3084745, 6.60379982, 4.29244804, 3.46441603, 3.30520177},
 {34.7690544, 905.96521, 14199.0693, 0, 0, 51.9325256, 4640.19531, 26946.084, 61510.2773, 87957.9922, 102043.648}},
{11, 2.798, {59.6460304, 12.1248703, 5.59599972, 3.63738155, 2.93571448, 2.80079794},
 {3.09930158, 214.75148, 9521.85254, 227992.469, 0, 10.0361357, 3448.90625, 40089.4727, 136316.672, 242563.859, 1774240.5}},
{11, 2.683, {57.1945343, 11.6265287, 5.36600018, 3.48788238, 2.81505442, 2.68568301},
 {2.09772229, 214.449371, 11856.1709, 319487.094, 0, 9.14648628, 4746.33545, 60081.2852, 210904.781, 381529.75, 2765607.75}},
{10, 0.9953, {452.294922, 18.6901512, 3.98119998, 1.68203998, 1.09568632, 0},
 {15.6847782, 183.452728, 1620.00366, 10283.457, 40372.1289, 0.207703725, 383.014008, 6601.27393, 22815.6367, 35706.043, 0}},
{10, 0.851, {386.720581, 15.9804268, 3.40400004, 1.43817544, 0.93683219, 0},
 {8.82795811, 163.769943, 1962.6001, 12855.3604, 32902.6055, 0.0855031386, 556.334351, 10723.8887, 28273.1523, 33159.0313, 0}},
{10, 0.705, {320.373688, 13.2387791, 2.81999993, 1.19143796, 0.776106596, 0},
 {8.70744705, 218.372986, 3379.19727, 29969.9551, 142041.094, 0.08800361305, 1331.04065, 33340.2109, 117201.172, 178629.156, 0}},
{10, 0.5002, {227.306259, 9.39296055, 2.00079989, 0.845329463, 0.550650358, 0},
 {0.834614336, 52.3687592, 1930.88782, 33889.8672, 198118.484, 0.01086249482, 1641.40857, 77060.2422, 216239.391, 163054.609, 0}},
{10, 0.4735, {215.127518, 8.88969898, 1.89359999, 0.80003792, 0.521147311, 0},
 {0.564270496, 52.2017975, 2397.88818, 47040.0234, 306972.5, 0.007494420279, 2406.56445, 122601.32, 353810.344, 272668.813, 0}},
{10, 0.1753, {79.6816254, 3.29267836, 0.701375306, 0.296328098, 0.193029076, 0},
 {0.005490167998, 1.5962323, 233.664749, 15881.7715, 452387.063, 0.005607036874, 11417.5176, 1004739.5, 1346146.38, 148290.984, 0}},
{10, 0.1694, {76.9632339, 3.18034625, 0.677447379, 0.286218643, 0.186443746, 0},
 {0.006257853005, 1.87339306, 283.922241, 19864.832, 580929.25, 0.007690316066, 16142.7695, 1399407.63, 1789339.88, 204764.938, 0}},
{10, 0.1709, {77.6650696, 3.2093482, 0.683625102, 0.288828701, 0.188143954, 0},
 {3.59758449, 41.5541191, 370.38327, 2536.7644, 13722.8467, 3.8461256, 2268.73438, 22722.3086, 50754.1758, 55935.4102, 0}},
{10, 0.1257, {57.1197777, 2.36035657, 0.502780914, 0.212422788, 0.138372898, 0},
 {1.84124422, 33.3191872, 397.50885, 2802.7998, 11776.8926, 4.65631342, 4009.55371, 22531.4258, 105178.719, 468122.031, 0}},
{10, 0.0983, {44.6769676, 1.8461833, 0.393256575, 0.166149214, 0.108230144, 0},
 {1.77056026, 43.1420021, 658.957031, 6033.04395, 35510.2031, 10.3189754, 13909.4502, 111363.953, 296456.781, 870142.25, 0}},
{10, 0.0314, {14.2691259, 0.589642107, 0.125599995, 0.0530654639, 0.03456701711, 0},
 {0.125401482, 7.42411327, 260.130432, 4433.51367, 31102.9941, 60.4447212, 51791.1797, 671768.875, 21656248, 2575326, 0}},
{10, 0.0314, {14.2691259, 0.589642107, 0.125599995, 0.0530654639, 0.03456701711, 0},
 {0.08318448067, 7.24629307, 315.867188, 6001.97363, 45624.3789, 68.0741501, 80080.9219, 830642.25, 31194066, 4331104.5, 0}},
{10, 0.01678, {7.62429142, 0.315058053, 0.0671107024, 0.0283539854, 0.01846987754, 0},
 {0.542585373, 5.90712309, 51.3193245, 356.646271, 2053.41846, 68.6135941, 10094.2344, 41401.5625, 322850.406, 2148358.25, 0}},
{10, 0.00756, {3.43538022, 0.14195998, 0.0302389767, 0.01277583838, 0.00832222309, 0},
 {0.19140403, 3.12850118, 35.5647087, 253.442856, 1136.24341, 204.103409, 22289.6895, 1829071.25, 15444491, 30499764, 0}},
{10, 0.005395, {2.45154977, 0.101305217, 0.02157908306, 0.009117070585, 0.005938889459, 0},
 {0.07235468179, 1.55668128, 22.3732834, 205.542908, 1246.63428, 294.814789, 26750.709, 1685096, 14813727, 35154412, 0}},

// Z=85,  24 orbitals
{10, 95.7299, {441.992126, 199.279358, 135.382523, 109.148605, 98.0574646, 0},
 {0, 0, 0, 0, 0, 33.4477615, 267.846222, 734.165283, 1273.40491, 1664.6239, 0}},
{11, 17.493, {372.904938, 75.8042755, 34.9860001, 22.7407856, 18.3539867, 17.5104923},
 {276.584686, 2912.15015, 0, 0, 0, 6.14143181, 313.561523, 1706.15686, 3932.88013, 5771.06641, 6200.07471}},
{11, 16.7847, {357.805847, 72.7349243, 33.5694008, 21.8199997, 17.6108246, 16.8014851},
 {172.179092, 3860.64307, 0, 0, 0, 1.83471894, 227.642502, 2070.0271, 6580.41406, 11404.3096, 12789.5}},
{11, 14.2135, {302.994598, 61.5928688, 28.4269981, 18.4774551, 14.9130726, 14.2277126},
 {159.925751, 4949.45166, 0, 0, 0, 1.72084296, 373.738007, 4108.19775, 14386.8389, 26292.6484, 29845.1484}},
{11, 4.317, {92.0271378, 18.7073154, 8.63399982, 5.61207151, 4.52947807, 4.3213172},
 {62.1954842, 711.726196, 5696.37939, 0, 0, 44.6913452, 1457.77856, 5995.04395, 11764.0693, 15779.8418, 16738.8672}},
{11, 4.008, {85.4400635, 17.3682919, 8.01599979, 5.21037388, 4.20526934, 4.01200819},
 {37.5388908, 705.635925, 8058.35693, 0, 0, 31.1980515, 1973.77893, 9784.1543, 19706.2461, 25816.8613, 30766.2188}},
{11, 3.426, {73.0333481, 14.8462505, 6.85200024, 4.45377731, 3.59462404, 3.42942595},
 {36.9701653, 951.975098, 14722.2813, 0, 0, 49.168354, 4418.1123, 25790.3516, 59101.8438, 84776, 97927.75}},
{11, 2.9087, {62.005867, 12.604579, 5.81739998, 3.78129101, 3.05186319, 2.9116087},
 {3.39954376, 232.614685, 10169.75, 245534.641, 0, 9.42238808, 3247.65674, 37907.1563, 129371.938, 231361.875, 1661647.88}},
{11, 2.7867, {59.4051476, 12.0759029, 5.57340002, 3.62269163, 2.9238584, 2.78948665},
 {2.29122949, 231.771805, 12651.3115, 335991.906, 0, 8.4607296, 4449.70996, 56726.8945, 200025.984, 362108.594, 2547308.75}},
{11, 1.042, {22.2127113, 4.51540947, 2.08400011, 1.35459316, 1.09328616, 1.04304194},
 {16.3888626, 190.113007, 1664.51538, 10423.3281, 0, 278.480927, 5473.95068, 17455.6113, 29099.9414, 35251.8594, 36664.1602}},
{10, 0.886, {402.625641, 16.6376705, 3.54399991, 1.49732482, 0.975362301, 0},
 {9.50923252, 173.291367, 2035.1377, 12990.9355, 31972.6758, 0.08351007849, 532.28125, 10250.1025, 27125.6035, 32082.8555, 0}},
{10, 0.74, {336.278748, 13.8960238, 2.96000004, 1.25058734, 0.814636707, 0},
 {9.30827808, 230.684082, 3526.37354, 30878.0488, 144276.859, 0.08129967004, 1240.09729, 31601.5723, 112773.047, 173837.141, 0}},
{10, 0.5332, {242.302475, 10.0126486, 2.1328001, 0.901098907, 0.586978734, 0},
 {0.923649311, 57.1134109, 2069.04883, 35504.5117, 199755.578, 0.008756686933, 1450.80774, 71079.7188, 211478.125, 175526.531, 0}},
{10, 0.4754, {216.029388, 8.92696762, 1.90153849, 0.803391874, 0.523332119, 0},
 {0.621270359, 56.660141, 2550.72974, 48721.6914, 312525.25, 0.008164047264, 2526.78271, 124631.594, 362389.063, 312908.813, 0}},
{10, 0.1971, {89.5572205, 3.70076704, 0.788302481, 0.333054453, 0.216952756, 0},
 {0.006374954246, 1.83107507, 263.773804, 17566.8594, 485916.688, 0.003499210579, 8310.03711, 839269.25, 1534910, 132977.016, 0}},
{10, 0.1906, {86.6041489, 3.57873726, 0.762308896, 0.322072238, 0.209798917, 0},
 {0.007247803267, 2.14576125, 320.237701, 21964.8242, 624340, 0.004749699496, 11686.6729, 1168259.88, 2054845.5, 183562.797, 0}},
{10, 0.1856, {84.3500137, 3.48558974, 0.742467463, 0.313689321, 0.204338253, 0},
 {3.8314631, 43.9284134, 388.764069, 2640.73657, 14159.3047, 3.40237308, 2059.03638, 21035.6172, 48738.1797, 55508.8516, 0}},
{10, 0.1385, {62.9383812, 2.60079837, 0.553997576, 0.234061599, 0.152468488, 0},
 {2.03308749, 36.1748352, 423.725555, 2930.54126, 12088.4502, 3.97840047, 3629.36279, 21085.6875, 88093.5234, 389902.156, 0}},
{10, 0.1084, {49.2721024, 2.03606772, 0.433703929, 0.183238059, 0.11936187, 0},
 {1.94541907, 46.8733482, 707.780518, 6413.43506, 37545.8477, 8.31841469, 12550.8926, 105896.617, 270469.938, 729713.875, 0}},
{10, 0.0416, {18.9016819, 0.781072855, 0.166376784, 0.07029347867, 0.04578940198, 0},
 {0.147015348, 8.59739876, 296.193359, 4941.33105, 33947.9648, 27.701849, 44771.9414, 226886.781, 12036526, 2545714.25, 0}},
{10, 0.03766, {17.1146832, 0.707228839, 0.150647223, 0.0636478141, 0.04146039113, 0},
 {0.097447671, 8.40332127, 360.461853, 6706.18555, 50046.8281, 41.8762245, 74827.9375, 402869.313, 22061360, 4196645.5, 0}},
{10, 0.01934, {8.78822899, 0.363155365, 0.07735593617, 0.03268255293, 0.02128952183, 0},
 {0.619182169, 6.72205544, 58.1308899, 401.326813, 2302.45313, 59.5306664, 9449.20703, 41729.8555, 253140.563, 1605019.5, 0}},
{10, 0.009031, {4.10398197, 0.169588551, 0.03612415865, 0.01526230201, 0.009941913188, 0},
 {0.239918485, 3.89793944, 43.7474556, 306.356781, 1357.96216, 184.359955, 18016.6719, 1456158.25, 14650160, 28570330, 0}},
{10, 0.006245, {2.83769464, 0.117261857, 0.02497801557, 0.01055310573, 0.006874327082, 0},
 {0.13962248, 3.0097971, 43.043972, 391.753876, 2370.4729, 433.118469, 39754.7734, 2284654.75, 23007766, 52331796, 0}},

// Z=86,  24 orbitals
{10, 98.404, {454.338654, 204.845993, 139.164276, 112.19754, 100.796585, 0},
 {0, 0, 0, 0, 0, 32.7082405, 259.658112, 709.805664, 1230.23901, 1608.56274, 0}},
{11, 18.049, {384.757416, 78.2136536, 36.0979996, 23.4635811, 18.9373531, 18.067049},
 {286.324463, 2972.9043, 0, 0, 0, 5.97114515, 301.749786, 1641.25891, 3782.34375, 5549.04736, 5963.90527}},
{11, 17.337, {369.581573, 75.1287003, 34.6741982, 22.5381165, 18.1904144, 17.3544369},
 {184.543777, 4069.38232, 0, 0, 0, 1.80936372, 221.735947, 2004.33875, 6352.86719, 10987.8975, 12345.5225}},
{11, 14.619, {311.647308, 63.351799, 29.2388, 19.0051231, 15.3389511, 14.6340189},
 {169.612091, 5201.73242, 0, 0, 0, 1.67247784, 361.44751, 3972.32251, 13912.0547, 25430.2305, 28817.8984}},
{11, 4.482, {95.5445023, 19.422327, 8.96399975, 5.82657051, 4.70259953, 4.48648214},
 {64.6178741, 732.53833, 5786.84521, 0, 0, 42.535656, 1389.38672, 5717.67676, 11219.5264, 15059.6963, 15867.3984}},
{11, 4.159, {88.658989, 18.0226364, 8.31799984, 5.40667248, 4.36370134, 4.16315889},
 {40.2067871, 741.913757, 8263.90039, 0, 0, 30.1537457, 1891.12329, 9358.37988, 18830.8457, 24668.4512, 29424.4844}},
{11, 3.538, {75.4208908, 15.3315916, 7.07599974, 4.59937668, 3.71213651, 3.541538},
 {39.2416992, 998.335022, 15222.5459, 0, 0, 47.1776733, 4247.63916, 24860.709, 57111.8984, 81949.1953, 94896.7188}},
{11, 3.0215, {64.4104691, 13.0933867, 6.04299974, 3.92793012, 3.17021489, 3.02452135},
 {3.72480917, 251.646774, 10847.8906, 0, 0, 8.86661434, 3063.10254, 35888.8672, 122888.352, 219814.375, 1555675.25}},
{11, 2.8924, {61.6583939, 12.5339441, 5.78480005, 3.76010108, 3.03476095, 2.89529252},
 {2.4998889, 250.186829, 13483.7422, 359265.281, 0, 7.84287739, 4177.10596, 53618.7813, 189895.703, 344056.375, 2367022.25}},
{11, 1.097, {23.3851681, 4.75374651, 2.19400001, 1.42609274, 1.15099323, 1.09809697},
 {17.1224976, 196.950211, 1709.77856, 10565.5049, 0, 259.228363, 5127.47314, 16426.4648, 27496.5684, 33383.9609, 34959.8516}},
{10, 0.929, {422.166168, 17.4451427, 3.71600008, 1.56999409, 1.02269936, 0},
 {10.2345371, 183.232803, 2109.27319, 13117.0352, 30837.2793, 0.07991340756, 501.215485, 9699.64551, 25842.377, 30759.2852, 0}},
{10, 0.768, {349.002808, 14.4218187, 3.07200003, 1.29790688, 0.845460773, 0},
 {9.93726158, 243.32457, 3672.85864, 31733.0859, 146491.719, 0.07789496332, 1185.83594, 30462.9922, 109736.867, 170845.469, 0}},
{10, 0.5666, {257.480469, 10.6398468, 2.2664001, 0.957544327, 0.623747468, 0},
 {1.02049899, 62.188385, 2213.63403, 37139.5391, 201148.625, 0.00751187047, 1292.48083, 65799.7656, 206196.516, 184906.563, 0}},
{10, 0.537, {244.029312, 10.0840063, 2.148, 0.907520831, 0.591162026, 0},
 {0.684727073, 61.7869377, 2747.36743, 51662.1875, 315463.781, 0.005322730169, 1855.35278, 104139.547, 337458, 316493.656, 0}},
{10, 0.2196, {99.8069687, 4.12431622, 0.878523052, 0.37117222, 0.241782799, 0},
 {0.007387246937, 2.09458923, 296.885498, 19372.9922, 520578.719, 0.00218582619, 6192.96729, 704435.938, 1681183.25, 119902.75, 0}},
{10, 0.2126, {96.6063461, 3.99205732, 0.85035044, 0.35926941, 0.234029278, 0},
 {0.008376910351, 2.45077538, 360.115967, 24213.7227, 669276, 0.00296500884, 8667.49023, 979375.5, 2265214.75, 165190.688, 0}},
{10, 0.2008, {91.2637482, 3.7712853, 0.803323686, 0.339400828, 0.2210868, 0},
 {4.07389593, 46.3649178, 407.378784, 2744.46777, 14580.5107, 3.03716993, 1877.47229, 19520.6914, 46664.3086, 54876.0547, 0}},
{10, 0.1518, {68.9694824, 2.85002112, 0.607084632, 0.256490648, 0.167078853, 0},
 {2.23659277, 39.163929, 450.547058, 3056.46533, 12372.6953, 3.31904221, 3301.19556, 19788.6934, 75461.7031, 327917.344, 0}},
{10, 0.1188, {53.9939003, 2.23118615, 0.475266248, 0.20079796, 0.130800441, 0},
 {2.12870932, 50.7740746, 758.425842, 6803.62549, 39616.2148, 6.97234297, 11397.9805, 100879.609, 252484.063, 626281.563, 0}},
{10, 0.0487, {22.1267567, 0.914342403, 0.194764599, 0.0822872147, 0.05360215902, 0},
 {0.170608014, 9.86708832, 334.363861, 5458.49316, 36754.0469, 18.7006512, 40889.3008, 129308.539, 7570712, 2955377.25, 0}},
{10, 0.04426, {20.1108532, 0.831039429, 0.177020177, 0.07479026914, 0.04871862754, 0},
 {0.112919755, 9.6537981, 407.838867, 7431.88428, 54478.7461, 27.1053352, 68483.875, 222144.75, 13698569, 4497876, 0}},
{10, 0.02194, {9.97006226, 0.411992192, 0.08775869757, 0.03707767278, 0.02415251732, 0},
 {0.697724998, 7.55132723, 64.9880676, 445.701447, 2547.71094, 52.4515724, 8863.20605, 41573.1172, 212389.359, 1263301.25, 0}},
{10, 0.01057, {4.80451918, 0.198536798, 0.04229043797, 0.01786752976, 0.01163896732, 0},
 {0.290859699, 4.692204, 51.9614487, 357.583649, 1567.24304, 165.204025, 15162.6973, 1169024.63, 13539781, 26944454, 0}},
{10, 0.007126, {3.23821735, 0.133812636, 0.02850350551, 0.01204260904, 0.007844595239, 0},
 {0.230380446, 4.94895029, 70.2489929, 633.438477, 3824.1377, 551.077393, 52582.9805, 2657891.25, 30586460, 67924296, 0}},

// Z=87,  24 orbitals
{10, 101.137, {466.957123, 210.535233, 143.029312, 115.313629, 103.596031, 0},
 {0, 0, 0, 0, 0, 31.9943752, 251.75592, 686.28363, 1188.50024, 1554.37158, 0}},
{11, 18.639, {397.334656, 80.7703629, 37.2779999, 24.2305775, 19.5563908, 18.6576385},
 {296.320923, 3034.51367, 0, 0, 0, 5.79519224, 289.814911, 1576.21301, 3633.44873, 5330.49902, 5726.51416}},
{11, 17.9065, {381.719696, 77.5961456, 35.8129997, 23.2783337, 18.7878399, 17.9244061},
 {197.661118, 4284.53076, 0, 0, 0, 1.7846905, 216.020752, 1940.70569, 6133.7002, 10599.9395, 11879.2715}},
{11, 15.0312, {320.425812, 65.1362915, 30.0623989, 19.5404606, 15.771019, 15.0462303},
 {179.735413, 5460.53027, 0, 0, 0, 1.6267916, 349.418213, 3842.1543, 13457.3525, 24601.6191, 27843.0977}},
{11, 4.652, {99.168457, 20.1590061, 9.3039999, 6.04756927, 4.88096666, 4.65665197},
 {67.0797348, 753.291138, 5871.17725, 0, 0, 40.2350731, 1324.10559, 5452.16016, 10698.1826, 14343.709, 15202.3613}},
{11, 4.327, {92.2403107, 18.7506485, 8.65400028, 5.62507153, 4.5399704, 4.33132696},
 {43.0241318, 779.583435, 8468.98926, 0, 0, 28.9278431, 1801.745, 8913.31836, 17933.7539, 23499.6621, 24924.9941}},
{11, 3.663, {78.0855713, 15.8732672, 7.32600021, 4.76187611, 3.8432889, 3.66666293},
 {41.6333008, 1046.61279, 15742.6631, 0, 0, 44.9014091, 4058.74146, 23851.6895, 54968.7344, 79106.7031, 91997.1406}},
{11, 3.1362, {66.8555756, 13.5904293, 6.2723999, 4.07703924, 3.29056048, 3.13933635},
 {4.07376003, 271.746887, 11546.2959, 0, 0, 8.35686111, 2892.1228, 33985.2539, 116673.094, 208211.438, 1432288.88}},
{11, 2.9997, {63.9457474, 12.9989185, 5.99940014, 3.89959025, 3.14734197, 3.00269961},
 {2.72257781, 269.586182, 14340.2451, 0, 0, 7.28321743, 3925.10522, 50692.0859, 180248.219, 327725.219, 2252029.25}},
{11, 1.153, {24.5789413, 4.99641752, 2.30599999, 1.49889243, 1.20974946, 1.15415299},
 {17.86339, 203.80542, 1753.93909, 10689.7549, 0, 241.903488, 4810.63721, 15469.8379, 25993.2148, 31721.2422, 33730.0469}},
{10, 0.98, {445.342133, 18.4028416, 3.92000008, 1.65618324, 1.07884324, 0},
 {11.0042925, 193.559799, 2184.04248, 13225.3623, 29552.2559, 0.07072333992, 465.379456, 9094.54102, 24473.4766, 29370.959, 0}},
{10, 0.81, {368.088898, 15.2105122, 3.24000001, 1.36888611, 0.89169693, 0},
 {10.6014233, 256.580383, 3827.44263, 32666.8652, 148676, 0.07096084207, 1090.48291, 28607.0156, 104814.398, 165223.531, 0}},
{10, 0.6033, {274.158081, 11.3290148, 2.4131999, 1.01956666, 0.664149106, 0},
 {1.12570786, 67.6019821, 2364.32104, 38783.3672, 202300.578, 0.006375303492, 1142.47266, 60534.2539, 200025.719, 191195.453, 0}},
{10, 0.577, {262.206543, 10.8351431, 2.30800009, 0.975120127, 0.635196447, 0},
 {0.753192127, 67.1091461, 2937.00317, 54118.6641, 319030.969, 0.004454412032, 1579.79993, 94146.6641, 324777.156, 324697, 0}},
{10, 0.2465, {112.011917, 4.62866068, 0.985953689, 0.416561216, 0.271349341, 0},
 {0.008524795994, 2.38788056, 333.179413, 21303.709, 556148.375, 0.001353483414, 4465.52393, 575197.938, 1800094.88, 109305.172, 0}},
{10, 0.2389, {108.546692, 4.48546743, 0.955451965, 0.403674394, 0.262954801, 0},
 {0.009641480632, 2.78954172, 403.793365, 26616.8008, 715483.375, 0.001830850029, 6212.43799, 797995.813, 2448281.25, 148537.156, 0}},
{10, 0.22, {99.9908829, 4.13191652, 0.880141914, 0.371856183, 0.242228344, 0},
 {4.31972933, 48.8407059, 426.17984, 2847.59814, 14988.2461, 2.51213217, 1672.80847, 17778.0117, 43958.2109, 55705.2422, 0}},
{10, 0.169, {76.803009, 3.17372537, 0.676037073, 0.285622776, 0.186055616, 0},
 {2.45826054, 42.3176231, 477.88623, 3179.37451, 12626.582, 2.73784781, 2921.13599, 18260.4551, 62075.5117, 251623.844, 0}},
{10, 0.1329, {60.4199524, 2.49672937, 0.531829774, 0.224695817, 0.14636758, 0},
 {2.32740712, 54.9004822, 810.926147, 7203.26953, 41725.6406, 5.50116587, 9947.12988, 93407.375, 232022.438, 531928.188, 0}},
{10, 0.05954, {27.0558186, 1.11802578, 0.238151312, 0.100617915, 0.06554283947, 0},
 {0.197200686, 11.2590389, 374.991882, 5991.90234, 39557.9727, 10.7602911, 34290.2656, 76739.4609, 3658158.75, 16185262, 0}},
{10, 0.05455, {24.7905121, 1.02441657, 0.218211561, 0.0921934545, 0.06005511805, 0},
 {0.130038559, 11.0167799, 458.344269, 8182.57959, 58945.6406, 14.4817677, 57082.9531, 127450.063, 6339316.5, 18006056, 0}},
{10, 0.02787, {12.6640158, 0.523314238, 0.111471474, 0.04709622264, 0.03067863174, 0},
 {0.786665022, 8.52335644, 73.14534, 498.471558, 2841.0896, 37.4133263, 7173.56934, 38451.9102, 127378.5, 560245.438, 0}},
{10, 0.01516, {6.89143038, 0.284774095, 0.06065989658, 0.02562854625, 0.01669451967, 0},
 {0.361461133, 5.83639669, 64.0223694, 433.053284, 1879.62244, 105.092667, 9165.91406, 527249.125, 7122332.5, 30174114, 0}},
{10, 0.01061, {4.82253885, 0.199281439, 0.04244905338, 0.01793454401, 0.01168262027, 0},
 {0.309049726, 6.73805857, 95.7926941, 856.510742, 5166.65625, 348.738373, 41605.0625, 1091980, 18438080, 72377288, 0}},

// Z=88,  24 orbitals
{10, 103.922, {479.815216, 216.332504, 146.967758, 118.488899, 106.448647, 0},
 {0, 0, 0, 0, 0, 31.3100777, 244.163834, 663.638, 1147.81702, 1499.77893, 0}},
{11, 19.2367, {410.07608, 83.3604355, 38.4734001, 25.0075836, 20.1835098, 19.2559376},
 {306.343689, 3094.58105, 0, 0, 0, 5.63081169, 278.546875, 1514.61377, 3491.29004, 5121.86621, 5501.38721}},
{11, 18.4843, {394.036865, 80.0999832, 36.9686012, 24.0294685, 19.3940773, 18.5027847},
 {211.452972, 4504.41064, 0, 0, 0, 1.76240766, 210.690384, 1880.81946, 5925.33057, 10221.9863, 11428.7754}},
{11, 15.4444, {329.234161, 66.9268646, 30.8888016, 20.0776196, 16.2045574, 15.4598446},
 {190.220993, 5724.52295, 0, 0, 0, 1.58466554, 338.502838, 3719.79443, 13024.5957, 23807.623, 26958.5313}},
{11, 4.822, {102.792412, 20.8956852, 9.64400005, 6.26856852, 5.0593338, 4.8268218},
 {69.5919037, 773.902649, 5947.17578, 0, 0, 38.3973579, 1264.32166, 5205.85107, 10210.7266, 13711.1406, 14467.4326}},
{11, 4.4895, {95.7043839, 19.4548283, 8.97900009, 5.8363204, 4.71046829, 4.49398947},
 {45.9796181, 817.75769, 8657.50586, 0, 0, 27.9548969, 1725.4906, 8520.21973, 17128.5, 22414.0684, 23930.1797}},
{11, 3.7918, {80.8312454, 16.4314098, 7.58360004, 4.92931509, 3.97842836, 3.79559183},
 {44.1171417, 1096.23083, 16268.6279, 0, 0, 42.7194214, 3877.75952, 22881.248, 52901.7656, 76370.5938, 82590.9453}},
{11, 3.2484, {69.2473831, 14.0766373, 6.49679995, 4.22289848, 3.40828276, 3.25164843},
 {4.4482975, 292.898895, 12257.7246, 0, 0, 7.88969946, 2745.00854, 32304.4824, 111103.805, 198679.281, 177188.359}},
{11, 3.1049, {66.1883392, 13.454793, 6.20980024, 4.03634977, 3.25771999, 3.10800505},
 {2.96144414, 289.992767, 15213.833, 0, 0, 6.81387997, 3706.67676, 48087.6328, 171447.391, 311220.844, 214832.594}},
{11, 1.2084, {25.7599239, 5.23648834, 2.41680002, 1.570912, 1.26787615, 1.20960844},
 {18.6201916, 210.705627, 1796.98657, 10796.8408, 0, 226.921906, 4529.73535, 14609.6328, 24648.9258, 30129.293, 32176.0742}},
{11, 1.0576, {22.5452633, 4.58301067, 2.11520004, 1.37487304, 1.10965395, 1.05865765},
 {11.8345547, 204.558075, 2264.33569, 13339.873, 0, 305.868378, 7298.83887, 19680.2715, 26306.8438, 27586.1445, 29280.1055}},
{10, 0.8791, {399.490082, 16.5081005, 3.5164001, 1.48566401, 0.967766345, 0},
 {11.3144207, 270.821228, 3999.07324, 33804.1758, 150625.859, 0.05938782915, 935.522278, 25656.2949, 96879.2031, 155201.5, 0}},
{10, 0.6359, {288.972504, 11.9411907, 2.54360008, 1.07466018, 0.700037122, 0},
 {1.24038506, 73.3497391, 2518.40259, 40360.0078, 203748.609, 0.005654674955, 1039.39172, 56579.8047, 194990.953, 198087.109, 0}},
{10, 0.6027, {273.885406, 11.3177481, 2.41079998, 1.01855266, 0.663488567, 0},
 {0.826027393, 72.6433868, 3123.74219, 56255.668, 323509.313, 0.004171425942, 1465.78882, 89226.8438, 319066.719, 341851.063, 0}},
{10, 0.2989, {135.829346, 5.61286688, 1.19560003, 0.505135894, 0.329047173, 0},
 {0.009832645766, 2.72480321, 376.080597, 23819.7715, 611912.938, 0.0005584582686, 2402.48877, 393222.969, 1755934, 120076.273, 0}},
{10, 0.2989, {135.829346, 5.61286688, 1.19560003, 0.505135894, 0.329047173, 0},
 {0.01109748799, 3.18180394, 456.724854, 29953.3633, 797488.875, 0.000651778304, 2963.92896, 509079.594, 2362534, 161796.984, 0}},
{10, 0.2544, {115.607178, 4.7772274, 1.01760006, 0.429931641, 0.280058891, 0},
 {4.58083582, 51.4267845, 445.765533, 2958.88501, 15452.5674, 1.90909231, 1359.26526, 15092.7217, 39369.9531, 50218.1484, 0}},
{10, 0.2004, {91.0679245, 3.76319337, 0.801599979, 0.338672578, 0.220612422, 0},
 {2.69415879, 45.6471405, 506.502319, 3306.69385, 12853.5693, 1.91933167, 2339.09937, 15853.7207, 47174.8281, 179422.031, 0}},
{10, 0.1528, {69.4370193, 2.8693409, 0.611199975, 0.258229375, 0.16821146, 0},
 {2.53644729, 59.2411537, 866.09198, 7624.38721, 43971.8086, 3.86100101, 8254.29102, 83413.5703, 208301.781, 416361.406, 0}},
{10, 0.0672, {30.5377464, 1.26190913, 0.26879999, 0.113566853, 0.07397782058, 0},
 {0.226390973, 12.7725086, 418.153687, 6537.43896, 42370.2734, 8.0606432, 31118.6367, 70931.5938, 2077290.13, 27086270, 0}},
{10, 0.0672, {30.5377464, 1.26190913, 0.26879999, 0.113566853, 0.07397782058, 0},
 {0.149072587, 12.5121536, 512.665894, 8971.58398, 63542.8398, 7.59291506, 45706.7578, 114507.891, 2777723, 44059340, 0}},
{10, 0.0435, {19.7677383, 0.816860855, 0.173999995, 0.07351426035, 0.04788742587, 0},
 {0.886055529, 9.58439827, 81.9342346, 555.459351, 3162.22949, 17.509119, 4264.07861, 28889.3945, 65586.4219, 260055.984, 0}},
{10, 0.0188, {8.54329777, 0.353034109, 0.07519999892, 0.03177167848, 0.02069617435, 0},
 {0.436373144, 7.01217413, 75.9711151, 505.077942, 2168.33765, 82.3294144, 7616.53564, 335531.406, 4094729.75, 19157046, 0}},
{10, 0.0188, {8.54329777, 0.353034109, 0.07519999892, 0.03177167848, 0.02069617435, 0},
 {0.385011405, 8.42786121, 119.413391, 1059.34106, 6383.57666, 130.730331, 25809.3125, 308458, 4742936.5, 31345792, 0}},

// Z=89,  24 orbitals
{10, 106.755, {492.897217, 222.230743, 150.974792, 121.71946, 109.350937, 0},
 {0, 0, 0, 0, 0, 30.6567307, 236.888184, 641.889221, 1109.16467, 1449.53955, 0}},
{11, 19.84, {422.936829, 85.9747772, 39.6800003, 25.7918701, 20.8165035, 19.8598404},
 {316.477295, 3151.50293, 0, 0, 0, 5.48200941, 267.833832, 1456.66809, 3357.55054, 4923.81934, 5289.55908}},
{11, 19.0832, {406.803864, 82.6952591, 38.1664009, 24.8080349, 20.0224552, 19.1022835},
 {226.114304, 4733.34521, 0, 0, 0, 1.74003148, 205.477539, 1822.59863, 5724.74121, 9864.77539, 11008.3662}},
{11, 15.871, {338.328156, 68.7754898, 31.7420006, 20.6321964, 16.6521549, 15.8868713},
 {201.240692, 6000.46094, 0, 0, 0, 1.54255128, 327.727875, 3600.14429, 12607.2275, 23046.5723, 26069.3262}},
{11, 5.0021, {106.629539, 21.6756973, 10.0039997, 6.50256729, 5.24819326, 5.00700188},
 {72.1379547, 794.61084, 6019.70459, 0, 0, 36.5615044, 1205.20618, 4964.37451, 9734.85059, 13080.2842, 13720.7324}},
{11, 4.656, {99.2537231, 20.1763401, 9.31200027, 6.05276918, 4.88516331, 4.66065598},
 {49.1042252, 856.942261, 8836.01172, 0, 0, 26.8872776, 1653.85034, 8148.21484, 16358.6445, 21413.0723, 22850.3496}},
{11, 3.909, {83.3296432, 16.9392853, 7.81799984, 5.0816741, 4.10139704, 3.91290903},
 {46.6983032, 1146.28943, 16771.6113, 0, 0, 41.137394, 3736.69727, 22090.2813, 51169.0547, 74077.4141, 79779.8984}},
{11, 3.3703, {71.8438416, 14.6044455, 6.74039984, 4.38123798, 3.53607774, 3.3735702},
 {4.85433435, 315.624695, 13025.123, 0, 0, 7.43538141, 2592.81348, 30611.1035, 105576.883, 188526.906, 195328.656}},
{11, 3.219, {68.6206512, 13.949235, 6.4380002, 4.18467903, 3.37743568, 3.22221899},
 {3.21741652, 311.810425, 16155.4209, 0, 0, 6.32328796, 3482.07495, 45468.3867, 162752.125, 295087.625, 338237.219}},
{11, 1.269, {27.0517559, 5.49909258, 2.53800011, 1.6496917, 1.33145881, 1.27026904},
 {19.3954716, 217.705322, 1840.07129, 10899.5684, 0, 211.895935, 4251.27832, 13759.5889, 23288.9883, 28643.2637, 29793.3809}},
{11, 1.08, {23.0227718, 4.68007898, 2.16000009, 1.40399289, 1.13315642, 1.08107996},
 {12.6863012, 215.212051, 2330.64307, 13359.8379, 0, 305.483002, 7156.78564, 19147.0156, 25651.1367, 27066.9844, 29640.3281}},
{10, 0.89, {404.443359, 16.7127857, 3.55999994, 1.50408483, 0.979765713, 0},
 {12.0329638, 284.424713, 4142.14111, 34493.0586, 152798.938, 0.06149294972, 947.457397, 25668.2832, 96631.8672, 155586.656, 0}},
{10, 0.675, {306.695313, 12.6735487, 2.69959998, 1.14056945, 0.742970705, 0},
 {1.3641398, 79.5076447, 2682.74268, 42052, 204243.516, 0.004857109394, 925.638916, 52211.957, 188149.891, 202629.75, 0}},
{10, 0.637, {289.472382, 11.9618473, 2.5480001, 1.07651913, 0.70124805, 0},
 {0.904943466, 78.5924606, 3324.62476, 58619.4414, 326985.094, 0.003750892123, 1309.66064, 82803.7422, 309026.563, 346923.031, 0}},
{10, 0.3039, {138.121368, 5.70758009, 1.21577489, 0.513659716, 0.334599614, 0},
 {0.01126796473, 3.07849669, 416.329987, 25565.6836, 630210.063, 0.0005881576217, 2465.20459, 390488.156, 1801447.88, 137987.719, 0}},
{10, 0.2952, {134.08757, 5.54089117, 1.18026841, 0.498658359, 0.324827671, 0},
 {0.01267871819, 3.58470702, 503.618317, 31913.9668, 811952.313, 0.0007991899038, 3393.05811, 539408.875, 2470865, 183959.109, 0}},
{10, 0.2614, {118.722435, 4.90595865, 1.04502118, 0.441516966, 0.287605584, 0},
 {4.84823561, 54.0132103, 464.316376, 3051.76855, 15752.6426, 1.90675843, 1345.0105, 14846.002, 39057.2734, 50900.0313, 0}},
{10, 0.2063, {93.6906281, 3.8715713, 0.824685633, 0.348426163, 0.226965934, 0},
 {2.94850135, 49.0861168, 534.170288, 3416.86621, 13048.9395, 1.8778075, 2321.24805, 15658.3936, 45475.8203, 160936.078, 0}},
{10, 0.1632, {74.1787567, 3.06528354, 0.652937829, 0.275863439, 0.179698348, 0},
 {2.76166415, 63.7891426, 922.139343, 8038.06787, 46089.5313, 3.45410872, 7736.16016, 80580.0547, 203540.281, 375716.531, 0}},
{10, 0.0832, {37.7796021, 1.5611639, 0.332544416, 0.140498593, 0.09152124077, 0},
 {0.259258419, 14.4386101, 464.838562, 7122.88965, 45260.4453, 4.38500404, 24185.0078, 72250.2734, 1012145.5, 6806866, 0}},
{10, 0.077, {34.9634132, 1.4447906, 0.307755679, 0.130025461, 0.08469901234, 0},
 {0.170383453, 14.1434507, 570.172852, 9780.2002, 68143.4922, 5.16854811, 39766.3242, 124322.398, 1605791.13, 11435348, 0}},
{10, 0.0405, {18.3879013, 0.759841919, 0.161854371, 0.06838277727, 0.04454476759, 0},
 {0.987139583, 10.6504345, 90.5672455, 609.544067, 3449.40454, 22.4649811, 5156.58691, 33851.1367, 80294.3672, 275116.5, 0}},
{10, 0.0252, {11.4448824, 0.472936064, 0.100740388, 0.04256238416, 0.02772527561, 0},
 {0.515477836, 8.19324684, 87.4149399, 571.221741, 2425.96313, 52.8046532, 5783.98682, 182337.078, 2409595, 9638924, 0}},
{10, 0.0184, {8.36247349, 0.345561922, 0.07360834628, 0.03109921142, 0.02025812678, 0},
 {0.456029475, 9.97208118, 140.421387, 1234.9104, 7425.07813, 161.046219, 30990.6211, 365109.531, 4785396, 23329580, 0}},

// Z=90,  24 orbitals
{10, 109.651, {506.266418, 228.258469, 155.069794, 125.020943, 112.31694, 0},
 {0, 0, 0, 0, 0, 30.0256462, 229.867676, 620.880432, 1071.32349, 1398.48145, 0}},
{11, 20.4721, {436.41156, 88.7139282, 40.9441986, 26.6135941, 21.4797153, 20.4925709},
 {326.74823, 3206.74072, 0, 0, 0, 5.329422, 257.386353, 1399.47424, 3225.70532, 4730.33398, 5081.20703}},
{11, 19.6932, {419.807465, 85.3386383, 39.3864021, 25.6010303, 20.6624794, 19.7128944},
 {241.548141, 4967.70508, 0, 0, 0, 1.71924424, 200.574661, 1767.55457, 5533.49854, 9516.22852, 10637.3252}},
{11, 16.3003, {347.479706, 70.6358185, 32.6006012, 21.1902828, 17.1025829, 16.3166008},
 {212.661148, 6283.76123, 0, 0, 0, 1.50324297, 317.605621, 3487.47925, 12211.0791, 22325.4277, 25228.9023}},
{11, 5.1823, {110.473068, 22.4570122, 10.3646002, 6.73695612, 5.43736744, 5.18748236},
 {74.7322159, 815.165771, 6084.66406, 0, 0, 34.9095573, 1150.94543, 4740.57813, 9292.06641, 12494.0732, 13091.6123}},
{11, 4.8304, {102.971481, 20.9320869, 9.66079998, 6.27948856, 5.06814718, 4.83523035},
 {52.3884277, 897.195007, 9006.37988, 0, 0, 25.9803047, 1583.89282, 7787.53174, 15613.9453, 20437.0313, 21649.168}},
{11, 4.0461, {86.2522583, 17.5333958, 8.09220028, 5.25990343, 4.24524498, 4.0501461},
 {49.4018021, 1199.05383, 17318.6953, 0, 0, 39.106926, 3568.88379, 21193.166, 49261.0703, 71324.6953, 77848.9531}},
{11, 3.4909, {74.414711, 15.1270542, 6.98159981, 4.53801727, 3.66261339, 3.49429083},
 {5.28865862, 339.583618, 13815.9453, 0, 0, 7.04424381, 2459.69165, 29104.2305, 100584.469, 179252.141, 217435.875}},
{11, 3.332, {71.0295105, 14.4389095, 6.66400003, 4.33157825, 3.49599743, 3.33533192},
 {3.49079299, 334.796844, 17127.3691, 0, 0, 5.86712837, 3284.89453, 43138.0977, 154989.938, 281640, 284612.781}},
{11, 1.3296, {28.3414593, 5.76126385, 2.65899992, 1.72834122, 1.39493656, 1.3308295},
 {20.1967239, 224.785431, 1882.17468, 10988.6104, 0, 198.749786, 4002.68408, 12988.8184, 22050.8066, 27266.7383, 27966.627}},
{11, 1.1682, {24.9029655, 5.06228542, 2.33640003, 1.51865232, 1.22569752, 1.16936815},
 {13.6110401, 227.000778, 2411.93018, 13423.79, 0, 267.61322, 6459.70215, 17602.873, 23771.4902, 25100.8711, 26530.9922}},
{10, 0.9674, {439.570862, 18.1643562, 3.86919999, 1.63472044, 1.06486225, 0},
 {12.8120422, 299.762665, 4325.41553, 35709.0195, 154997.109, 0.05151344091, 808.669556, 22940.1172, 89038.2969, 145699.672, 0}},
{10, 0.7141, {324.509003, 13.4096622, 2.85640001, 1.20681679, 0.786124408, 0},
 {1.49804378, 86.0589066, 2854.00293, 43763.1836, 204403.484, 0.004222495947, 831.180847, 48406.6719, 181424.375, 203290.922, 0}},
{10, 0.6764, {307.376953, 12.7017164, 2.70560002, 1.14310443, 0.744621992, 0},
 {0.990160763, 84.9544144, 3537.86011, 61136.7109, 329522.906, 0.003322858829, 1151.62366, 76118.0234, 296861.813, 351032.781, 0}},
{10, 0.3445, {156.505951, 6.46728468, 1.37759995, 0.582030118, 0.379136324, 0},
 {0.01291557774, 3.48761749, 465.507446, 28162.8242, 678548.563, 0.0003316395741, 1685.94482, 305010.875, 1679498.13, 175044.375, 0}},
{10, 0.3352, {152.32518, 6.29452324, 1.34080005, 0.566482306, 0.369008392, 0},
 {0.01449720468, 4.05466318, 562.677917, 35155.6523, 875816.438, 0.0004542820679, 2291.95605, 418691.313, 2306520.25, 230358.953, 0}},
{10, 0.2903, {131.875809, 5.44949436, 1.16079998, 0.490433037, 0.31946969, 0},
 {5.12543678, 56.6921768, 483.870819, 3157.56494, 16144.9473, 1.59335065, 1164.63464, 13204.5693, 35900.0742, 47829.9609, 0}},
{10, 0.2295, {104.246414, 4.30776739, 0.917599976, 0.38768208, 0.25253737, 0},
 {3.22147226, 52.7174034, 563.078918, 3531.89648, 13205.3047, 1.54382432, 2036.42456, 14326.3066, 39397.5703, 133587.547, 0}},
{10, 0.1818, {82.615509, 3.41391516, 0.727200031, 0.307238907, 0.200136423, 0},
 {3.00122476, 68.5812454, 981.033936, 8479.2998, 48400.3281, 2.72940397, 6723.08105, 73879.2813, 190433.984, 334627.75, 0}},
{10, 0.0943, {42.8528175, 1.77080405, 0.377200007, 0.159365386, 0.103811137, 0},
 {0.295375645, 16.2332058, 513.687317, 7711.88379, 48096.0586, 3.09335065, 21043.4316, 77838.1016, 631898.438, 3774984.25, 0}},
{10, 0.0879, {39.9444618, 1.65062225, 0.351599991, 0.148549497, 0.09676562995, 0},
 {0.193716198, 15.9032726, 630.955017, 10614.918, 72772.0156, 3.41730022, 34181.8906, 135000.875, 965734.438, 5904875, 0}},
{10, 0.0595, {27.0386295, 1.11731541, 0.238000005, 0.100553982, 0.06550119072, 0},
 {1.09475064, 11.7419624, 99.2631836, 664.996521, 3763.64331, 11.4499903, 3186.3562, 24798.3672, 56315.5508, 174274.469, 0}},
{10, 0.049, {22.267107, 0.920142114, 0.195999995, 0.08280916512, 0.05394215882, 0},
 {0.599823773, 9.4090662, 98.8109894, 635.234314, 2668.6936, 14.2233953, 2937.01074, 40275.5586, 688249.25, 3604660.25, 0}},
{10, 0.043, {19.5405216, 0.807471633, 0.172000006, 0.07266926765, 0.0473369956, 0},
 {0.530631721, 11.5330715, 161.249908, 1410.66431, 8507.92773, 25.3249302, 11615.5664, 91773.8047, 862096.063, 6226572, 0}},

// Z=91,  24 orbitals
{10, 112.601, {519.889099, 234.400482, 159.242432, 128.385025, 115.33918, 0},
 {0, 0, 0, 0, 0, 29.421793, 223.136414, 600.697205, 1035.42517, 1351.72815, 0}},
{11, 21.1046, {449.894806, 91.4548035, 42.209198, 27.4358406, 22.1433449, 21.1257038},
 {337.117065, 3259.90503, 0, 0, 0, 5.19536877, 247.796692, 1346.48499, 3103.39087, 4549.07129, 4889.17822}},
{11, 20.3137, {433.034882, 88.0275192, 40.6273994, 26.4076767, 21.3135185, 20.334013},
 {257.90567, 5207.93652, 0, 0, 0, 1.70157337, 195.931076, 1716.45898, 5356.15381, 9198.30371, 10276.5957}},
{11, 16.7331, {356.705872, 72.511322, 33.4662018, 21.7529202, 17.5566864, 16.7498341},
 {224.646347, 6579.08398, 0, 0, 0, 1.46701753, 308.214417, 3383.15088, 11847.1455, 21659.8418, 24503.1035}},
{11, 5.3669, {114.408257, 23.256958, 10.7337999, 6.97693491, 5.63105297, 5.37226677},
 {77.3622284, 835.906982, 6148.06592, 0, 0, 33.3525963, 1100.01904, 4531.10254, 8875.43359, 11910.4385, 12559.3232}},
{11, 5.0009, {106.606094, 21.6709328, 10.0018005, 6.50113726, 5.24703884, 5.00590086},
 {55.8595657, 938.530823, 9165.85547, 0, 0, 25.257082, 1523.98364, 7469.33203, 14942.1523, 19542.1445, 20611.7461}},
{11, 4.1738, {88.9744873, 18.086771, 8.34759998, 5.42591286, 4.37923002, 4.17797375},
 {52.2285728, 1253.22778, 17866.4277, 0, 0, 37.5625, 3435.58594, 20462.5117, 47677.2148, 69167.0703, 75423.3516}},
{11, 3.6112, {76.9813232, 15.648797, 7.22239971, 4.69453621, 3.78893948, 3.61481118},
 {5.75961304, 365.341827, 14668.9941, 0, 0, 6.70611525, 2344.94092, 27821.252, 96390.9688, 172092.281, 199769.953}},
{11, 3.4418, {73.370163, 14.9147177, 6.88359976, 4.47431755, 3.61120176, 3.44524169},
 {3.78538799, 359.353912, 18164.543, 0, 0, 5.52108574, 3122.3291, 41225.8398, 148619.266, 270616.875, 243039.594}},
{11, 1.3871, {29.5693417, 6.01086807, 2.7742002, 1.80322099, 1.45537162, 1.3884871},
 {21.0021286, 231.888794, 1923.84741, 11069.9834, 0, 187.909515, 3793.61597, 12330.0967, 20968.8438, 25971.4316, 26566.6719}},
{11, 1.2243, {26.0988693, 5.3053894, 2.44860005, 1.59158194, 1.28455877, 1.22552431},
 {14.5760651, 238.804214, 2485.8457, 13426.3496, 0, 251.954651, 6102.91895, 16669.2676, 22509.7363, 23724.2344, 26403.6543}},
{11, 1.0067, {21.460207, 4.36244011, 2.01340008, 1.30870342, 1.05624866, 1.00770664},
 {13.6208639, 315.044312, 4494.86768, 36691.5938, 0, 555.756836, 18990.5801, 68248.2109, 119065.188, 148118.625, 170989.891}},
{10, 0.7434, {337.823822, 13.9598703, 2.97359991, 1.25633335, 0.818379641, 0},
 {1.64230478, 92.996254, 3031.00464, 45459.7344, 204356.516, 0.003935094923, 784.074402, 46293.3555, 177025.781, 205539.109, 0}},
{10, 0.7082, {321.82785, 13.2988701, 2.83279991, 1.19684589, 0.77962935, 0},
 {1.08176517, 91.7048569, 3760.60034, 63706.4531, 331339.969, 0.003110173624, 1058.86145, 71959.8672, 288208.75, 349291.844, 0}},
{10, 0.3712, {168.684692, 6.97054577, 1.48479998, 0.627321661, 0.408639371, 0},
 {0.01475244574, 3.9360373, 518.092651, 30809.9121, 723687.375, 0.0002541629074, 1390.44495, 268268, 1602622, 154225.938, 0}},
{10, 0.3595, {163.367859, 6.75083828, 1.43799996, 0.607548833, 0.395759314, 0},
 {0.01651385054, 4.56706667, 625.223999, 38389.2852, 933006.688, 0.0003638048947, 1922.26392, 372285.031, 2212278.25, 202474.141, 0}},
{10, 0.3096, {140.691757, 5.81379557, 1.23839998, 0.523218691, 0.340826362, 0},
 {5.39867878, 59.2882462, 502.305939, 3254.1792, 16468.6211, 1.38925898, 1076.47644, 12352.8877, 33862.8359, 45490.0625, 0}},
{10, 0.2336, {106.165833, 4.38708353, 0.934495151, 0.394820213, 0.257187188, 0},
 {3.50052953, 56.288475, 589.689575, 3624.10645, 13284.4854, 1.60460746, 2043.36145, 14154.667, 39618.5391, 134922.734, 0}},
{10, 0.1831, {83.1835709, 3.4373889, 0.732200205, 0.309351444, 0.201512545, 0},
 {3.2330935, 73.07621, 1033.85815, 8852.53125, 50242.8086, 2.8820374, 6937.99805, 75841.0234, 194803.375, 341959.656, 0}},
{10, 0.0966, {43.9338608, 1.81547582, 0.386715561, 0.163385674, 0.106429957, 0},
 {0.330870241, 17.903368, 556.262878, 8182.76172, 50086.0859, 3.14749432, 21223.2441, 79335.6953, 639017, 3930551.75, 0}},
{10, 0.0892, {40.5537643, 1.67580032, 0.356963187, 0.150815427, 0.09824166447, 0},
 {0.215997905, 17.4952431, 682.322693, 11258.5234, 75960.1172, 3.5651474, 35011.4727, 137911.125, 1001618.38, 6357603, 0}},
{10, 0.0454, {20.6577549, 0.853638947, 0.181834131, 0.07682414353, 0.05004349723, 0},
 {1.13140631, 12.0350523, 100.865013, 670.138062, 3754.40869, 20.0664749, 4731.21875, 31794.8516, 81577.9766, 277154.563, 0}},
{10, 0.0285, {12.9717808, 0.536032021, 0.114180483, 0.04824076593, 0.03142419457, 0},
 {0.630836248, 9.71265221, 99.9530487, 630.238342, 2609.59546, 47.1750145, 5376.10059, 160318.656, 2133790.75, 8486705, 0}},
{10, 0.0203, {9.23429012, 0.381587923, 0.08128226548, 0.0343414098, 0.02237010561, 0},
 {0.545299411, 11.6766758, 161.000015, 1393.05579, 8330.9082, 148.432877, 30669.6621, 333496.375, 4265278.5, 21050740, 0}},

// Z=92,  24 orbitals
{10, 115.606, {533.762024, 240.655319, 163.491714, 131.810898, 118.416931, 0},
 {0, 0, 0, 0, 0, 28.8420296, 216.662613, 581.247925, 1000.755, 1306.55957, 0}},
{11, 21.7574, {463.810791, 94.2836533, 43.514801, 28.2844772, 22.8282757, 21.7791576},
 {347.511749, 3312.69507, 0, 0, 0, 5.04670286, 238.440948, 1294.8396, 2983.78101, 4374.05371, 4702.30908}},
{11, 20.9476, {446.547974, 90.7744598, 41.8951988, 27.2317429, 21.9786186, 20.9685478},
 {275.066284, 5450.26367, 0, 0, 0, 1.6846112, 191.638474, 1667.37158, 5184.47363, 8887, 9914.98242}},
{11, 17.1663, {365.940582, 74.3885498, 34.3325996, 22.3160782, 18.0112076, 17.183466},
 {237.038956, 6878.43262, 0, 0, 0, 1.4338361, 299.443634, 3284.61499, 11500.2021, 21017.8418, 23796.3301}},
{11, 5.548, {118.268829, 24.0417385, 11.0959997, 7.21236372, 5.82106686, 5.55354786},
 {80.0252914, 856.124634, 6198.50488, 0, 0, 31.993351, 1054.08093, 4338.32813, 8489.07422, 11394.7461, 11993.1641}},
{11, 5.1822, {110.470932, 22.4565792, 10.3643999, 6.73682594, 5.43726254, 5.18738222},
 {59.4962463, 980.76001, 9314.13086, 0, 0, 24.4866905, 1463.05823, 7151.396, 14279.6621, 18608.2363, 19801.8965}},
{11, 4.3034, {91.7372208, 18.6483803, 8.60680008, 5.59439182, 4.51520872, 4.3077035},
 {55.1471939, 1308.34705, 18408.3984, 0, 0, 36.0961418, 3308.43286, 19758.125, 46138.4844, 67105.4063, 72027.0781}},
{11, 3.7276, {79.4626694, 16.1532059, 7.45519972, 4.84585571, 3.91106844, 3.73132753},
 {6.25987339, 392.130035, 15519.8467, 0, 0, 6.42884302, 2246.63086, 26677.9121, 92562.4453, 165677.625, 161128.875}},
{11, 3.5517, {75.712944, 15.3909588, 7.10339975, 4.61718655, 3.726511, 3.5552516},
 {4.09870529, 385.030487, 19217.252, 0, 0, 5.21098137, 2972.62866, 39420.1914, 142481.063, 259586.344, 218465.656}},
{11, 1.4408, {30.7140846, 6.24357176, 2.88160014, 1.87303054, 1.5117147, 1.44224083},
 {21.8197193, 238.938248, 1962.87341, 11122.5977, 0, 179.128845, 3616.70117, 11754.8545, 20010.6035, 24775.6016, 25608.8555}},
{11, 1.2726, {27.1285, 5.51469278, 2.54519987, 1.65437162, 1.33523607, 1.27387261},
 {15.5962124, 250.807953, 2555.2627, 13389.1104, 0, 241.629105, 5830.25635, 15898.3486, 21466.627, 22644.9102, 25456.2285}},
{11, 1.0449, {22.2745323, 4.52797604, 2.08979988, 1.35836315, 1.09632885, 1.04594493},
 {14.4561348, 330.642548, 4663.99316, 37632.7266, 0, 529.207092, 18250.4551, 66057.5, 115953.281, 145725.063, 141866.953}},
{10, 0.7804, {354.637756, 14.6546717, 3.12159991, 1.31886268, 0.859111428, 0},
 {1.79938984, 100.411514, 3217.19214, 47229.0898, 203494.406, 0.003538780846, 720.358765, 43531.6289, 171140.234, 204807.703, 0}},
{10, 0.7377, {335.233551, 13.8528328, 2.95079994, 1.24670041, 0.812104702, 0},
 {1.18061268, 98.8114548, 3986.34106, 66153.5703, 333877.531, 0.002970987931, 987.69397, 68520.5469, 280925.094, 351001.25, 0}},
{10, 0.3913, {177.818756, 7.34799194, 1.56519997, 0.661290348, 0.430766672, 0},
 {0.01679878868, 4.42458105, 573.094788, 33343.6875, 759708.875, 0.0002203819167, 1242.74731, 246723.141, 1552035.75, 155875.063, 0}},
{10, 0.3809, {173.092667, 7.15269661, 1.52359998, 0.643714488, 0.419317722, 0},
 {0.01875866763, 5.12651873, 691.397156, 41594.4375, 982996.938, 0.0003104589123, 1681.1377, 338206.375, 2135759.75, 203558.438, 0}},
{10, 0.3237, {147.099228, 6.07857132, 1.29480004, 0.547047496, 0.356348515, 0},
 {5.68614292, 61.9411621, 520.420532, 3343.23804, 16722.4473, 1.3232187, 1028.75464, 11836.4854, 32616.0156, 44416.0352, 0}},
{10, 0.2593, {117.833893, 4.86924171, 1.03719997, 0.438212574, 0.285453111, 0},
 {3.80143571, 60.0876083, 617.836304, 3722.43945, 13322.1113, 1.32734489, 1789.09875, 12876.5469, 34906.4883, 119073.344, 0}},
{10, 0.1951, {88.6594391, 3.66366792, 0.780400038, 0.329715669, 0.214777872, 0},
 {3.48343849, 77.9058609, 1090.78564, 9263.05469, 52299.1992, 2.57604432, 6471.06104, 72662.7734, 188445.047, 325467.781, 0}},
{10, 0.105, {47.715229, 1.97173309, 0.419999987, 0.177448213, 0.115590341, 0},
 {0.370925695, 19.7877483, 604.217896, 8715.6084, 52328.7227, 2.62085009, 19450.3066, 81746.0547, 509536.844, 3205391, 0}},
{10, 0.0963, {43.7616806, 1.80836093, 0.385199994, 0.162745357, 0.106012858, 0},
 {0.24121362, 19.3023205, 740.695251, 11996.0439, 79631.9766, 2.94554043, 32409.4219, 142580.594, 805837.188, 5159188.5, 0}},
{10, 0.0707, {32.1282539, 1.32763362, 0.282799989, 0.119481795, 0.07783082873, 0},
 {1.1968112, 12.6579933, 105.484665, 698.402161, 3911.9209, 8.77220631, 2555.1228, 20492.0859, 50161.6914, 174645.375, 0}},
{10, 0.0423, {19.2224197, 0.794326723, 0.169200003, 0.07148627937, 0.04656639323, 0},
 {0.68819207, 10.4296017, 105.452431, 653.458801, 2669.10205, 21.7011681, 3465.78979, 66133.5859, 1070067, 5223840, 0}},
{10, 0.0323, {14.678113, 0.606542647, 0.129199997, 0.05458644778, 0.03555779159, 0},
 {0.585494578, 12.4282341, 169.84407, 1460.21033, 8725.37793, 54.2631149, 17623.1504, 145704.859, 1739226.38, 11441855, 0}}
  };
  assert(z >= 3 && z <= 92);
  *n_orb = indices[z-2] - indices[z-3];
  return coefs + indices[z-3];
}

#ifndef GEMMI_FPRIM_ALL_DOUBLE
# if __GNUC__-0 > 4
#  pragma GCC diagnostic pop
# endif
# if defined(_MSC_VER)
#  pragma warning(pop)
# endif
#endif

} // namespace impl_fprim


// Cromer-Liberman calculation of anomalous scattering factors.
// input:
//   z      - atomic number
//   npts   - array length
//   energy - energies in eV
// output:
//   fp     - f' (real part of anomalous scattering)
//   fpp    - f" (imaginary part of anomalous scattering)
inline
void cromer_liberman_for_array(int z, int npts, const double* energy,
                                double* fp, double* fpp) {
  using namespace impl_fprim;
  if (z < 3 || z > 92)
    return;
  int norbz;
  OrbitalCoef* coefs = get_orbital_coefficients(z, &norbz);
  for (int i = 0; i < npts; ++i) {
    double f1, f2;
    cromer(z, energy[i], norbz, coefs, &f1, &f2);
    fp[i] = f1 + calculate_correction(z);
    fpp[i] = f2;
  }
}

// returns fp, fpp is returned through the last argument.
inline double cromer_liberman(int z, double energy, double* fpp) {
  double fp = 0., fpp_ = 0.;
  cromer_liberman_for_array(z, 1, &energy, &fp, &fpp_);
  if (fpp)
    *fpp = fpp_;
  return fp;
}

// out must point to
template <typename Iter>
void add_cl_fprime_for_all_elements(Iter out, double energy) {
  using Real = decltype(*out+0.f);  // used to avoid conversion warning
  for (int z = 1; z <= 92; ++z, ++out)
    *out += (Real) cromer_liberman(z, energy, nullptr);
}

} // namespace gemmi
#endif
