# creates: files.png, lines.png
import datetime as dt
import subprocess
from collections import defaultdict
from pathlib import Path

import matplotlib.pyplot as plt
import matplotlib.ticker as tck


LABELS = [
    'Python (code)',
    'Python (documentation)',
    'Python (tests)',
    'C-code',
    'C++-code',
    'Documentation',
    'Other',
    'Fortran']


def plot_count():
    years = []
    files = defaultdict(list)
    lines = defaultdict(list)
    for year, month, (F, L) in DATA:
        years.append(year + (month - 1) / 12)
        for name, f, l in zip(LABELS, F, L):
            files[name].append(f)
            lines[name].append(l)

    for name, things in [('files', files), ('lines', lines)]:
        fig, ax = plt.subplots()
        ax.stackplot(years, things.values(), labels=things.keys(), alpha=0.8)
        ax.legend(loc='upper left', reverse=True)
        ax.set_xlabel('Year')
        ax.set_ylabel('Number of ' + name)
        ax.xaxis.set_major_locator(tck.MultipleLocator(4))
        plt.savefig(name + '.png')
        # plt.show()


def count_all(year=2005, month=11):
    now = dt.date.today()
    stop = now.year, now.month

    data = []
    while (year, month) <= stop:
        hash = subprocess.check_output(
            'git rev-list -n 1 --before="{}-{}-01 12:00" master'
            .format(year, month), shell=True).strip()
        print(year, month, hash)
        subprocess.call(['git', 'checkout', hash])
        data.append((year, month, count(silent=True)))
        month += 1
        if month == 13:
            month = 1
            year += 1
    return data


def count(silent=False):
    lines = defaultdict(int)
    files = defaultdict(int)
    for p in Path().glob('**/*.*'):
        s = str(p)
        if s.startswith('.git'):
            continue
        if p.suffix in ['.pyc', '.png', '.ico', '.bib', '.traj']:
            continue
        if s.startswith('doc'):
            x = 'doc'
        elif s.startswith('gpaw/test'):
            x = 'test'
        else:
            x = 'code'
        try:
            L = len(p.read_text().splitlines())
        except UnicodeDecodeError:
            print('UNICODE error:', p)
            L = 0
        if p.suffix in ['.c', '.h', '.py', '.rst', '.cpp']:
            if x == 'code' and p.suffix == '.rst':
                x = 'doc'
            lines[x + p.suffix] += L
            files[x + p.suffix] += 1
        else:
            lines['other'] += L
            files['other'] += 1
            if not silent:
                print(p)

    return [
        (x['code.py'],
         x['doc.py'],
         x['test.py'],
         x['code.h'] + x['code.c'],
         x['code.cpp'],
         x['code.rst'] + x['doc.rst'],
         x['other'],
         0) for x in [files, lines]]


DATA = [
    (2005, 11,
     ((106, 0, 0, 33, 0, 0, 2, 0),
      (13306, 0, 0, 4193, 0, 0, 62, 0))),
    (2005, 12,
     ((106, 0, 0, 33, 0, 0, 2, 0),
      (14125, 0, 0, 4364, 0, 0, 61, 0))),
    (2006, 1,
     ((127, 0, 0, 34, 0, 0, 2, 0),
      (15820, 0, 0, 4454, 0, 0, 61, 0))),
    (2006, 2,
     ((114, 0, 0, 36, 0, 0, 2, 0),
      (15154, 0, 0, 5244, 0, 0, 61, 0))),
    (2006, 3,
     ((116, 0, 0, 36, 0, 0, 2, 0),
      (15695, 0, 0, 5255, 0, 0, 61, 0))),
    (2006, 4,
     ((119, 0, 0, 36, 0, 0, 2, 0),
      (16195, 0, 0, 5334, 0, 0, 61, 0))),
    (2006, 5,
     ((125, 0, 0, 36, 0, 0, 2, 0),
      (17452, 0, 0, 5129, 0, 0, 61, 0))),
    (2006, 6,
     ((129, 0, 0, 36, 0, 0, 2, 0),
      (17640, 0, 0, 5135, 0, 0, 61, 0))),
    (2006, 7,
     ((131, 0, 0, 36, 0, 0, 2, 0),
      (17903, 0, 0, 5597, 0, 0, 61, 0))),
    (2006, 8,
     ((121, 0, 0, 36, 0, 0, 2, 0),
      (17526, 0, 0, 5632, 0, 0, 59, 0))),
    (2006, 9,
     ((121, 0, 0, 36, 0, 0, 2, 0),
      (18075, 0, 0, 5636, 0, 0, 50, 0))),
    (2006, 10,
     ((122, 0, 0, 36, 0, 0, 2, 0),
      (18425, 0, 0, 5714, 0, 0, 50, 0))),
    (2006, 11,
     ((123, 0, 0, 36, 0, 0, 2, 0),
      (18236, 0, 0, 5792, 0, 0, 50, 0))),
    (2006, 12,
     ((123, 0, 0, 38, 0, 0, 2, 0),
      (18370, 0, 0, 6498, 0, 0, 50, 0))),
    (2007, 1,
     ((128, 0, 0, 40, 0, 0, 2, 0),
      (19808, 0, 0, 6709, 0, 0, 50, 0))),
    (2007, 2,
     ((134, 0, 0, 41, 0, 0, 2, 0),
      (20847, 0, 0, 6995, 0, 0, 50, 0))),
    (2007, 3,
     ((134, 0, 0, 41, 0, 0, 2, 0),
      (21221, 0, 0, 6995, 0, 0, 50, 0))),
    (2007, 4,
     ((137, 0, 2, 41, 0, 0, 2, 0),
      (21202, 0, 228, 6996, 0, 0, 50, 0))),
    (2007, 5,
     ((146, 0, 2, 41, 0, 0, 2, 0),
      (22982, 0, 228, 7023, 0, 0, 50, 0))),
    (2007, 6,
     ((167, 0, 3, 41, 0, 0, 2, 0),
      (25706, 0, 2195, 7052, 0, 0, 50, 0))),
    (2007, 7,
     ((177, 0, 7, 42, 0, 0, 2, 0),
      (26864, 0, 3327, 7175, 0, 0, 50, 0))),
    (2007, 8,
     ((185, 0, 7, 70, 0, 0, 21, 0),
      (27971, 0, 3378, 12027, 0, 0, 8214, 0))),
    (2007, 9,
     ((193, 0, 7, 72, 0, 0, 21, 0),
      (29121, 0, 3378, 12191, 0, 0, 8214, 0))),
    (2007, 10,
     ((205, 0, 8, 74, 0, 0, 22, 0),
      (31523, 0, 3614, 12346, 0, 0, 8216, 0))),
    (2007, 11,
     ((216, 0, 8, 75, 0, 0, 22, 0),
      (33147, 0, 3534, 12462, 0, 0, 8216, 0))),
    (2007, 12,
     ((236, 0, 8, 101, 0, 0, 29, 0),
      (36302, 0, 3537, 14775, 0, 0, 11437, 0))),
    (2008, 1,
     ((237, 0, 13, 101, 0, 0, 28, 0),
      (36516, 0, 4301, 14825, 0, 0, 11437, 0))),
    (2008, 2,
     ((254, 0, 13, 102, 0, 0, 28, 0),
      (36849, 0, 4301, 14976, 0, 0, 11437, 0))),
    (2008, 3,
     ((263, 0, 13, 111, 0, 0, 29, 0),
      (38870, 0, 4378, 15887, 0, 0, 12085, 0))),
    (2008, 4,
     ((267, 0, 11, 111, 0, 0, 29, 0),
      (39597, 0, 3778, 15981, 0, 0, 12085, 0))),
    (2008, 5,
     ((271, 0, 11, 111, 0, 0, 29, 0),
      (40821, 0, 3778, 16039, 0, 0, 12085, 0))),
    (2008, 6,
     ((278, 0, 11, 121, 0, 0, 31, 0),
      (43266, 0, 3835, 17752, 0, 0, 12231, 0))),
    (2008, 7,
     ((295, 3, 12, 122, 0, 8, 31, 0),
      (44874, 380, 4269, 17907, 0, 420, 12231, 0))),
    (2008, 8,
     ((316, 25, 12, 122, 0, 79, 34, 0),
      (46157, 1531, 4269, 18081, 0, 7606, 13327, 0))),
    (2008, 9,
     ((274, 72, 12, 122, 0, 83, 35, 0),
      (45762, 2644, 4293, 18097, 0, 7686, 12758, 0))),
    (2008, 10,
     ((276, 72, 12, 122, 0, 84, 35, 0),
      (46529, 2650, 4293, 18308, 0, 7806, 12758, 0))),
    (2008, 11,
     ((279, 75, 13, 125, 0, 87, 42, 0),
      (47348, 2683, 4498, 19675, 0, 8410, 12930, 0))),
    (2008, 12,
     ((289, 75, 13, 125, 0, 87, 42, 0),
      (48326, 2685, 4498, 19992, 0, 8644, 12930, 0))),
    (2009, 1,
     ((294, 80, 14, 125, 0, 89, 45, 0),
      (49919, 3180, 4583, 20033, 0, 9157, 13010, 0))),
    (2009, 2,
     ((293, 80, 13, 125, 0, 89, 45, 0),
      (49758, 3180, 2513, 20052, 0, 9218, 13010, 0))),
    (2009, 3,
     ((336, 86, 13, 128, 0, 115, 52, 0),
      (52642, 3926, 2505, 20947, 0, 9698, 13082, 0))),
    (2009, 4,
     ((348, 93, 13, 129, 0, 119, 58, 0),
      (56196, 4279, 2505, 21970, 0, 10092, 13988, 0))),
    (2009, 5,
     ((361, 93, 13, 129, 0, 120, 58, 0),
      (58840, 4283, 2505, 22108, 0, 10126, 14005, 0))),
    (2009, 6,
     ((369, 99, 13, 129, 0, 121, 59, 0),
      (60743, 4402, 2505, 22090, 0, 10753, 14074, 0))),
    (2009, 7,
     ((388, 102, 14, 131, 0, 123, 60, 0),
      (66329, 4479, 2827, 23168, 0, 11329, 14138, 0))),
    (2009, 8,
     ((411, 107, 15, 131, 0, 125, 62, 0),
      (75494, 4682, 2975, 23809, 0, 11706, 14164, 0))),
    (2009, 9,
     ((424, 105, 15, 132, 0, 125, 62, 0),
      (78144, 4807, 2975, 24101, 0, 11758, 14173, 0))),
    (2009, 10,
     ((296, 112, 147, 132, 0, 126, 62, 0),
      (68677, 5003, 10666, 24172, 0, 11866, 14176, 0))),
    (2009, 11,
     ((278, 116, 151, 133, 0, 126, 62, 0),
      (68312, 5480, 11989, 24966, 0, 11975, 14181, 0))),
    (2009, 12,
     ((288, 120, 158, 134, 0, 130, 64, 0),
      (72486, 5557, 12578, 25500, 0, 12266, 14235, 0))),
    (2010, 1,
     ((289, 121, 164, 136, 0, 130, 64, 0),
      (73026, 5587, 13042, 26471, 0, 12395, 14239, 0))),
    (2010, 2,
     ((292, 125, 166, 136, 0, 132, 64, 0),
      (75209, 5830, 13035, 26347, 0, 12917, 14237, 0))),
    (2010, 3,
     ((297, 135, 174, 138, 0, 135, 65, 0),
      (77054, 6113, 13879, 27103, 0, 13386, 15860, 0))),
    (2010, 4,
     ((305, 138, 177, 140, 0, 136, 65, 0),
      (78573, 6750, 14066, 27975, 0, 13490, 15860, 0))),
    (2010, 5,
     ((286, 141, 231, 140, 0, 137, 74, 0),
      (78291, 6916, 17432, 28011, 0, 14073, 20846, 0))),
    (2010, 6,
     ((295, 159, 227, 140, 0, 137, 75, 0),
      (80800, 7234, 17355, 28017, 0, 14278, 20879, 0))),
    (2010, 7,
     ((304, 181, 225, 141, 0, 139, 75, 0),
      (78777, 8421, 15659, 28098, 0, 15260, 20879, 0))),
    (2010, 8,
     ((313, 183, 229, 141, 0, 142, 75, 0),
      (81170, 8587, 15844, 28184, 0, 15661, 20879, 0))),
    (2010, 9,
     ((313, 184, 236, 136, 0, 145, 76, 0),
      (82056, 8589, 16704, 27031, 0, 16128, 21085, 0))),
    (2010, 10,
     ((314, 185, 236, 136, 0, 145, 76, 0),
      (82466, 8636, 16717, 27034, 0, 16194, 21116, 0))),
    (2010, 11,
     ((323, 185, 238, 135, 0, 146, 76, 0),
      (81978, 8631, 16790, 26672, 0, 16415, 21116, 0))),
    (2010, 12,
     ((325, 191, 245, 133, 0, 150, 83, 0),
      (81963, 8784, 17600, 25938, 0, 16723, 21693, 0))),
    (2011, 1,
     ((323, 193, 252, 146, 0, 151, 104, 0),
      (82524, 8930, 17881, 27847, 0, 16824, 29672, 0))),
    (2011, 2,
     ((305, 196, 267, 133, 0, 150, 83, 0),
      (77381, 9076, 19253, 26057, 0, 17702, 21862, 0))),
    (2011, 3,
     ((304, 197, 269, 133, 0, 152, 84, 0),
      (77779, 9183, 19441, 26073, 0, 18003, 21885, 0))),
    (2011, 4,
     ((307, 197, 270, 133, 0, 152, 86, 0),
      (78511, 9188, 19591, 26074, 0, 18037, 21960, 0))),
    (2011, 5,
     ((305, 197, 269, 133, 0, 152, 85, 0),
      (78578, 9183, 19459, 26652, 0, 18015, 21925, 0))),
    (2011, 6,
     ((311, 225, 276, 131, 0, 156, 87, 0),
      (79909, 9921, 19840, 25281, 0, 19136, 21967, 0))),
    (2011, 7,
     ((311, 226, 282, 131, 0, 157, 91, 0),
      (80934, 9966, 20036, 25330, 0, 19195, 22132, 0))),
    (2011, 8,
     ((313, 228, 285, 131, 0, 159, 91, 0),
      (81648, 10013, 20188, 25330, 0, 19625, 22132, 0))),
    (2011, 9,
     ((314, 228, 289, 131, 0, 158, 91, 0),
      (81956, 10061, 20291, 25330, 0, 18918, 22132, 0))),
    (2011, 10,
     ((316, 228, 298, 133, 0, 158, 91, 0),
      (83051, 10062, 20674, 26243, 0, 19025, 22132, 0))),
    (2011, 11,
     ((320, 233, 299, 133, 0, 160, 91, 0),
      (83683, 10262, 20761, 26231, 0, 19223, 22133, 0))),
    (2011, 12,
     ((323, 235, 302, 133, 0, 160, 91, 0),
      (84791, 10306, 20978, 26528, 0, 19220, 22133, 0))),
    (2012, 1,
     ((323, 237, 312, 133, 0, 162, 91, 0),
      (85808, 10340, 21830, 26528, 0, 19256, 22133, 0))),
    (2012, 2,
     ((324, 242, 316, 135, 0, 162, 91, 0),
      (86187, 10496, 22164, 26911, 0, 19316, 22136, 0))),
    (2012, 3,
     ((324, 244, 327, 132, 0, 162, 91, 0),
      (87222, 10496, 22663, 26410, 0, 19409, 22136, 0))),
    (2012, 4,
     ((326, 240, 345, 133, 0, 185, 86, 0),
      (89319, 10483, 23482, 26552, 0, 20058, 21965, 0))),
    (2012, 5,
     ((330, 240, 358, 133, 0, 185, 86, 0),
      (89703, 10483, 24068, 26552, 0, 20071, 21965, 0))),
    (2012, 6,
     ((332, 241, 379, 134, 0, 186, 91, 0),
      (90990, 10500, 25037, 26799, 0, 20111, 22273, 0))),
    (2012, 7,
     ((333, 244, 380, 134, 0, 188, 98, 0),
      (91468, 10598, 25039, 26800, 0, 20443, 32261, 0))),
    (2012, 8,
     ((333, 244, 380, 134, 0, 190, 98, 0),
      (91632, 10598, 25046, 26806, 0, 20581, 32261, 0))),
    (2012, 9,
     ((333, 248, 380, 134, 0, 193, 99, 0),
      (91722, 10750, 25046, 26806, 0, 20844, 32292, 0))),
    (2012, 10,
     ((333, 248, 404, 134, 0, 194, 99, 0),
      (91847, 10781, 26903, 26806, 0, 20947, 32292, 0))),
    (2012, 11,
     ((333, 253, 410, 134, 0, 195, 100, 0),
      (92191, 11057, 26914, 26806, 0, 21062, 32300, 0))),
    (2012, 12,
     ((335, 253, 412, 134, 0, 195, 100, 0),
      (92641, 11057, 27472, 26806, 0, 21104, 32300, 0))),
    (2013, 1,
     ((335, 258, 413, 134, 0, 196, 103, 0),
      (92765, 11837, 27528, 26839, 0, 21171, 32342, 0))),
    (2013, 2,
     ((336, 261, 416, 136, 0, 198, 104, 0),
      (93599, 11901, 27751, 27486, 0, 21392, 33094, 0))),
    (2013, 3,
     ((336, 262, 417, 136, 0, 199, 104, 0),
      (93545, 11938, 27746, 27486, 0, 21483, 23485, 0))),
    (2013, 4,
     ((335, 263, 420, 136, 0, 199, 104, 0),
      (93680, 11971, 27921, 27490, 0, 21524, 23486, 0))),
    (2013, 5,
     ((340, 260, 413, 74, 0, 198, 106, 0),
      (95696, 11865, 27501, 21177, 0, 21378, 30186, 0))),
    (2013, 6,
     ((336, 271, 422, 137, 0, 199, 105, 0),
      (94541, 12300, 28128, 27746, 0, 21936, 23756, 0))),
    (2013, 7,
     ((337, 275, 425, 61, 0, 198, 77, 0),
      (97498, 12420, 28203, 18152, 0, 21750, 11562, 0))),
    (2013, 8,
     ((337, 280, 425, 61, 0, 199, 77, 0),
      (97502, 12504, 28217, 18152, 0, 21879, 11562, 0))),
    (2013, 9,
     ((337, 280, 425, 61, 0, 199, 77, 0),
      (97538, 12350, 28218, 18207, 0, 21879, 11561, 0))),
    (2013, 10,
     ((339, 280, 428, 61, 0, 199, 77, 0),
      (98265, 12366, 28515, 18254, 0, 21880, 11632, 0))),
    (2013, 11,
     ((339, 280, 428, 61, 0, 199, 77, 0),
      (98243, 12359, 28524, 18250, 0, 21890, 11632, 0))),
    (2013, 12,
     ((343, 281, 432, 61, 0, 200, 78, 0),
      (98571, 12376, 28741, 18251, 0, 21999, 11654, 0))),
    (2014, 1,
     ((347, 284, 435, 61, 0, 200, 78, 0),
      (100495, 12392, 29022, 18263, 0, 21977, 11654, 0))),
    (2014, 2,
     ((339, 281, 423, 61, 0, 199, 78, 0),
      (99440, 12377, 28528, 18250, 0, 22006, 11654, 0))),
    (2014, 3,
     ((339, 281, 425, 61, 0, 199, 78, 0),
      (99573, 12376, 28619, 18251, 0, 22009, 11654, 0))),
    (2014, 4,
     ((339, 282, 435, 61, 0, 199, 81, 0),
      (99622, 12425, 28748, 18251, 0, 22039, 12808, 0))),
    (2014, 5,
     ((344, 281, 428, 61, 0, 199, 81, 0),
      (101271, 12377, 28887, 18399, 0, 22009, 11991, 0))),
    (2014, 6,
     ((338, 293, 436, 60, 0, 200, 81, 0),
      (98855, 12575, 28747, 17658, 0, 22266, 12808, 0))),
    (2014, 7,
     ((341, 294, 440, 60, 0, 200, 87, 0),
      (99675, 12618, 28906, 17658, 0, 22291, 24203, 0))),
    (2014, 8,
     ((348, 311, 442, 60, 0, 199, 88, 0),
      (100766, 12937, 29054, 17671, 0, 22322, 24219, 0))),
    (2014, 9,
     ((348, 319, 445, 60, 0, 203, 90, 0),
      (100955, 13175, 29193, 17671, 0, 23049, 25777, 0))),
    (2014, 10,
     ((280, 328, 444, 57, 0, 203, 90, 0),
      (98185, 13399, 28096, 17302, 0, 23212, 26132, 0))),
    (2014, 11,
     ((291, 341, 454, 57, 0, 208, 91, 0),
      (102579, 14039, 28695, 17347, 0, 23691, 26142, 0))),
    (2014, 12,
     ((293, 342, 458, 57, 0, 209, 91, 0),
      (103641, 14063, 28936, 17347, 0, 23859, 26162, 0))),
    (2015, 1,
     ((292, 346, 456, 57, 0, 210, 91, 0),
      (103047, 14342, 28849, 17376, 0, 23947, 26162, 0))),
    (2015, 2,
     ((292, 346, 458, 57, 0, 210, 92, 0),
      (103150, 14350, 28973, 17383, 0, 23883, 26217, 0))),
    (2015, 3,
     ((302, 351, 462, 57, 0, 210, 93, 0),
      (104847, 14551, 29224, 17464, 0, 23860, 26238, 0))),
    (2015, 4,
     ((303, 352, 464, 57, 0, 211, 93, 0),
      (106041, 14572, 29450, 17464, 0, 23845, 26237, 0))),
    (2015, 5,
     ((305, 359, 466, 57, 0, 212, 93, 0),
      (107040, 14226, 29633, 17494, 0, 23894, 26237, 0))),
    (2015, 6,
     ((314, 362, 483, 60, 0, 213, 93, 0),
      (109168, 14299, 31851, 18308, 0, 24043, 26350, 0))),
    (2015, 7,
     ((319, 386, 484, 60, 0, 215, 95, 0),
      (109587, 14862, 31392, 18414, 0, 24646, 26419, 0))),
    (2015, 8,
     ((317, 389, 486, 60, 0, 217, 95, 0),
      (108763, 14967, 31459, 18345, 0, 25031, 26419, 0))),
    (2015, 9,
     ((320, 389, 481, 61, 0, 216, 95, 0),
      (109752, 14962, 30905, 18563, 0, 24883, 26419, 0))),
    (2015, 10,
     ((320, 389, 482, 61, 0, 216, 95, 0),
      (110111, 14962, 30953, 18552, 0, 24883, 26419, 0))),
    (2015, 11,
     ((320, 406, 485, 61, 0, 217, 96, 0),
      (112136, 15292, 31090, 18519, 0, 25097, 26529, 0))),
    (2015, 12,
     ((320, 410, 513, 61, 0, 218, 96, 0),
      (112476, 15407, 31040, 18533, 0, 25410, 26529, 0))),
    (2016, 1,
     ((321, 412, 514, 61, 0, 219, 96, 0),
      (112728, 15439, 31091, 18533, 0, 25622, 26529, 0))),
    (2016, 2,
     ((323, 412, 515, 61, 0, 219, 96, 0),
      (113851, 15439, 31122, 18533, 0, 25624, 26529, 0))),
    (2016, 3,
     ((319, 413, 513, 61, 0, 223, 95, 0),
      (114046, 15482, 31014, 18550, 0, 25764, 26484, 0))),
    (2016, 4,
     ((319, 412, 513, 61, 0, 224, 95, 0),
      (113624, 15435, 31044, 18550, 0, 25469, 25924, 0))),
    (2016, 5,
     ((319, 426, 513, 61, 0, 225, 96, 0),
      (114130, 15613, 31016, 18564, 0, 25657, 23075, 0))),
    (2016, 6,
     ((308, 424, 492, 61, 0, 227, 96, 0),
      (106563, 15462, 29680, 18564, 0, 25625, 21585, 0))),
    (2016, 7,
     ((307, 423, 487, 61, 0, 226, 94, 0),
      (105584, 15420, 29575, 18564, 0, 25600, 21543, 0))),
    (2016, 8,
     ((308, 423, 482, 61, 0, 226, 94, 0),
      (104387, 15420, 28145, 18564, 0, 25585, 21543, 0))),
    (2016, 9,
     ((307, 424, 482, 61, 0, 227, 94, 0),
      (103993, 15399, 28128, 18564, 0, 25851, 21543, 0))),
    (2016, 10,
     ((309, 424, 484, 61, 0, 227, 94, 0),
      (105549, 15394, 28250, 18564, 0, 25864, 21543, 0))),
    (2016, 11,
     ((309, 425, 485, 61, 0, 227, 94, 0),
      (105653, 15405, 28295, 18564, 0, 25866, 21558, 0))),
    (2016, 12,
     ((310, 433, 484, 61, 0, 227, 95, 0),
      (105911, 15893, 28338, 18568, 0, 25938, 21620, 0))),
    (2017, 1,
     ((296, 382, 482, 60, 0, 225, 95, 0),
      (96932, 13763, 28200, 17870, 0, 25275, 21920, 0))),
    (2017, 2,
     ((297, 398, 482, 60, 0, 227, 96, 0),
      (97297, 13972, 28202, 17870, 0, 25329, 21922, 0))),
    (2017, 3,
     ((298, 401, 484, 60, 0, 227, 96, 0),
      (97433, 14021, 28218, 17870, 0, 25408, 21922, 0))),
    (2017, 4,
     ((298, 401, 484, 60, 0, 227, 97, 0),
      (97533, 14043, 28220, 17870, 0, 25413, 21937, 0))),
    (2017, 5,
     ((301, 400, 486, 60, 0, 227, 97, 0),
      (97545, 13979, 28252, 17879, 0, 25421, 21942, 0))),
    (2017, 6,
     ((303, 402, 487, 61, 0, 227, 97, 0),
      (99352, 14135, 28345, 18074, 0, 25630, 21942, 0))),
    (2017, 7,
     ((303, 402, 488, 61, 0, 227, 97, 0),
      (99625, 14118, 28343, 18096, 0, 25684, 21942, 0))),
    (2017, 8,
     ((304, 402, 491, 61, 0, 228, 97, 0),
      (99868, 14094, 28509, 18096, 0, 25950, 21944, 0))),
    (2017, 9,
     ((304, 402, 493, 61, 0, 228, 97, 0),
      (100037, 14082, 28584, 18096, 0, 25972, 21943, 0))),
    (2017, 10,
     ((310, 403, 493, 61, 0, 228, 97, 0),
      (100435, 14098, 28397, 18096, 0, 25995, 22096, 0))),
    (2017, 11,
     ((311, 403, 498, 61, 0, 229, 97, 0),
      (101445, 14097, 28504, 18347, 0, 26088, 22096, 0))),
    (2017, 12,
     ((312, 403, 502, 61, 0, 230, 97, 0),
      (102316, 14101, 28609, 18348, 0, 26193, 22085, 0))),
    (2018, 1,
     ((315, 405, 509, 61, 0, 231, 97, 0),
      (103333, 14153, 29111, 18348, 0, 26291, 22087, 0))),
    (2018, 2,
     ((316, 405, 510, 61, 0, 232, 97, 0),
      (103455, 14171, 29177, 18348, 0, 26319, 22088, 0))),
    (2018, 3,
     ((330, 418, 520, 61, 0, 232, 97, 0),
      (105613, 14507, 29772, 18364, 0, 26514, 22088, 0))),
    (2018, 4,
     ((330, 418, 523, 61, 0, 232, 95, 0),
      (105535, 14505, 33049, 18364, 0, 26522, 18855, 0))),
    (2018, 5,
     ((331, 419, 525, 61, 0, 232, 95, 0),
      (106232, 14551, 33252, 18444, 0, 26501, 18855, 0))),
    (2018, 6,
     ((330, 411, 515, 61, 0, 231, 96, 0),
      (106128, 14538, 32363, 18444, 0, 26219, 19070, 0))),
    (2018, 7,
     ((331, 414, 517, 61, 0, 237, 96, 0),
      (106610, 14742, 32462, 18444, 0, 26838, 19070, 0))),
    (2018, 8,
     ((332, 428, 519, 61, 0, 245, 111, 0),
      (106704, 15136, 32525, 18444, 0, 27302, 27822, 0))),
    (2018, 9,
     ((344, 427, 522, 61, 0, 245, 111, 0),
      (113184, 15119, 32830, 18444, 0, 27303, 28121, 0))),
    (2018, 10,
     ((333, 426, 523, 61, 0, 245, 111, 0),
      (107296, 15053, 32894, 18127, 0, 27303, 27805, 0))),
    (2018, 11,
     ((332, 429, 535, 61, 0, 245, 111, 0),
      (108262, 15145, 33561, 18274, 0, 27319, 27805, 0))),
    (2018, 12,
     ((349, 427, 522, 61, 0, 245, 111, 0),
      (114849, 15119, 32830, 18444, 0, 27303, 28121, 0))),
    (2019, 1,
     ((335, 441, 544, 62, 0, 248, 113, 0),
      (109789, 15584, 34076, 18555, 0, 28017, 27877, 0))),
    (2019, 2,
     ((342, 453, 544, 62, 0, 251, 113, 0),
      (112446, 15905, 34092, 18553, 0, 28419, 27877, 0))),
    (2019, 3,
     ((338, 458, 541, 62, 0, 252, 112, 0),
      (112892, 16070, 33818, 18558, 0, 28821, 27877, 0))),
    (2019, 4,
     ((338, 457, 541, 62, 0, 252, 112, 0),
      (113005, 16066, 33818, 18585, 0, 28833, 27880, 0))),
    (2019, 5,
     ((338, 457, 542, 62, 0, 252, 112, 0),
      (112888, 16065, 33954, 18587, 0, 28793, 27880, 0))),
    (2019, 6,
     ((343, 457, 540, 62, 0, 252, 112, 0),
      (115251, 16065, 33771, 18587, 0, 28801, 27880, 0))),
    (2019, 7,
     ((343, 457, 540, 62, 0, 252, 112, 0),
      (115912, 16065, 33784, 18587, 0, 28801, 27880, 0))),
    (2019, 8,
     ((341, 457, 546, 63, 0, 252, 108, 0),
      (113303, 16067, 34154, 19188, 0, 28888, 27852, 0))),
    (2019, 9,
     ((341, 457, 547, 63, 0, 252, 108, 0),
      (112924, 16060, 34188, 19239, 0, 28935, 29621, 0))),
    (2019, 10,
     ((352, 462, 546, 61, 0, 254, 106, 0),
      (117994, 16271, 34044, 18414, 0, 29242, 29610, 0))),
    (2019, 11,
     ((340, 458, 543, 61, 0, 252, 106, 0),
      (113006, 16045, 33826, 18414, 0, 28872, 29608, 0))),
    (2019, 12,
     ((345, 459, 544, 61, 0, 252, 106, 0),
      (116201, 16079, 34058, 18414, 0, 28910, 29608, 0))),
    (2020, 1,
     ((349, 459, 549, 61, 0, 252, 106, 0),
      (116624, 16077, 34290, 18036, 0, 28923, 29608, 0))),
    (2020, 2,
     ((350, 471, 551, 59, 0, 253, 108, 0),
      (117196, 16367, 34425, 17360, 0, 29251, 29722, 0))),
    (2020, 3,
     ((345, 471, 536, 59, 0, 253, 115, 0),
      (115021, 16326, 33264, 17368, 0, 29238, 29883, 0))),
    (2020, 4,
     ((350, 471, 536, 59, 0, 253, 115, 0),
      (115274, 16325, 33286, 17388, 0, 29238, 29883, 0))),
    (2020, 5,
     ((357, 473, 539, 59, 0, 253, 115, 0),
      (116472, 16414, 33621, 17388, 0, 29248, 29879, 0))),
    (2020, 6,
     ((350, 465, 536, 59, 0, 228, 104, 0),
      (115120, 15509, 32762, 17385, 0, 28100, 29544, 0))),
    (2020, 7,
     ((350, 466, 535, 58, 0, 229, 104, 0),
      (114815, 15536, 32690, 17169, 0, 28121, 29545, 0))),
    (2020, 8,
     ((350, 465, 538, 58, 0, 228, 104, 0),
      (114951, 15514, 33088, 17169, 0, 28115, 29542, 0))),
    (2020, 9,
     ((350, 484, 542, 58, 0, 229, 90, 0),
      (115940, 20857, 33163, 17189, 0, 28213, 20817, 0))),
    (2020, 10,
     ((362, 479, 553, 58, 0, 229, 82, 0),
      (118688, 20506, 34056, 17189, 0, 28144, 20599, 0))),
    (2020, 11,
     ((389, 474, 552, 59, 0, 255, 108, 0),
      (132099, 16521, 34444, 17360, 0, 29528, 29722, 0))),
    (2020, 12,
     ((371, 487, 562, 58, 0, 230, 82, 0),
      (120288, 20774, 34641, 17189, 0, 28424, 20602, 0))),
    (2021, 1,
     ((372, 490, 565, 58, 0, 231, 82, 0),
      (120430, 20830, 34724, 17189, 0, 28458, 20621, 0))),
    (2021, 2,
     ((373, 488, 569, 62, 0, 231, 83, 0),
      (119600, 20732, 35137, 17312, 0, 28579, 20713, 0))),
    (2021, 3,
     ((370, 498, 567, 62, 0, 231, 84, 0),
      (119193, 20694, 35119, 17312, 0, 28578, 20796, 0))),
    (2021, 4,
     ((376, 510, 579, 62, 0, 235, 81, 0),
      (120288, 21012, 35714, 17327, 0, 29020, 21605, 0))),
    (2021, 5,
     ((376, 490, 567, 73, 0, 231, 101, 0),
      (122380, 20839, 34728, 20781, 0, 28574, 28023, 0))),
    (2021, 6,
     ((378, 514, 589, 62, 0, 236, 81, 0),
      (119772, 21111, 36722, 17327, 0, 29227, 21657, 0))),
    (2021, 7,
     ((408, 537, 600, 62, 0, 237, 85, 0),
      (133664, 21689, 37186, 17327, 0, 29653, 21711, 0))),
    (2021, 8,
     ((379, 539, 594, 62, 0, 238, 85, 0),
      (120330, 21612, 37032, 17327, 0, 29608, 21711, 0))),
    (2021, 9,
     ((387, 539, 596, 62, 0, 239, 84, 0),
      (120889, 21614, 37276, 17327, 0, 29807, 21623, 0))),
    (2021, 10,
     ((412, 537, 600, 62, 0, 251, 83, 0),
      (124143, 21499, 37339, 17339, 0, 30230, 21416, 0))),
    (2021, 11,
     ((397, 547, 603, 62, 0, 251, 83, 0),
      (123981, 22231, 37899, 17327, 0, 30553, 21458, 0))),
    (2021, 12,
     ((426, 555, 612, 62, 0, 254, 84, 0),
      (129286, 22456, 38258, 17339, 0, 30864, 21549, 0))),
    (2022, 1,
     ((429, 570, 616, 62, 0, 255, 85, 0),
      (130611, 23275, 38584, 17339, 0, 31060, 21556, 0))),
    (2022, 2,
     ((443, 574, 616, 62, 0, 256, 86, 0),
      (131140, 23308, 38524, 17339, 0, 31381, 21676, 0))),
    (2022, 3,
     ((438, 571, 615, 62, 0, 256, 85, 0),
      (130710, 23311, 38502, 17339, 0, 31221, 21563, 0))),
    (2022, 4,
     ((449, 576, 622, 62, 0, 255, 86, 0),
      (133137, 23222, 38757, 17339, 0, 30756, 21717, 0))),
    (2022, 5,
     ((451, 577, 628, 62, 0, 255, 86, 0),
      (134173, 23396, 39365, 17339, 0, 30761, 21589, 0))),
    (2022, 6,
     ((453, 582, 627, 62, 0, 256, 86, 0),
      (134135, 23603, 39049, 17339, 0, 30912, 21724, 0))),
    (2022, 7,
     ((471, 582, 639, 62, 0, 259, 86, 0),
      (135487, 23560, 40641, 17296, 0, 31091, 21602, 0))),
    (2022, 8,
     ((413, 544, 607, 62, 0, 240, 85, 0),
      (134456, 21834, 37536, 17327, 0, 29910, 21711, 0))),
    (2022, 9,
     ((478, 591, 641, 62, 0, 260, 86, 0),
      (136664, 24410, 41588, 17296, 0, 31457, 21602, 0))),
    (2022, 10,
     ((482, 595, 643, 62, 0, 262, 86, 0),
      (137769, 24735, 41781, 16632, 0, 31917, 21602, 0))),
    (2022, 11,
     ((482, 599, 645, 62, 0, 263, 87, 0),
      (136842, 24986, 41842, 16632, 0, 32079, 21602, 0))),
    (2022, 12,
     ((485, 602, 649, 62, 0, 264, 87, 0),
      (138745, 25214, 42134, 16728, 0, 32233, 21605, 0))),
    (2023, 1,
     ((379, 490, 574, 76, 16, 231, 84, 0),
      (122484, 20839, 34982, 22385, 5875, 28574, 20819, 0))),
    (2023, 2,
     ((378, 490, 574, 75, 16, 231, 82, 0),
      (122103, 20839, 34983, 20316, 5088, 28574, 20636, 0))),
    (2023, 3,
     ((496, 596, 645, 66, 1, 266, 87, 0),
      (138737, 25172, 42383, 16972, 146, 32351, 21825, 0))),
    (2023, 4,
     ((499, 595, 648, 66, 1, 266, 87, 0),
      (139576, 25204, 43164, 16977, 146, 32346, 21857, 0))),
    (2023, 5,
     ((499, 597, 652, 68, 1, 266, 87, 0),
      (139769, 25346, 43586, 16973, 146, 32389, 21858, 0))),
    (2023, 6,
     ((506, 609, 664, 68, 1, 268, 91, 0),
      (142755, 25797, 44767, 16977, 146, 32836, 21930, 0))),
    (2023, 7,
     ((507, 609, 669, 80, 17, 268, 91, 0),
      (144040, 25795, 45483, 19557, 5240, 32864, 21938, 0))),
    (2023, 8,
     ((507, 607, 674, 80, 17, 268, 91, 0),
      (144015, 25777, 45922, 19557, 5240, 32891, 21938, 0))),
    (2023, 9,
     ((513, 606, 686, 80, 17, 268, 91, 0),
      (146654, 25783, 47057, 19557, 5240, 32916, 21948, 0))),
    (2023, 10,
     ((516, 606, 686, 80, 17, 268, 91, 0),
      (147348, 25847, 47665, 19762, 5412, 32953, 21962, 0))),
    (2023, 11,
     ((535, 619, 717, 80, 17, 271, 91, 0),
      (153818, 26402, 49401, 19916, 5881, 33436, 21962, 0))),
    (2023, 12,
     ((538, 619, 723, 80, 19, 271, 91, 0),
      (154575, 26398, 49912, 20043, 6143, 33455, 21998, 0))),
    (2024, 1,
     ((538, 619, 726, 80, 19, 272, 91, 0),
      (154753, 26398, 50105, 20043, 6143, 33659, 21998, 0))),
    (2024, 2,
     ((538, 619, 726, 80, 19, 272, 91, 0),
      (154759, 26398, 50130, 20043, 6143, 33674, 22005, 0))),
    (2024, 3,
     ((540, 622, 727, 80, 19, 273, 91, 0),
      (154906, 26454, 50101, 20043, 6143, 33715, 22005, 0))),
    (2024, 4,
     ((541, 622, 733, 80, 19, 273, 91, 0),
      (155435, 26542, 50567, 20066, 6143, 33756, 22006, 0))),
    (2024, 5,
     ((544, 625, 736, 80, 19, 273, 91, 0),
      (155569, 26630, 50705, 20072, 6143, 33753, 22006, 0))),
    (2024, 6,
     ((544, 626, 739, 80, 19, 273, 91, 0),
      (155882, 26670, 50878, 20072, 6143, 33793, 22006, 0))),
    (2024, 7,
     ((543, 626, 738, 81, 19, 274, 91, 0),
      (156065, 26692, 50635, 20234, 6143, 33813, 22014, 0))),
    (2024, 8,
     ((544, 631, 741, 81, 19, 272, 91, 0),
      (156474, 26964, 50936, 20287, 6143, 33966, 22014, 0))),
    (2024, 9,
     ((546, 637, 734, 81, 19, 272, 91, 0),
      (156527, 27427, 50731, 20291, 6155, 34308, 22014, 0))),
    (2024, 10,
     ((549, 644, 733, 81, 19, 273, 91, 0),
      (156567, 27555, 50510, 20295, 6155, 34238, 21997, 0))),
    (2024, 11,
     ((551, 644, 738, 81, 19, 273, 91, 0),
      (156850, 27567, 50690, 20388, 6227, 34353, 21997, 0))),
    (2024, 12,
     ((550, 647, 742, 81, 19, 273, 91, 0),
      (157480, 27709, 50986, 20483, 6227, 34393, 21997, 0))),
    (2025, 1,
     ((552, 647, 742, 81, 19, 273, 91, 0),
      (158116, 27924, 51010, 20483, 6227, 34448, 21997, 0))),
    (2025, 2,
     ((554, 645, 744, 81, 19, 261, 91, 0),
      (158854, 27960, 51166, 20483, 6227, 32762, 22008, 0))),
    (2025, 3,
     ((558, 654, 746, 81, 19, 262, 91, 0),
      (160033, 28159, 51512, 20568, 6324, 32841, 22009, 0))),
    (2025, 4,
     ((561, 654, 754, 84, 19, 262, 91, 0),
      (161502, 28172, 52388, 21330, 6451, 32928, 22009, 0)))]


if 0:
    data = count_all()
    for y, m, (f, l) in data:
        print(f'    ({y}, {m},\n     ({f},\n      {l})),')
elif 0:
    for name, f, l in count():
        print(name, f, l)
else:
    plot_count()
