/*
 * Copyright (c) 2016, 2017, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

/**
 * @test
 * @bug 8153391
 * @summary Verify javac behaves properly in JDK image limited to jdk.compiler module
 * @library /tools/lib
 * @modules jdk.compiler/com.sun.tools.javac.api
 *          jdk.compiler/com.sun.tools.javac.main
 * @run main/othervm --limit-modules jdk.compiler LimitedImage
 */

import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;

import toolbox.JavacTask;
import toolbox.JarTask;
import toolbox.Task.Expect;
import toolbox.Task.Mode;
import toolbox.ToolBox;

public class LimitedImage {
    public static void main(String... args) throws IOException {
        ToolBox tb = new ToolBox();

        //showing help should be OK
        new JavacTask(tb, Mode.CMDLINE)
                .options("--help")
                .run().writeAll();

        Path testSource = Paths.get("Test.java");
        tb.writeFile(testSource, "class Test {}");

        //when zip/jar FS is not needed, compilation should succeed
        new JavacTask(tb, Mode.CMDLINE)
                .classpath()
                .files(testSource)
                .outdir(".")
                .run()
                .writeAll();

        Path testJar = Paths.get("test.jar").toAbsolutePath();

        new JarTask(tb, testJar).run();

        //check proper diagnostics when zip/jar FS not present:
        System.err.println("Test " + testJar + " on classpath");
        new JavacTask(tb, Mode.CMDLINE)
                .classpath(testJar)
                .options("-XDrawDiagnostics")
                .files(testSource)
                .outdir(".")
                .run(Expect.SUCCESS);

        System.err.println("Test " + testJar + " on sourcepath");
        new JavacTask(tb, Mode.CMDLINE)
                .sourcepath(testJar)
                .options("-XDrawDiagnostics")
                .files(testSource)
                .outdir(".")
                .run(Expect.SUCCESS);

        System.err.println("Test " + testJar + " on modulepath");
        new JavacTask(tb, Mode.CMDLINE)
                .options("-XDrawDiagnostics",
                         "--module-path", testJar.toString())
                .files(testSource)
                .outdir(".")
                .run(Expect.SUCCESS);

        System.err.println("Test directory containing " + testJar + " on modulepath");
        new JavacTask(tb, Mode.CMDLINE)
                .classpath()
                .options("-XDrawDiagnostics",
                         "--module-path", testJar.getParent().toString())
                .files(testSource)
                .outdir(".")
                .run(Expect.SUCCESS);
    }

}
