# Copyright (c) 1997 Sun Microsystems, Inc.
# All rights reserved.
# 
# Permission is hereby granted, without written agreement and without
# license or royalty fees, to use, copy, modify, and distribute this
# software and its documentation for any purpose, provided that the
# above copyright notice and the following two paragraphs appear in
# all copies of this software.
# 
# IN NO EVENT SHALL SUN MICROSYSTEMS, INC. BE LIABLE TO ANY PARTY FOR
# DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING
# OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF SUN
# MICROSYSTEMS, INC. HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
# 
# SUN MICROSYSTEMS, INC. SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING,
# BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
# FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT.  THE SOFTWARE PROVIDED
# HEREUNDER IS ON AN "AS IS" BASIS, AND SUN MICROSYSTEMS, INC. HAS NO
# OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR
# MODIFICATIONS.

package SyncCM;

use Carp;
use SyncCM::pilot;
use Data::Dumper;
use Time::Local;
use Tk;
use TkUtils;
use strict;
use sigtrap;
use POSIX;

# This guy might fail to load if the user doesn't have the
# CDE libraries installed.  Handle it carefully.
#
eval "use SyncCM::cm";
if ($@)
{
    $@ = qq|
	SyncCM Error!

	SyncCM was unable to load.  Perhaps this is because you do
	not have the appropriate CDE library (libcsa) installed on
	your system?

	Note: This can also happen if your CDE libraries are out of
	date.  Please make sure you have the latest release of CDE
	installed on your system.
	    \n| . $@;

    croak($@);
}

my ($DBFILE);
my ($DEBUG) = 0;
my ($DONT_CHANGE) = 0;
my ($DEBUGFILE);
my ($ERRORFILE);
my ($LOGFILE);
my ($LOGFILE_THRESHOLD) = 100000;
my (%STATS);
my ($RCFILE);
my $VERSION = '1.102';
my ($PREFS);
my ($gConfigDialog);
my ($gRangeBegin);
my ($gRangeEnd);
my (@PrefsVars);
my ($WARNLIMIT) = 50;
my ($gBeginDateEntry);
my ($gEndDateEntry);
my ($gAlarmMenu);
my (@gPrivacyMenu);
my ($gLastCheckedTime);
my ($CANCEL);
my ($DBNAME) = "DatebookDB";

sub conduitCancel
{
    # Set the cancel flag.  We check it fairly often during a sync
    #
    $CANCEL = "cancel";
}

sub checkCancel
{
    # Just croak on cancel.  The sync operation will catch the 
    # croak and Do The Right Thing with it.
    #
    croak("cancel")
	if ($CANCEL);
}

sub conduitInit
{
    $DEBUGFILE = "SyncCM/SyncCM.debug";
    $ERRORFILE = "SyncCM/SyncCM.error";
    $LOGFILE = "SyncCM/SyncCM.log";
    $DBFILE = "SyncCM/pilot-cm.db";
    $RCFILE = "SyncCM/SyncCM.prefs";
    &loadPrefs;

    print "SyncCM: DEBUG is on!\n"
	if ($DEBUG);

    print "SyncCM: DONT_CHANGE is on!  Your calendars will not be changed!\n"
	if ($DONT_CHANGE);

    unless (defined($PREFS->{"cal"}))
    {
	$PREFS->{"cal"} = "";
    }

    unless (defined($PREFS->{"alarm"}))
    {
	$PREFS->{"alarm"}->{"on"} = 0;
	$PREFS->{"alarm"}->{"value"} = "Audio";
    }

    unless (defined($PREFS->{"privacy"}))
    {
	$PREFS->{"privacy"}->{"default"} = "Show Time and Text";
	$PREFS->{"privacy"}->{"on"} = 0;
	$PREFS->{"privacy"}->{"mapping"} = "Show Time only";
    }

    foreach ("Audio", "Flashing", "Popup", "Mail")
    {
	next if defined($PREFS->{reminders}->{$_});

	$PREFS->{"reminders"}->{$_}->{"on"} = 0;
	$PREFS->{"reminders"}->{$_}->{"value"} = 15;
	$PREFS->{"reminders"}->{$_}->{"units"} = "minutes";
    }
    $PREFS->{"reminders"}->{"Mail"}->{"data"} = ""
	unless(defined($PREFS->{"reminders"}->{"Mail"}->{"data"}));

    $PREFS->{"beginDate"} ||= &getMDY(time - 1 * 365 * 24 * 60 * 60);
    $PREFS->{"endDate"} ||= &getMDY(time + 3 * 365 * 24 * 60 * 60);
    $PREFS->{"syncRange"} ||= "Sync Date Range";
    $PREFS->{"dupeCheck"} = 1
	unless (defined($PREFS->{"dupeCheck"}));
    $PREFS->{"logChanges"} = 1
	unless (defined($PREFS->{"logChanges"}));
    $PREFS->{"syncMode"} = "Full Merge"
	unless (defined($PREFS->{"syncMode"}));
    $PREFS->{"fastDelete"} = 0
	unless (defined($PREFS->{"fastDelete"}));
}

sub getMDY
{
    my ($tick) = @_;
    my ($mon, $day, $year);

    ($day, $mon, $year) = (localtime($tick))[3,4,5];
    $mon++;
    $mon = "0$mon"
	if (length($mon) == 1);

    # No y2000 problems here!
    #
    $year += 1900;

    $day = "0$day"
	if (length($day) == 1);
    
    return "$mon/$day/$year";
}
    
sub conduitQuit
{
    &savePrefs;
}

sub conduitInfo
{
    return
    {
	"version" => $VERSION,
	"author" => "Bharat Mediratta",
	"email"	=> 'bharat@menalto.com',
    };
}

sub conduitConfigure
{
    my ($this, $wm) = @_;
    my (@frames);
    my ($obj, $text);
    my (@objs);

    unless (defined($gConfigDialog) && $gConfigDialog->Exists)
    {
	$gConfigDialog = $wm->Toplevel(-title => "Configuring SyncCM");
	$gConfigDialog->withdraw;
	$gConfigDialog->transient($wm);
	$frames[0] = $gConfigDialog->Frame;

	#
	# Desktop calendar box
	#
	$frames[1] = $frames[0]->Frame(-relief => 'ridge', -bd => 4);
	$obj = TkUtils::Label($frames[1], "Desktop Settings");
	$obj->pack(-anchor => 'center');
	
	$frames[2] = $frames[1]->Frame;

	$frames[3] = $frames[2]->Frame(-relief => 'ridge', -bd => 2);

	@objs = TkUtils::AlignedLabelEntries($frames[3],
					    "Calendar:",
					     \$PREFS->{"cal"}
					     );
	$objs[0]->[0]->parent->parent->pack(-expand => 'true',
					    -fill => 'both',
					    -anchor => 'n');
	$objs[1]->[0]->bind("<FocusOut>", sub{&interpretPrefs("cal")});

	$frames[4] = $frames[3]->Frame;
	(@objs) = TkUtils::Radiobuttons($frames[4], 
					\$PREFS->{"syncRange"}, 
					"Sync All", "Sync Date Range");
	$objs[0]->parent->pack(-side => 'top');
	$objs[0]->bind("<Button>", \&updateDateRangeEntries);
	$objs[1]->bind("<Button>", \&updateDateRangeEntries);

	@objs = TkUtils::AlignedLabelEntries($frames[3],
					     "Begin Date:",
					     \$PREFS->{"beginDate"},
					     "End Date:",
					     \$PREFS->{"endDate"}
					     );
	$objs[0]->[0]->parent->parent->pack;
	$gBeginDateEntry = $objs[1]->[0];
	$gEndDateEntry = $objs[1]->[1];
	$gBeginDateEntry->bind("<FocusOut>", sub{&interpretPrefs("begin")});
	$gEndDateEntry->bind("<FocusOut>", sub{&interpretPrefs("end")});

	&updateDateRangeEntries;

	$text =
	    "Date formats are:\n" .
	    "  MM/DD/YYYY for fixed dates\n" .
	    "  -MM/DD/YY   for relative date before today\n" .
	    "  +MM/DD/YY   for relative date after today\n" .
	    "Ex: -06/00/01 is 1 year, 6 months before today.";
	($obj) = TkUtils::Label($frames[3], $text);
	$obj->pack(-anchor => 'center');
	$obj->configure(-justify => 'left');

	$frames[3]->pack(-side => 'top',
			 -expand => 'true',
			 -fill => 'both');

	$frames[2]->pack(-side => 'left',
			 -expand => 'true',
			 -fill => 'both');

	$frames[2] = $frames[1]->Frame(-relief => 'ridge', -bd => 2);

	$obj = TkUtils::Label($frames[2], "Defaults for new\nDesktop calendar appointments");
	$obj->pack(-side => 'top',
		   -anchor => 'center');
	
	$frames[3] = $frames[2]->Frame;
	$frames[4] = $frames[3]->Frame;
	$frames[5] = $frames[3]->Frame;
	$frames[6] = $frames[3]->Frame;
	foreach ("Audio", "Flashing", "Popup", "Mail")
	{
	    $obj = &makeAlarmChoice($_, @frames[4..6]);
	    $obj->pack(-side => 'top',
		       -anchor => 'w');
	}
	$frames[6]->pack(-side => 'right');
	$frames[5]->pack(-side => 'right');
	$frames[4]->pack(-side => 'right');
	$frames[3]->pack(-side => 'top');

	$frames[3] = $frames[2]->Frame;
	$frames[4] = $frames[2]->Frame;

	$obj = TkUtils::Label($frames[3], "Mail To:");
	$obj->pack(-side => 'top');

	$obj = TkUtils::Entry($frames[4],
			      \$PREFS->{"reminders"}->{"Mail"}->{"data"});
	$obj->pack(-side => 'top');

	$obj = TkUtils::Label($frames[3], "Privacy: ");
	$obj->pack(-side => 'left');
	
	my (@PRIVACYMENU) =
	    (
	     "Show Time and Text", [],
	     "Show Time only", [],
	     "Show Nothing", []
	     );

	$gPrivacyMenu[0] =
	    TkUtils::Menu($frames[4],
			  $PREFS->{"privacy"}->{"default"},
			  sub{ ($PREFS->{"privacy"}->{"default"} = $_[0]) =~  s|.*/ ||;
			       $gPrivacyMenu[0]->configure(-text => $PREFS->{"privacy"}->{"default"}); 
			      },
			  @PRIVACYMENU);
	$gPrivacyMenu[0]->pack(-side => 'left',
			    -expand => 'true',
			    -fill => 'x');

	$frames[4]->pack(-side => 'right',
			 -expand => 'true',
			 -fill => 'both');

	$frames[3]->pack(-side => 'right',
			 -expand => 'true',
			 -fill => 'both');

	$frames[2]->pack(-side => 'top',
			 -expand => 'true',
			 -fill => 'both');

	$frames[1]->pack(-expand => 'true',
			 -fill => 'both');


	# Mappings
	#
	#
	$frames[1] = $frames[0]->Frame(-relief => 'ridge', -bd => 4);

	$obj = TkUtils::Label($frames[1], "Mappings");
	$obj->pack(-side => 'top',
		   -anchor => 'center');

	$obj = TkUtils::Label($frames[1],
			      "This section controls how Pilot calendar " .
			      "attributes are\nmapped to Desktop calendar " .
			      "appointment attributes");
	$obj->pack(-side => 'top',
		   -anchor => 'center');

	$frames[2] = $frames[1]->Frame(-relief => 'ridge', -bd => '2');

	$frames[3] = $frames[2]->Frame;
	$frames[4] = $frames[2]->Frame;

	$obj = TkUtils::Checkbutton($frames[3],
				    "Attach the Pilot Alarm to:",
				    \$PREFS->{"alarm"}->{"on"});
	
	$obj->pack(-side => 'top',
		   -anchor => 'nw');

	my (@ALARMMENU) = (
			   "Audio", [],
			   "Flashing", [],
			   "Popup", [],
			   "Mail", []
			   );

	$gAlarmMenu = TkUtils::Menu($frames[4], 
				    $PREFS->{"alarm"}->{"value"},
				    sub{ ($PREFS->{"alarm"}->{"value"} = $_[0]) =~  s|.*/ ||;
					 $gAlarmMenu->configure(-text => $PREFS->{"alarm"}->{"value"}); 
				     },
				    @ALARMMENU);

	$gAlarmMenu->pack(-side => 'top',
			  -anchor => 'center',
			  -expand => 'false',
			  -fill => 'x');

	$obj = TkUtils::Checkbutton($frames[3],
				    "Attach the Pilot Private flag to:",
				    \$PREFS->{"privacy"}->{"on"});
	
	$obj->pack(-side => 'top',
		   -anchor => 'nw');
	
	$gPrivacyMenu[1] =
	    TkUtils::Menu($frames[4],
			  $PREFS->{"privacy"}->{"mapping"},
			  sub{ ($PREFS->{"privacy"}->{"mapping"} = $_[0]) =~  s|.*/ ||;
			       $gPrivacyMenu[1]->configure(-text => $PREFS->{"privacy"}->{"mapping"}); 
			      },
			  @PRIVACYMENU);

	$gPrivacyMenu[1]->pack(-side => 'top',
			       -anchor => 'center',
			       -expand => 'false',
			       -fill => 'x');

	$frames[4]->pack(-side => 'right',
			 -anchor => 'w',
			 -expand => 'true');

	$frames[3]->pack(-side => 'left',
			 -anchor => 'e',
			 -expand => 'true');

	$frames[2]->pack(-side => 'top',
			 -anchor => 'center',
			 -expand => 'true',
			 -fill => 'x');

	$frames[1]->pack(-side => 'top',
			 -expand => 'true',
			 -fill => 'both');


	#
	# GENERAL SETTINGS
	#
	$frames[1] = $frames[0]->Frame(-relief => 'ridge',
				       -bd => 4);
	$obj = TkUtils::Label($frames[1], "General Settings");
	$obj->pack(-anchor => 'center');
	$frames[2] = $frames[1]->Frame(-relief => 'ridge', -bd => 2);
	$obj = TkUtils::Label($frames[2], "Miscellaneous");
	$obj->pack(-anchor => 'center');
	$obj = TkUtils::Checkbutton($frames[2], 
				    "Remove duplicates",
				    \$PREFS->{"dupeCheck"});
	$obj->pack(-side => 'top');
	$obj = TkUtils::Checkbutton($frames[2], 
				    "Log all changes",
				    \$PREFS->{"logChanges"});
	$obj->pack(-side => 'top');
	$frames[2]->pack(-side => 'left',
			 -expand => 'true',
			 -fill => 'both');

	$frames[2] = $frames[1]->Frame(-relief => 'ridge', -bd => 2);
	$obj = TkUtils::Label($frames[2], "Type of Sync");
	$obj->pack(-anchor => 'center'); 

	@objs = TkUtils::Radiobuttons($frames[2],
				     \$PREFS->{"syncMode"},
				     "Full Merge",
				     "Pilot overwrites Desktop",
				     "Desktop overwrites Pilot");
	map($_->pack(-side => 'top'), @objs);
	$objs[0]->parent->pack(-side => "top");

	$frames[2]->pack(-side => 'left',
			 -expand => 'true',
			 -fill => 'both');

	$frames[1]->pack(-side => 'top',
			 -expand => 'true',
			 -fill => 'both');


	$obj = TkUtils::Button($frames[0], "Dismiss", \&hideConfig);
	$obj->pack(-side => "bottom");



	$frames[0]->pack(-expand => 'true',
			 -fill => 'both');


	# Advanced settings
	#
	$frames[1] = $frames[0]->Frame(-relief => 'ridge',
				       -bd => 4);

	$obj = TkUtils::Label($frames[1], "Advanced Settings");
	$obj->pack(-side => 'top',
		   -anchor => 'center');

	$obj = TkUtils::Checkbutton($frames[1],
				    "Use fast calendar delete on Desktop",
				    \$PREFS->{"fastDelete"});
	$obj->pack(-side => 'top',
		   -anchor => 'center');


	$text = "Fast calendar delete is only applicable when the" .
		"'Pilot overwrites\nDesktop' option is in force.  " .
		"Fast delete will actually delete the calendar\nfrom ".
		"the server and then create a new one (which is " .
		"significantly\nfaster).  However, if you do not have " .
		"permission to create a new\ncalendar on your " .
		"calendar server SyncCM will be unable to create a\n".
		"calendar for you!  Use this only if you know ".
		"what you're doing!";
	$obj = TkUtils::Label($frames[1], $text);
	$obj->configure(-justify => 'left');
	$obj->pack(-side => 'top',
		   -anchor => 'center');

	$frames[1]->pack(-expand => 'true',
			 -fill => 'both');
	# Special options
	#
	$frames[1] = $frames[0]->Frame(-relief => 'ridge',
				       -bd => 4);
	$obj = TkUtils::Label($frames[1], "Special Options");
	$obj->pack(-anchor => 'center');

	$text = "If something happens to either your Pilot datebook " .
		"or your Desktop\ncalendar that causes either database " .
		"to be erased, you should reset\nthe SyncCM conduit.  " .
		"Resetting SyncCM will *not* cause any\nappointments to " .
		"be deleted.";

	$obj = TkUtils::Label($frames[1], $text);
	$obj->configure(-justify => 'left');
	$obj->pack(-anchor => 'center');

	$obj = TkUtils::Button($frames[1], "Reset SyncCM", 
			       sub{ &resetSyncCM; });
	$obj->pack(-side => "bottom");

	$frames[1]->pack(-expand => 'true',
			 -fill => 'both');

	PilotMgr::setColors($gConfigDialog);
    }

    $gConfigDialog->Popup;
}

sub makeAlarmChoice
{
    my ($name, @frames) = @_;
    my ($frame, $obj);

    my (@menu) =
	(
	 'minutes' => [],
	 'hours' => [],
	 'days' => []
	 );

    ($obj) = TkUtils::Checkbutton($frames[0], $name, 
				  \$PREFS->{"reminders"}->{$name}->{"on"});

    $obj->pack(-side => 'top');

    $obj = TkUtils::Entry($frames[1],
			  \$PREFS->{"reminders"}->{$name}->{"value"});
    $obj->configure(-width => 5);
    $obj->pack(-side => 'top');

    # Amazingly enough, the callback actually works here.  I'm not
    # sure why '$obj' gets scoped to refer to the correct menu.
    #
    # Perhaps a Perl guru can someday tell me.
    #
    $obj = TkUtils::Menu($frames[2],
			 $PREFS->{"reminders"}->{$name}->{"units"},
			 sub{ ($PREFS->{"reminders"}->{$name}->{"units"} =
			       $_[0]) =~ s|.*/ ||;
			      $obj->configure(-text => 
					      $PREFS->{"reminders"}->{$name}->{"units"})},
			 @menu);

    $obj->pack(-side => 'top');
}

sub updateDateRangeEntries
{
    if ($PREFS->{"syncRange"} eq "Sync All")
    {
	$gBeginDateEntry->configure(-state => 'disabled',
				    -show => '*');
	$gEndDateEntry->configure(-state => 'disabled',
				  -show => '*');
    }
    else
    {
	$gBeginDateEntry->configure(-state => 'normal',
				    -show => undef);
	$gEndDateEntry->configure(-state => 'normal',
				  -show => undef);
    }
}

sub hideConfig
{
    if (&interpretPrefs("all"))
    {
	&savePrefs; 
	$gConfigDialog->withdraw;
    }
}

sub resetSyncCM
{
    my ($ans);
    $ans = PilotMgr::askUser("Do you really want to reset SyncCM?", 
			     "Yes", 
			     "No");

    if ($ans eq "Yes")
    {
	unlink($DBFILE);
    }
}

sub interpretPrefs
{
    my ($which) = @_;

    unless ($PREFS->{"syncRange"} eq "Sync All")
    {
	$gRangeBegin = &strToTick($PREFS->{"beginDate"});
	if ($which eq "begin" || $which eq "all")
	{
	    if ($gRangeBegin < 0)
	    {
		PilotMgr::tellUser("Your Sync begin date is malformed!");
		return 0;
	    }
	    else
	    {
		$PREFS->{"beginDate"} = &getMDY($gRangeBegin)
		    unless ($PREFS->{"beginDate"} =~ /^[-+]/);

		# XXX: Reiterate this connection because the way that
		# I do colors exposes a bug in the Tk library breaking
		# this connection.
		#
		$gBeginDateEntry->configure(-textvariable =>
					    \$PREFS->{"beginDate"})
		    if (defined($gBeginDateEntry) && Exists($gBeginDateEntry));

	    }
	}

	$gRangeEnd = &strToTick($PREFS->{"endDate"});
	if ($which eq "end" || $which eq "all")
	{
	    if ($gRangeEnd < 0)
	    {
		PilotMgr::tellUser("Your Sync end date is malformed!");
		return 0;
	    }
	    else
	    {
		$PREFS->{"endDate"} = &getMDY($gRangeEnd)
		    unless ($PREFS->{"endDate"} =~ /^[-+]/);

		# XXX: Reiterate this connection because the way that
		# I do colors exposes a bug in the Tk library breaking
		# this connection.
		#
		$gEndDateEntry->configure(-textvariable =>
					  \$PREFS->{"endDate"})
		    if (defined($gEndDateEntry) && Exists($gEndDateEntry));
	    }
	}

	if ($which eq "all")
	{
	    if ($gRangeEnd <= $gRangeBegin)
	    {
		PilotMgr::tellUser("The Sync end date must be after the " .
				   "Sync begin date!");
		return 0;
	    }
	}
    }

    if ($which eq "cal" || $which eq "all")
    {
	$PREFS->{"cal"} =~ /(.*)@(.*)/;
	unless ($1 && $2)
	{
	    PilotMgr::tellUser("Your SyncCM calendar is set " .
			       "incorrectly.  Please fix it!");
	    return 0;
	}
    }
    return 1;
}


sub loadPrefs
{
    my ($line);

    open(FD, "<$RCFILE") || return;
    
    $line = <FD>;

    if ($line =~ /\$SyncCM::/)
    {
	# Old format (v1.006x and before)
	#
	do
	{
	    my ($key, $val);

	    if ($line)
	    {
		$line =~ /\$SyncCM::(\w+) = '(.*)';$/;
		($key, $val) = ($1, $2);

		$PREFS->{$key} = $val
		    if ($key && $val);
	    }

	    $line = <FD>;
	} while ($line);

	close(FD);
    }
    else
    {
	close(FD);
	eval `cat $RCFILE`;
    }

    # Now bring out-of-date preferences back up to speed
    # What a nice developer I am, taking care of the
    # behind-the-times users. :-)
    #
    if (!defined($PREFS->{"version"}))
    {
	# Old format (v1.009 and before)
	#
	# First, convert from yucky old names to cool new names.
	#
	my (%table) =
	    (
	     "gAlarm" => "alarm",
	     "gBeginDate" => "beginDate",
	     "gCMWriteProtect" => "CMWriteProtect",
	     "gCal" => "cal",
	     "gDupeCheck" => "dupeCheck",
	     "gEndDate" => "endDate",
	     "gLogChanges" => "logChanges",
	     "gPilotWriteProtect" => "pilotWriteProtect",
	     "gSyncRange" => "syncRange",
 	     );

	foreach (keys %$PREFS)
	{
	    $PREFS->{$table{$_}} = $PREFS->{$_};
	    delete $PREFS->{$_};
	}

	# Now update a few structures
	#
	if (ref($PREFS->{"alarm"}) ne "HASH")
	{
	    my ($tmp) = $PREFS->{"alarm"};
	    
	    $tmp = "Flashing" if ($tmp eq "Flash");
	    $tmp = "Audio" if ($tmp eq "Beep");
	    
	    delete $PREFS->{"alarm"};
	    $PREFS->{"alarm"}->{"value"} = $tmp;
	    
	    if ($tmp ne "None")
	    {
		$PREFS->{"alarm"}->{"on"} = 1;
	    }
	    else
	    {
		$PREFS->{"alarm"}->{"on"} = 0;
		$PREFS->{"alarm"}->{"value"} = "Audio";
	    }
	}

	if ($PREFS->{"pilotWriteProtect"} &&
	    $PREFS->{"CMWriteProtect"})
	{
	    msg("WARNING: Both Pilot and Desktop are write protected\n" .
		"SyncCM configuration is invalid!");
	}
	elsif ($PREFS->{"CMWriteProtect"})
	{
	    $PREFS->{"syncMode"} = "Desktop overwrites Pilot";
	}
	elsif ($PREFS->{"pilotWriteProtect"})
	{
	    $PREFS->{"syncMode"} = "Pilot overwrites Desktop";
	}
	else
	{
	    $PREFS->{"syncMode"} = "Full Merge";
	}

	delete $PREFS->{"pilotWriteProtect"};
	delete $PREFS->{"CMWriteProtect"};
    }

    #
    # Prefs are now 1.100 compliant
    #
}

sub savePrefs
{
    my ($var);

    $Data::Dumper::Purity = 1;
    $Data::Dumper::Deepcopy = 1;
    $Data::Dumper::Indent = 0;

    $Data::Dumper::Indent = 1
	if ($DEBUG);

    $PREFS->{"version"} = $VERSION;

    if (open(FD, ">$RCFILE"))
    {
        print FD Data::Dumper->Dump([$PREFS], ['PREFS']);
        print FD "1;\n";
        close(FD);
    }
    else
    {
        PilotMgr::msg("Unable to save preferences to $RCFILE!");
    }
}

sub conduitSync
{
    my ($this, $dlp, $info) = @_;
    my ($db, $master, $cm_session, $pilot_dbhandle);

    &cycleLogs;

    # Use watchdog to keep pilot connection alive
    #
    $dlp->watchdog(20);

    # In case we aborted last time while the cancel flag
    # was set...
    #
    $CANCEL = "";

    if (-f $DBFILE)
    {
	eval `cat $DBFILE`;
    }

    if (defined($master) && 
	(!exists($master->{"version"}) ||
	 $master->{"version"} ne $VERSION))
    {
	PilotMgr::msg("SyncCM has been upgraded.  A full sync is required");
	$master = undef;
    }

    # If an upgrade happened or SyncCM was reset, populate it with
    # reasonable defaults.
    #
    if (!defined($master))
    {
	$master = &defaults;
    }

    # Pass our sync defaults down to the device modules
    #
    SyncCM::cm::setup({
	"Audio" => $PREFS->{"reminders"}->{"Audio"},
	"Flashing" => $PREFS->{"reminders"}->{"Flashing"},
	"Popup" => $PREFS->{"reminders"}->{"Popup"},
	"Mail" => $PREFS->{"reminders"}->{"Mail"},
	"Privacy" => $PREFS->{"privacy"},
	"Alarm" => $PREFS->{"alarm"},
    });

    SyncCM::pilot::setup({
	"Alarm" => $PREFS->{"alarm"},
	"Privacy" => $PREFS->{"privacy"},
    });

    if ($PREFS->{"logChanges"})
    {
	if (open(LOGFD, ">>$LOGFILE"))
	{
	    select((select(LOGFD), $| = 1)[0]);
	    print LOGFD "-" x 30, "\n";
	    print LOGFD PilotMgr::prettyTime(time), " Sync beginning\n";
	}
	else
	{
	    PilotMgr::msg("Unable to log details to $LOGFILE\n".
			  "Logging disabled");
	    $PREFS->{"logChanges"} = 0;
	}
    }

    unless (&interpretPrefs("all"))
    {
	PilotMgr::msg("SyncCM prefs are invalid.  Aborting...");

	close(LOGFD)
	    if ($PREFS->{"logChanges"});

	return;
    }

    &checkCancel;

    $STATS{"cm_create"} = 0;
    $STATS{"cm_delete"} = 0;
    $STATS{"cm_change"} = 0;
    $STATS{"cm_dupe"} = 0;
    $STATS{"cm_range"} = 0;
    $STATS{"pilot_create"} = 0;
    $STATS{"pilot_delete"} = 0;
    $STATS{"pilot_change"} = 0;
    $STATS{"pilot_dupe"} = 0;
    $STATS{"dupe_merge"} = 0;
    $STATS{"delete_both"} = 0;

    # Now eval some stuff that's going to potentially crash
    # Everything called here can safely croak()
    #
    eval
    {
	&loadDatabases(\$master, $info, $dlp,
		       \$cm_session, \$pilot_dbhandle,
		       $gRangeBegin, $gRangeEnd);

	unless ($DONT_CHANGE)
	{
	    &reconcile($master, $info, $cm_session, $pilot_dbhandle);
	}
    };

    if ($@)
    {
	if ($@ =~ /^cancel/)
	{
	    PilotMgr::msg("SyncCM cancelled [at user request]");
	}
	elsif ($@ =~ /^abort/)
	{
	    PilotMgr::msg("SyncCM aborted. [at user request]");
	    $CANCEL = "abort";
	}
	else 
	{
	    PilotMgr::msg("Error: $@");
	    $CANCEL = "fail";
	}
    }
    else
    {
	# We terminated normally so it's ok to clean up the
	# Pilot database
	#
	SyncCM::pilot::cleanup($pilot_dbhandle);
    }

    # We either finished or got cancelled.  Either way, wrap it up.
    #
    eval
    {
	$cm_session->logoff();

	$pilot_dbhandle->close();
    };

    &cleanup($master, $info);

    $master->{"lastSyncDate"} = $info->{"thisSyncDate"};

    $master->{"cal"} = $PREFS->{"cal"};
    $master->{"size_cm"} = grep(/^cm_/, keys %$master);
    $master->{"size_pilot"} = grep(/^pilot_/, keys %$master);

    my ($line);
    $line = &makeStatusLine("cm_create", "created on Desktop calendar");
    $line .= &makeStatusLine("cm_delete", "deleted on Desktop calendar");
    $line .= &makeStatusLine("cm_change", "changed on Desktop calendar");
    $line .= &makeStatusLine("cm_range", 
			     "moved out of range on Desktop calendar");
    $line .= &makeStatusLine("cm_dupe", "removed on Desktop calendar", 
			     "duplicate record");
    $line .= &makeStatusLine("pilot_create", "created on the Pilot");
    $line .= &makeStatusLine("pilot_delete", "deleted on the Pilot");
    $line .= &makeStatusLine("pilot_change", "changed on the Pilot");
    $line .= &makeStatusLine("pilot_dupe", "removed on the Pilot", 
			     "duplicate record");
    $line .= &makeStatusLine("dupe_merge", "merged on both sides",
			     "duplicate record");

    if ($line)
    {
	PilotMgr::msg($line);
    }
    else
    {
	PilotMgr::msg("No changes");
    }
	
    if ($CANCEL ne "abort" && $CANCEL ne "fail")
    {
	status("Saving database", 0);
	&saveDB($master);
	status("Saving database", 100);
    }
    else
    {
	PilotMgr::msg("SyncCM encountered an error during operation\n" .
		      "To avoid causing further complications, the next\n" .
		      "calendar sync will be a full sync.");
	unlink($DBFILE);
    }

    my ($total) = scalar(grep(/^master_/, keys %$master));
    $dlp->log("SyncCM synchronized $total appts\n\n");

    close(LOGFD)
	if ($PREFS->{"logChanges"});

    $CANCEL = "";
    $dlp->watchdog(0);
}

sub loadDatabases
{
    my ($master, $info, $dlp, 
	$cm_session, $pilot_dbhandle,
	$gRangeBegin, $gRangeEnd) = @_;
    my ($db);

    eval
    {
	$$cm_session = SyncCM::cm::logon($PREFS->{"cal"});
    };
    if ($@ =~ /NOT.EXIST/)
    {
	PilotMgr::msg("'$PREFS->{cal}' does not exist.  " .
		      "Attempting to create it.");
	
	SyncCM::cm::createCalendar($PREFS->{"cal"});
	$$cm_session = SyncCM::cm::logon($PREFS->{"cal"});
    }
    elsif ($@)
    {
	croak($@);
    }

    # Make sure that this calendar is supported to the version
    # that we require.
    #
    my (@attrs);
    eval
    {
	@attrs = $$cm_session->list_calendar_attributes();
    };
    if ($@ =~ /NO AUTHORITY/)
    {
	PilotMgr::msg
	    ("SyncCM can't examine the attributes for '$PREFS->{cal}'".
	     "\nEither you do not have appropriate permissions for this\n".
	     "calendar or your calendar (and calendar server) needs to\n".
	     "be updated to the CDE standard.");
	croak("fail");
    }
    elsif ($@)
    {
	croak($@);
    }


    my $errorMessage = 
	"SyncCM cannot interface with your calendar server\n" .
	"Please upgrade to the CDE calendar server\n";
    my $serverVersion = 0;

    if (grep(/^Server Version$/, @attrs))
    {
	my (%attr) = $$cm_session->read_calendar_attributes("Server Version");
	$serverVersion = $attr{"Server Version"}->{"value"};
	if ($serverVersion < 5)
	{
	    PilotMgr::msg($errorMessage);
	    PilotMgr::msg("(Server Version $serverVersion < 5)");
	    croak("fail");
	}
    }
    else
    {
	PilotMgr::msg($errorMessage);
	PilotMgr::msg("(Server has no Version)");
	croak("fail");
    }

    if (grep(/^Data Version/, @attrs))
    {
	my (%attr) = $$cm_session->read_calendar_attributes("Data Version");

	if ($attr{"Data Version"}->{"value"} != 4)
	{
	    PilotMgr::msg("$PREFS->{cal} is in an obsolete data format\n" .
			  "It must be updated to the CDE format before\n" .
			  "SyncCM can interface with it.");
	    croak("fail");
	}
    }
    elsif ($serverVersion < 6)
    {
	# (Data Version attr not used in server v6)
	PilotMgr::msg($errorMessage);
	PilotMgr::msg("(Server Version check OK but Data Version is wrong)");
	croak("fail");
    }

    if ($PREFS->{"cal"} ne $$master->{"cal"})
    {
	my ($ans) = 
	    PilotMgr::askUser("You have changed your Desktop calendar but " .
			      "not reset SyncCM.\nIf you continue, many " .
			      "Pilot datebook appointments may be deleted " .
			      "to sync your specified calendars.  The " .
			      "safest thing to do is to reset SyncCM.  What " .
			      "do you want to do?",
			      "Reset SyncCM", "Abort", "Continue with Sync");

	if ($ans eq "Reset SyncCM")
	{
	    $$master = &defaults;
	}
	elsif ($ans eq "Abort")
	{
	    croak("abort");
	}
    }

    if ($PREFS->{"syncMode"} eq "Pilot overwrites Desktop")
    {
	PilotMgr::msg("Overwriting $PREFS->{cal}");

	SyncCM::status("Erasing $PREFS->{cal}", 0);

	if ($PREFS->{"fastDelete"})
	{
	    # Need to delete and recreate Desktop calendar
	    # This is the fast way to do it.  However, this leads
	    # to major problems because servers may not allow you
	    # to create a new calendar.  So, you may delete the 
	    # calendar ok and be left without a calendar!
	    #
	    eval
	    {
		status("Erasing $PREFS->{cal}", 0);
		SyncCM::cm::deleteCalendar($$cm_session);
	    };
	    if ($@)
	    {
		# Different revs of Calendar::CSA have
		# 'NOT_EXIST' vs. 'NOT EXIST'
		if ($@ !~ /NOT.EXIST/)
		{
		    # It's ok if it doesn't exist -- 
		    # we were gonna delete it anyway
		}
		else
		{
		    croak($@);
		}
	    }

	    status("Erasing $PREFS->{cal}", 30);

	    eval
	    {
		SyncCM::cm::createCalendar($PREFS->{"cal"});
	    };
	    if ($@ =~ /add_calendar.*NO AUTHORITY/)
	    {
		PilotMgr::tellUser
		    ("Uh-oh.  SyncCM deleted $PREFS->{cal} but could not " .
		     "recreate it.  This is because you are using 'fast " .
		     "calendar delete' on a system where you do not have " .
		     "permissions to create a calendar.  Either give " .
		     "yourself permissions to create new calendars and sync ".
		     "again, or have your sysadmin create a new calendar for ".
		     "you and turn off 'fast calendar delete' for future " .
		     "syncs.  No appointments have been lost.");
		croak("SyncCM abort");
	    }
	    elsif ($@)
	    {
		croak($@);
	    }
	}
	else
	{
	    SyncCM::cm::emptyCalendar($$cm_session, $PREFS->{"cal"});
	}

	# Now we need to go through the database removing any
	# cm_id's.
	#
	my (@list) = keys %$$master;
	my ($count) = 0;
	my ($count_max) = scalar(@list);
	foreach (@list)
	{
	    if ($count++ % &fake_ceil($count_max / 20))
	    {
		status("Erasing $PREFS->{cal}",
		       75 + int(25 * $count / $count_max));
	    }

	    if (/^cm_/)
	    {
		delete $$master->{$_};
		next;
	    }

	    if (/^master/)
	    {
		delete $$master->{$_}->{'cm_id'};
	    }
	}

	$$master->{"size_cm"} = 0;

	status("Erasing $PREFS->{cal}", 100);

	$$cm_session = SyncCM::cm::logon($PREFS->{"cal"});
    }
    else
    {
	# Load in all appointments from the CM
	#
	PilotMgr::msg("Using calendar '$PREFS->{cal}'");
	if ($PREFS->{"syncRange"} eq "Sync All")
	{
	    PilotMgr::msg("Reading all dates.");

	    $db = SyncCM::cm::loadAppointmentsAll($$cm_session);
	}
	else
	{
	    PilotMgr::msg('Reading dates ', &getMDY($gRangeBegin),
			  ' to ', &getMDY($gRangeEnd), '.');

	    $db = SyncCM::cm::loadAppointmentsRange($$cm_session, 
						    $gRangeBegin,
						    $gRangeEnd);
	}
    }

    &checkCancel;

    if ($$master->{"size_cm"} - keys(%$db) > $WARNLIMIT)
    {
	askagain1:
	my ($ans) = 
	    PilotMgr::askUser("Your Desktop calendar shrunk " .
			      "significantly since your last sync (" .
			      "perhaps you changed the date range or " .
			      "switched calendars)\n" .
			      "If you continue, appoints may be deleted " .
			      "from your Pilot Datebook " .
			      "so that they are in sync.  Is " .
			      "this what you want?\n", 
			      "Yes", "No", "What can I do?");
	
	if ($ans eq "No")
	{
	    croak("cancel");
	}
	elsif ($ans eq "What can I do?")
	{
	    PilotMgr::tellUser("A common culprit for your Desktop " .
			       "calendar's disappearance is " .
			       "that your calendar has been moved. " .
			       "Please update " .
			       "your SyncCM configuration to your new " .
			       "calendar location, reset SyncCM via the " .
			       "'Reset SyncCM' button, and sync again.");
	    goto askagain1;
	}
    }

    # Merge the CM's appointments with the master
    #
    &merge($$master, $db, "cm", $info);

    &checkCancel;

    if ($PREFS->{"syncMode"} eq "Desktop overwrites Pilot")
    {
	# Need to delete and recreate Pilot calendar
	#
	status("Erasing Pilot Datebook", 0);

	$dlp->delete($DBNAME);

	status("Erasing Pilot Datebook", 30);

	$$pilot_dbhandle = $dlp->create($DBNAME, 'date', 'DATA', 0, 0);
	$dlp->getStatus();

	status("Erasing Pilot Datebook", 50);

	croak("Could not create Datebook")
	    unless defined($$pilot_dbhandle);

	# Now we need to go through the database removing any
	# pilot_id's.
	#
	my (@list) = keys %$$master;
	my ($count) = 0;
	my ($count_max) = scalar(@list);
	foreach (@list)
	{
	    if ($count++ % &fake_ceil($count_max / 20))
	    {
		status("Erasing Pilot Datebook",
		       50 + int(50 * $count / $count_max));
	    }

	    if (/^pilot_/)
	    {
		delete $$master->{$_};
		next;
	    }

	    if (/^master/)
	    {
		delete $$master->{$_}->{'pilot_id'};
	    }
	}

	$$master->{"size_pilot"} = 0;

	status("Erasing Pilot Datebook", 100);
    }
    else
    {
	eval
	{
	    $$pilot_dbhandle = $dlp->open($DBNAME);
	};
	if ($@ =~ /read-only value/)
	{
	    PilotMgr::msg("Pilot database '$DBNAME' does not exist.\n" .
			  "Creating it...");

	    $$pilot_dbhandle = $dlp->create($DBNAME, 'date', 'DATA', 0, 0);
	    
	    # And now we're done, since we know there's nothing
	    # to load from the Pilot.
	    #
	    return;
	}
	elsif ($@)
	{
	    croak($@);
	}

	# Alert the user
	#
	$dlp->getStatus();

	# Check to see if we can do a fast sync from the pilot
	#
	if (defined($$master->{"lastSyncDate"}) &&
	    $$master->{"lastSyncDate"} == $info->{"successfulSyncDate"} &&
	    $$master->{"cal"} eq $PREFS->{"cal"})
	{
	    # Yes we can.
	    # Load changed appointments only.
	    # This works even if the Pilot is overwriting the desktop, 
	    # because when we're done loading the changes we'll still 
	    # know about all the appts on the Pilot.
	    #
	    $db = SyncCM::pilot::loadChangedAppointments($$pilot_dbhandle);
	    &partialMerge($$master, $db, "pilot", $info);
	}
	else
	{
	    # No we can't.
	    # Load in all appointments from the Pilot
	    #
	    $db = SyncCM::pilot::loadAppointments($$pilot_dbhandle);

	    if ($$master->{"size_pilot"} - scalar(keys %$db) > $WARNLIMIT)
	    {
		askagain2:
		my ($ans) = 
		    PilotMgr::askUser("Your Pilot Datebook has shrunk " .
				      "significantly since your last " .
				      "sync.  " .
				      "If you continue, appointments " .
				      "may be " .
				      "deleted from your Calendar " . 
				      "Manager calendar " .
				      "so that they are in sync.  Is " .
				      "this what you want?", 
				      "Yes", "No", "What can I do?");
		
		if ($ans eq "No")
		{
		    croak("cancel");
		}
		elsif ($ans eq "What can I do?")
		{
		    PilotMgr::tellUser("A common culprit for your Pilot " .
				       "Datebook's disappearance is an " .
				       "upgrade.  If you did not " .
				       "intentionally delete your Pilot " .
				       "Datebook, you have two " .
				       "choices:\n" .
				       "\n1. Restore DatebookDB from " .
				       "your backups using the " .
				       "Installer " .
				       "conduit\n\n2. Have SyncCM " .
				       "rebuild " .
				       "it for you by resetting SyncCM " .
				       "via " .
				       "the SyncCM properties sheet and " .
				       "then syncing again.");
		    goto askagain2;
		}
	    }

	    # Merge the Pilot's appointments with the master
	    #
	    &merge($$master, $db, "pilot", $info);
	}
    }
}

sub makeStatusLine
{
    my ($key, $tag, $noun) = @_;
    my ($line);

    if (!defined($noun))
    {
	$noun = "record";
    }
    $line = "";
    if ($STATS{$key} > 0)
    {
	my ($val) = $STATS{$key};
	return sprintf("%d %s%s %s\n",
		       $val,
		       $noun,
		       $val > 1 ? "s were" : " was",
		       $tag);
    }

    return "";
}

sub saveDB
{
    my ($db) = @_;

    $Data::Dumper::Purity = 1;
    $Data::Dumper::Indent = 0;
    $Data::Dumper::Deepcopy = 1;

    $Data::Dumper::Indent = 1
	if ($DEBUG || $DONT_CHANGE);

    open(FD, ">$DBFILE")
	|| die "Unable to save database";

    $db->{"version"} = $VERSION;

    # Use the compiled version for speed, if it's around.
    #
    if (defined &Data::Dumper::Dumpxs)
    {
	print FD Data::Dumper->Dumpxs([$db], ['master']);
    }
    else
    {
	print FD Data::Dumper->Dump([$db], ['master']);
    }
    close(FD);
}

sub cleanup
{
    my ($db, $info) = @_;
    my ($key);

    foreach $key (sort keys %{$db})
    {
	next unless ($key =~ /(cm_|pilot_)/);

	if ($db->{$key}->{"timestamp"} != $info->{"thisSyncDate"})
	{
	    # Unupdated record (e.g., deleted cm record).  Expire it.
	    #
	    delete $db->{$key};
	    next;
	}

	unless ($DONT_CHANGE)
	{
	    if (exists $db->{$key}->{"changes"})
	    {
		# An unreconciled change.  This will happen if we cancel
		# a sync.  Delete it.  We'll pick it up again on the 
		# next sync.
		delete $db->{$key}->{"changes"};
	    }
	}
    }
}

sub reconcile
{
    my ($db, $info, $cm_session, $pilot_dbhandle) = @_;
    my ($key, $p_id, $p_changes, $c_id, $c_changes);
    my ($orig_id, $id);
    my (@changes);
    my ($pilot_changes, $cm_changes);
    my ($count, $count_max);

    $count = 0;
    if ($PREFS->{"dupeCheck"})
    {
	my ($key, %hash);
	my (@list);
	# Make a hash table
	#
	$count = 0;
	@list = grep(/master_/, keys %{$db});
	$count_max = scalar(@list);
	foreach $key (@list)
	{
	    unless ($count++ % &fake_ceil($count_max / 20))
	    {
		status("Scanning for duplicates",
		       int(50 * $count / $count_max));
	    }

	    # Sanity check -- don't do dupe checking on
	    # appointments that are about to be deleted.
	    #
	    my ($id);
	    if (defined($id = $db->{$key}->{"cm_id"}))
	    {
		next unless (defined($id = $db->{"cm_$id"}));
		next unless ($id->{"timestamp"} == $info->{"thisSyncDate"});
	    }

	    if (defined($id = $db->{$key}->{"pilot_id"}))
	    {
		next unless (defined($id = $db->{"pilot_$id"}));
		next unless ($id->{"timestamp"} == $info->{"thisSyncDate"});
	    }

	    push(@{$hash{$db->{$key}->{"begin"}}}, $key);
	}

	$count = 0;
	$count_max = scalar(keys %hash);
	foreach $key (keys %hash)
	{
	    my ($k1, $k2, $del, $i);

	    if (0)
	    {
		printf("%10s  ", $key);
		foreach $k1 (@{$hash{$key}})
		{
		    printf("%-.15s ", $db->{$k1}->{"description"});
		}
		print "\n";
	    }

	    my ($status);

	    unless ($count++ % &fake_ceil($count_max / 20))
	    {
		status("Removing duplicates",
		       50 + int(50 * $count / $count_max));
	    }

	    while (scalar(@{$hash{$key}}) > 1)
	    {
		$k1 = shift(@{$hash{$key}});
		undef $k2;
		$i = 0;
		foreach $k2 (@{$hash{$key}})
		{
		    if (&compareAppt($db->{$k1}, $db->{$k2}))
		    {
			# A duplicate! If one of them exists only on
			# the Pilot and the other exists only on the CM
			# then link them together.
			#
			if ((defined($db->{$k1}{"pilot_id"}) &&
			     !defined($db->{$k1}{"cm_id"})) &&
			    (defined($db->{$k2}{"cm_id"}) &&
			     !defined($db->{$k2}{"pilot_id"})))
			{
			    $db->{$k1}{"cm_id"} = $db->{$k2}{"cm_id"};
			    $db->{"cm_" . $db->{$k2}{"cm_id"}}{"id"} = $k1;
			    $del = $k2;
			    &log("dupe_merge", $db->{$del});
			}
			elsif ((!defined($db->{$k1}{"pilot_id"}) &&
				defined($db->{$k1}{"cm_id"})) &&
			       (!defined($db->{$k2}{"cm_id"}) &&
				defined($db->{$k2}{"pilot_id"})))
			{
			    $db->{$k2}{"cm_id"} = $db->{$k1}{"cm_id"};
			    $db->{"cm_" . $db->{$k1}{"cm_id"}}{"id"} = $k2;
			    $del = $k1;
			    &log("dupe_merge", $db->{$del});
			}
			else
			{
			    # Ok, that didn't work.  Try to blow away one that
			    # doesn't have a pilot entry so that we're
			    # not writing excessively to the pilot.
			    #
			    $del = $k2;
			    $del = $k1
				if (!defined($db->{$k1}{"pilot_id"}));
			    
			    if (defined($db->{$del}{"cm_id"}))
			    {
				my ($ret) = 
				    SyncCM::cm::deleteAppt($cm_session, 
							   $db->{$del});
				if ($ret)
				{
				    PilotMgr::msg("Unable to delete '" .
						  $db->{$del}{"cm_id"} .
						  "' on Desktop calendar " .
						  "[Error code $ret]");
				}
				else
				{
				    &log("cm_dupe", $db->{$del});
				    delete $db->{"cm_" . $db->{$del}{"cm_id"}};
				}
			    }
			    
			    if (defined($db->{$del}{"pilot_id"}))
			    {
				SyncCM::pilot::deleteAppt($pilot_dbhandle,
							  $db->{$del});
				&log("pilot_dupe", $db->{$del});
				delete $db->{"pilot_" . 
						 $db->{$del}{"pilot_id"}};
			    }
			}
			
			delete $db->{$del};

			if ($del eq $k2)
			{
			    $hash{$key}[$i] = $k1;
			}

			last;
		    }
		    $i++;
		}
	    }
	}
    }

    my ($sortfunc) = &byBeginDate($db);

    my (@keylist) = sort {&$sortfunc} grep(/^master/, keys %{$db});
    $count = 0;
    $count_max = scalar(@keylist);
    status("Synchronizing $count_max appointments", 0);
    foreach $key (@keylist)
    {
	next unless ($key =~ /^master/);

	$p_id = ""; $p_changes = "";
	$c_id = ""; $c_changes = "";

	unless ($count++ % &fake_ceil($count_max / 20))
	{
	    &checkCancel;
	    status("Synchronizing $count_max appointments",
		   int(100 * $count / $count_max));
	}

	if (defined($db->{$key}{"pilot_id"}))
	{
	    $p_id = "pilot_" . $db->{$key}{"pilot_id"};

	    if (defined($db->{$p_id}) && defined($db->{$p_id}{"changes"}))
	    {
		$p_changes = $db->{$p_id}{"changes"};
	    }
	}

	if (defined($db->{$key}{"cm_id"}))
	{
	    $c_id = "cm_" . $db->{$key}{"cm_id"};

	    if (defined($db->{$c_id}) && defined($db->{$c_id}{"changes"}))
	    {
		$c_changes = $db->{$c_id}{"changes"};
	    }
	}

	if (!$p_id && !$c_id)
	{
	    # Sanity check; it has no ids!
	    #
	    &log("delete_both", $db->{$key});
	    delete $db->{$key};
	}
	elsif ($p_id && !$c_id)
	{
	    # Exists in pilot, but not in cm
	    #
	    if ($p_changes)
	    {
		$db->{$key} = $p_changes;
		delete $db->{$p_id}{"changes"};
	    }

	    next unless &inRange($db->{$key});

	    $id = SyncCM::cm::createAppt($cm_session, $db->{$key});
	    if (!$id)
	    {
		PilotMgr::msg("Unable to create '$db->{$key}{description}' in Desktop calendar");
	    }
	    else
	    {
		&log("cm_create", $db->{$key});

		$db->{$key}{"cm_id"} = $id;
		$db->{"cm_$id"}{"timestamp"} = $info->{"thisSyncDate"};
		$db->{"cm_$id"}{"id"} = $key;
	    }
	}
	elsif (!$p_id && $c_id)
	{
	    # Exists in cm, but not in pilot
	    #
	    if ($c_changes)
	    {
		$db->{$key} = $c_changes;
		delete $db->{$c_id}{"changes"};
	    }

	    $id = SyncCM::pilot::createAppt($pilot_dbhandle, $db->{$key});
	    if (!defined($id))
	    {
		PilotMgr::msg("Unable to create '$db->{$key}{description}' in the Pilot");
	    }
	    else
	    {
		&log("pilot_create", $db->{$key});

		$db->{$key}{"pilot_id"} = $id;
		$db->{"pilot_$id"}{"timestamp"} = $info->{"thisSyncDate"};
		$db->{"pilot_$id"}{"id"} = $key;
	    }
	}
	elsif ((!defined($db->{$p_id}) || 
		(defined($db->{$p_id}) && 
		 $db->{$p_id}{"timestamp"} != $info->{"thisSyncDate"})) &&
	       (!defined($db->{$c_id}) ||
		(defined($db->{$c_id}) && 
		 $db->{$c_id}{"timestamp"} != $info->{"thisSyncDate"})))
	{
	    # Deleted on both sides
	    #
	    &log("delete_both", $db->{$key});
	    delete $db->{$p_id};
	    delete $db->{$c_id};
	    delete $db->{$key};
	}
	elsif (!defined($db->{$p_id}) || 
	       defined($db->{$p_id}) && 
	       $db->{$p_id}{"timestamp"} != $info->{"thisSyncDate"})
	{
	    if ($c_changes)
	    {
		$db->{$key} = $c_changes;
		delete $db->{$c_id}{"changes"};

		$id = SyncCM::pilot::createAppt($pilot_dbhandle, $db->{$key});
		if (!defined($id))
		{
		    PilotMgr::msg("Could not create $db->{$key}{description}");
		}
		else
		{
		    PilotMgr::msg("Appointment $db->{$key}{description} was deleted on the Pilot but changed on the Desktop calendar.  It has been recreated on the Pilot");
		    &log("pilot_create", $db->{$key});
		    $db->{$key}{"pilot_id"} = $id;
		    $db->{"pilot_$id"}{"timestamp"} = $info->{"thisSyncDate"};
		    $db->{"pilot_$id"}{"id"} = $key;
		}
	    }
	    else
	    {
		# Deleted on pilot, unchanged on CM
		#
		my ($ret);
		$ret = SyncCM::cm::deleteAppt($cm_session, $db->{$key});
		if ($ret != 0)
		{
		    PilotMgr::msg("Unable to delete '" .
				  "$db->{$key}{cm_id} " .
				  "'from Desktop calendar " .
				  "[Error code $ret]\n");
		}
		else
		{
		    &log("cm_delete", $db->{$key});
		    delete $db->{$p_id};
		    delete $db->{$c_id};
		    delete $db->{$key};
		}
	    }
	}
	elsif (!defined($db->{$c_id}) || 
	       defined($db->{$c_id}) && 
	       $db->{$c_id}{"timestamp"} != $info->{"thisSyncDate"})
	{
	    if ($p_changes)
	    {
		if (!&inRange($p_changes))
		{
		    # The new appt is not in the CM range, so this is
		    # functionally equivalent to a delete
		    #
		    SyncCM::cm::deleteAppt($cm_session, $db->{$key});

		    &log("cm_range", $db->{$key});

		    delete $db->{$p_id};
		    delete $db->{$c_id};
		    delete $db->{$key};
		    next;
		}

		$db->{$key} = $p_changes;
		delete $db->{$p_id}{"changes"};

		$id = SyncCM::cm::createAppt($cm_session, $db->{$key});
		if (!defined($id))
		{
		    PilotMgr::msg("Could not create $db->{$key}{description}");
		}
		else
		{
		    PilotMgr::msg("Appointment $db->{$key}{description} was deleted on Desktop calendar but changed on the Pilot.  It has been recreated on the Desktop calendar");
		    &log("cm_create", $db->{$key});
		    $db->{$key}{"cm_id"} = $id;
		    $db->{"cm_$id"}{"timestamp"} = $info->{"thisSyncDate"};
		    $db->{"cm_$id"}{"id"} = $key;
		}
	    }
	    else
	    {
		# Deleted on cm
		#
		my ($ret);
		$ret = SyncCM::pilot::deleteAppt($pilot_dbhandle, $db->{$key});

		if ($ret != 0)
		{
		    PilotMgr::msg("Unable to delete '$db->{$key}{pilot_id}' on pilot [Error code $ret]");
		}
		else
		{
		    &log("pilot_delete", $db->{$key});
		    delete $db->{$p_id};
		    delete $db->{$c_id};
		    delete $db->{$key};
		}
	    }
	}
	elsif ($p_changes && $c_changes)
	{
	    # Changed on both sides
	    #
	    # Turn the original into the pilot version and 
	    # add it to the desktop calendar
	    #
	    my ($err) = 0;

	    if (&inRange($p_changes))
	    {
		$db->{$key} = $db->{$p_id}{"changes"};
		delete $db->{$p_id}{"changes"};
		$id = SyncCM::cm::createAppt($cm_session, $db->{$key});
		if (!defined($id))
		{
		    PilotMgr::msg("Could not create " .
				      "$db->{$key}{description}");
		    $err = 1;
		}
		else
		{
		    &log("cm_create", $db->{$key});
		    
		    $db->{$key}{"cm_id"} = $id;
		    $db->{"cm_" . $id}{"id"} = $key;
		    $db->{"cm_" . $id}{"timestamp"} = $info->{"thisSyncDate"};
		}
	    }
	    else
	    {
		# Changed version from Pilot is out of range.
		# This is like a delete, except that the CM
		# version didn't change so don't delete that guy.
		#
		delete $db->{$p_id};
		delete $db->{$key};

		&log("cm_range", $db->{$key});
	    }
	    
	    # Create a new db record for the cm version
	    #
	    $key = "master_" . $db->{"ID_TRACK"}++;
	    $db->{$key} = $db->{$c_id}{"changes"};
	    delete $db->{$c_id}{"changes"};
	    $id = SyncCM::pilot::createAppt($pilot_dbhandle, $db->{$key});
	    if (!defined($id))
	    {
		PilotMgr::msg("Could not create $db->{$key}{description}");
		$err = 1;
	    }
	    else
	    {
		&log("pilot_create", $db->{$key});

		$db->{$key}{"pilot_id"} = $id;
		$db->{$c_id}{"id"} = $key; 
		$db->{"pilot_" . $id}{"id"} = $key;
		$db->{"pilot_" . $id}{"timestamp"} = $info->{"thisSyncDate"};
	    }

	    if ($err == 0)
	    {
		PilotMgr::msg("Appointment $db->{$key}{description} was changed on both the Desktop calendar and the Pilot.  It has been duplicated");
	    }
	}
	elsif ($p_changes)
	{
	    # Changed on Pilot
	    #
	    $orig_id = $db->{$key}{"cm_id"};

	    my ($orig_appt) = $db->{$key};

	    if (!&inRange($p_changes))
	    {
		# The new appt is not in the CM range, so this is
		# functionally equivalent to a delete
		#
		SyncCM::cm::deleteAppt($cm_session, $db->{$key});
		&log("cm_range", $db->{$key});
		delete $db->{$p_id};
		delete $db->{$c_id};
		delete $db->{$key};
		next;
	    }

	    $db->{$key} = $p_changes;
	    delete $db->{$p_id}{"changes"};

	    $id = SyncCM::cm::changeAppt($cm_session, $orig_appt, 
					 $db->{$key});
	    if (!defined($id))
	    {
		PilotMgr::msg("Could not change $db->{$key}{description}");
	    }
	    else
	    {
		&log("cm_change", $db->{$key});
		$db->{$key}{"cm_id"} = $id;

		if ($orig_id ne $id)
		{
		    delete $db->{"cm_" . $orig_id};
		}
	    }

	    $db->{"cm_$id"}{"timestamp"} = $info->{"thisSyncDate"};
	    $db->{"cm_$id"}{"id"} = $key;
	}
	elsif ($c_changes)
	{
	    # Changed on cm
	    #
	    $orig_id = $db->{$key}{"pilot_id"};

	    $db->{$key} = $c_changes;
	    delete $db->{$c_id}{"changes"};

	    $id = SyncCM::pilot::changeAppt($pilot_dbhandle, 
					    $db->{$key}, $orig_id);
	    if (!defined($id))
	    {
		PilotMgr::msg("Could not change $db->{$key}{description}");
	    }
	    else
	    {
		&log("pilot_change", $db->{$key});
		$db->{$key}{"pilot_id"} = $id;
	    }

	    if ("pilot_$id" ne "pilot_" . $db->{$key}{"pilot_id"})
	    {
		delete $db->{"pilot_" . $db->{$key}{"pilot_id"}};
	    }

	    $db->{"pilot_$id"}{"timestamp"} = $info->{"thisSyncDate"};
	    $db->{"pilot_$id"}{"id"} = $key;
	}
    }
}

sub log
{
    my ($key, $record) = @_;

    $STATS{$key}++;

    if ($PREFS->{"logChanges"})
    {
	my ($time) = PilotMgr::prettyTime($record->{begin});
	print LOGFD "$key: $time $record->{description}\n";
    }
}

sub partialMerge
{
    my ($master, $db, $name, $info) = @_;
    my ($key);

    # Remove all deleted records from the master
    #
    foreach $key (keys %$db)
    {
	if ($db->{$key} eq "DELETED")
	{
	    my ($tag) = $name . "_" . $key;
	    delete($master->{$tag})
		if (defined($master->{$tag}));
	    delete($db->{$key});
	}
    }

    # Merge on what's there
    #
    &merge($master, $db, $name, $info);

    # Mark all remaining keys as current (they weren't deleted
    # so they must still exist in the DB)
    #
    foreach $key (keys %$master)
    {
	next unless ($key =~ /${name}_/);

	$master->{$key}{"timestamp"} = $info->{"thisSyncDate"};
    }

    return $master;
}

sub byBeginDate
{
    my ($hash) = @_;

    return sub
    {
	return $hash->{$a}{"begin"} <=> $hash->{$b}{"begin"};
    }
}

sub merge
{
    my ($master, $db, $name, $info) = @_;
    my ($appt, $id, $key);

    foreach $key (keys %$db)
    {
	$appt = $db->{$key};
	$id = $name . "_" . $appt->{"${name}_id"};
	if (defined($master->{$id}))
	{
	    # $master->{$id} is the pilot/cm record.
	    #
	    # $master->{$id}->{"id"} is a pointer to a master record
	    # such as 'master_1234'
	    #
	    if (defined($master->{$master->{$id}->{"id"}}))
	    {
		# Ok, this record exists in the master.  
		# Check to see if it has changed.
		#
		if (&compareAppt($master->{$master->{$id}->{"id"}}, $appt))
		{
		    # Unchanged.  
		    #
		    $master->{$id}{"timestamp"} = $info->{"thisSyncDate"};
		}
		else
		{
		    # It's been changed.  Add this record to the
		    # change list for later reconciliation.
		    #
		    $master->{$id}{"timestamp"} = $info->{"thisSyncDate"};
		    $master->{$id}{"changes"} = {%$appt};
		}
	    }
	    else
	    {
		# Hmm.  We have a (pathological?) case where the pointer
		# exists, but the master does not.  Create a master
		# where the broken pointer points
		#
		$master->{$master->{$id}{"id"}} = {%$appt};
		$master->{$id}{"timestamp"} = $info->{"thisSyncDate"};
	    }
	}
	else
	{
	    # A new record!
	    #
	    $master->{$id}{"id"} = "master_" . $master->{"ID_TRACK"}++;
	    $master->{$master->{$id}{"id"}} = {%$appt};
	    $master->{$id}{"timestamp"} = $info->{"thisSyncDate"};
	}
    }
}



sub compareAppt
{
    my ($a, $b) = @_;

    unless(defined($a) && defined($b))
    {
	print "ERROR: in compareAppt($a, $b)\n";
	return;
    }

    my (%a) = %$a;
    my (%b) = %$b;
    my (%seen, $key);

    foreach $key (keys %a, keys %b)
    {
	next if $seen{$key}++;
	next if $key eq "changes";
	next if $key eq "pilot_id";
	next if $key eq "cm_id";

	return 0 unless (ref($a{$key}) eq ref($b{$key}));

	if (ref($a{$key}) eq "ARRAY")
	{
	    my (@tmp_a) = (@{$a{$key}});
	    my (@tmp_b) = (@{$b{$key}});

	    return 0 if (scalar(@tmp_a) ne scalar(@tmp_b));

	    while (scalar(@tmp_a))
	    {
		return 0 unless (&strEq(shift(@tmp_a), shift(@tmp_b)));
	    }
	}
	else
	{
	    return 0 unless (&strEq($a{$key}, $b{$key}));
	}
    }

    return 1;
}

sub strEq
{
    my ($x, $y) = @_;

    return 1 if (!defined($x) && !defined($y));
    return 0 if (!defined($x) || !defined($y));

    $x =~ tr/A-Z/a-z/;
    $x =~ s/\s//g;

    $y =~ tr/A-Z/a-z/;
    $y =~ s/\s//g;

    return 1 if ($x eq $y);
    return 0;
}

my (@PRIME) = 
(
    2, 3, 5, 7, 11, 13, 17, 19,
    23, 29, 31, 37, 41, 43, 47, 53,
    59, 61, 67, 71, 73, 79, 83, 89,
    97, 101, 103, 107, 109, 113, 127, 131,
    137, 139, 149, 151, 157, 163, 167, 173,
    179, 181, 191, 193, 197, 199, 211, 223,
    227, 229, 233, 239, 241, 251, 257, 263,
    269, 271, 277, 281, 283, 293, 307, 311,
    313, 317, 331, 337, 347, 349, 353, 359,
    367, 373, 379, 383, 389, 397, 401, 409,
    419, 421, 431, 433, 439, 443, 449, 457,
    461, 463, 467, 479, 487, 491, 499, 503,
    509, 521, 523, 541, 547, 557, 563, 569,
    571, 577, 587, 593, 599, 601, 607, 613,
    617, 619, 631, 641, 643, 647, 653, 659,
    661, 673, 677, 683, 691, 701, 709, 719,
    727, 733, 739, 743, 751, 757, 761, 769,
    773, 787, 797, 809, 811, 821, 823, 827,
    829, 839, 853, 857, 859, 863, 877, 881,
    883, 887, 907, 911, 919, 929, 937, 941,
    947, 953, 967, 971, 977, 983, 991, 997,
 );

sub stringHash
{
    my ($str) = @_;
    my ($i, $letter, $val);

    return 0 
	unless defined($str);
    
    $val = 0;
    $i = 1;

    # Make it lowercase and ignore spacing
    #
    $str =~ tr/A-Z/a-z/;
    $str =~ s/\s//g;

    foreach $letter (split(//, $str))
    {
	$val += $PRIME[$i % scalar(@PRIME)] * ord($letter);
	$i++;
    }

    return $val;
}

sub strToTick
{
    my ($str) = @_;
    my ($relative) = (0);

    # Check for relative instead of absolute date
    if ($str =~ s/^\+//)
    {
	$relative = 1;
    }
    elsif ($str =~ s/^\-//)
    {
	$relative = -1;
    }

    if ($str =~ m|(\d+)/(\d+)/(\d+)|)
    {
	my ($day, $mon, $year, $time) = ($2, $1, $3);

	unless ($relative)
	{
	    # Absolute date specified
	    if (length($year) == 4)
	    {
		$year -= 1900;
	    }

	    return -1
		unless ($mon && $day && $year);

	    eval
	    {
		$time = Time::Local::timelocal(0, 0, 0, $day, $mon-1, $year);
	    };
	    return -1 if ($@);

	    return $time;
	}
	else
	{
	    # Date specified is relative to current date
	    my ($SECSPERDAY, $SECSPERWEEK, $days) =
		(24 * 60 * 60, 7 * 24 * 60 * 60);

	    # THIS CALC IS APPROX- uses 31 days per month
	    $days = $year * 365 + $mon * 31 + $day;
	    $time = time + $relative * $days * $SECSPERDAY;

	    # round to a week
	    $time -= (localtime($time))[6] * $SECSPERDAY;
	    $time += $SECSPERWEEK if ($relative > 0);

	    return $time;
	}
    }
    else
    {
	return -1;
    }
}

sub inRange
{
    my ($appt) = @_;
    my ($begin) = $appt->{"begin"};

    return 1 if ($PREFS->{"syncRange"} eq "Sync All");

    if ($begin >= $gRangeBegin &&
	$begin <= $gRangeEnd)
    {
	return 1;
    }
}

sub debug
{
    my ($buf, @vals) = @_;
    my ($time, $pad);
    my ($i);

    return unless $DEBUG;

    for ($i = 0; $i < @vals; $i++)
    {
	$vals[$i] = &dump($vals[$i]);
    }

    $buf = sprintf($buf, @vals);

    if (open(FD, ">>$DEBUGFILE"))
    {
	chomp($buf);
	
	chop($time = `date +"%D %T`);
	$time .= "   ";
	$pad = " " x length($time);
	$buf =~ s/\n/\n$pad/g;
	print FD "$time$buf\n";
	close(FD);
    }
}

sub error
{
    my ($buf, @vals) = @_;
    my ($time, $pad);
    my ($i);

    for ($i = 0; $i < @vals; $i++)
    {
	$vals[$i] = &dump($vals[$i]);
    }

    $buf = sprintf($buf, @vals);

    if (open(FD, ">>$ERRORFILE"))
    {
	chomp($buf);
	
	chop($time = `date +"%D %T`);
	$time .= "   ";
	$pad = " " x length($time);
	$buf =~ s/\n/\n$pad/g;
	print FD "$time$buf\n";
	close(FD);
    }
}

sub dump
{
    my ($obj) = @_;
    $Data::Dumper::Purity = 1;
    $Data::Dumper::Indent = 1;
    $Data::Dumper::Deepcopy = 1;
    return Data::Dumper->Dumpxs([$obj], ['obj']);
}

sub status
{
    my ($msg, $perc) = @_;

    PilotMgr::status($msg, $perc);
}

sub showMemorySize
{
    my ($tag) = @_;

    my ($SIZE);
    chop($SIZE = `ps -p $$ -o vsz | tail -1`);
    print "[$tag] Size: $SIZE\n";
}

sub defaults
{
    return
    {
	"size_cm" => 0,
	"size_pilot" => 0,
	"cal" => $PREFS->{"cal"},
    }
}

sub cycleLogs
{
    if (-f $LOGFILE && (stat($LOGFILE))[7] > $LOGFILE_THRESHOLD)
    {
	rename($LOGFILE, $LOGFILE . ".old");
    }

    if (-f $ERRORFILE && (stat($ERRORFILE))[7] > $LOGFILE_THRESHOLD)
    {
	rename($ERRORFILE, $ERRORFILE . ".old");
    }
}


# POSIX's ceil is hosed on some systems
#
sub fake_ceil
{
    my ($val) = int($_[0]);

    return 1 if ($val == 0);
    return $val;
}

1;
