% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/transcriptToX.R
\name{transcriptToGenome}
\alias{transcriptToGenome}
\title{Map transcript-relative coordinates to genomic coordinates}
\usage{
transcriptToGenome(x, db, id = "name")
}
\arguments{
\item{x}{\code{IRanges} with the coordinates within the transcript. Coordinates
are counted from the start of the transcript (including the 5' UTR). The
Ensembl IDs of the corresponding transcripts have to be provided either
as \code{names} of the \code{IRanges}, or in one of its metadata columns.}

\item{db}{\code{EnsDb} object.}

\item{id}{\code{character(1)} specifying where the transcript identifier can be
found. Has to be either \code{"name"} or one of \code{colnames(mcols(prng))}.}
}
\value{
\code{GRangesList} with the same length (and order) than the input \code{IRanges}
\code{x}. Each \code{GRanges} in the \code{GRangesList} provides the genomic coordinates
corresponding to the provided within-transcript coordinates. The
original transcript ID and the transcript-relative coordinates are provided
as metadata columns as well as the ID of the individual exon(s). An empty
\code{GRanges} is returned for transcripts that can not be found in the database.
}
\description{
\code{transcriptToGenome} maps transcript-relative coordinates to genomic
coordinates. Provided coordinates are expected to be relative to the first
nucleotide of the \strong{transcript}, not the \strong{CDS}. CDS-relative coordinates
have to be converted to transcript-relative positions first with the
\code{\link[=cdsToTranscript]{cdsToTranscript()}} function.
}
\examples{

library(EnsDb.Hsapiens.v86)
## Restrict all further queries to chromosome x to speed up the examples
edbx <- filter(EnsDb.Hsapiens.v86, filter = ~ seq_name == "X")

## Below we map positions 1 to 5 within the transcript ENST00000381578 to
## the genome. The ID of the transcript has to be provided either as names
## or in one of the IRanges' metadata columns
txpos <- IRanges(start = 1, end = 5, names = "ENST00000381578")

transcriptToGenome(txpos, edbx)
## The object returns a GRangesList with the genomic coordinates, in this
## example the coordinates are within the same exon and map to a single
## genomic region.

## Next we map nucleotides 501 to 505 of ENST00000486554 to the genome
txpos <- IRanges(start = 501, end = 505, names = "ENST00000486554")

transcriptToGenome(txpos, edbx)
## The positions within the transcript are located within two of the
## transcripts exons and thus a `GRanges` of length 2 is returned.

## Next we map multiple regions, two within the same transcript and one
## in a transcript that does not exist.
txpos <- IRanges(start = c(501, 1, 5), end = c(505, 10, 6),
    names = c("ENST00000486554", "ENST00000486554", "some"))

res <- transcriptToGenome(txpos, edbx)

## The length of the result GRangesList has the same length than the
## input IRanges
length(res)

## The result for the last region is an empty GRanges, because the
## transcript could not be found in the database
res[[3]]

res
}
\seealso{
\code{\link[=cdsToTranscript]{cdsToTranscript()}} and \code{\link[=transcriptToCds]{transcriptToCds()}} for the mapping between
CDS- and transcript-relative coordinates.

Other coordinate mapping functions: 
\code{\link{cdsToTranscript}()},
\code{\link{genomeToProtein}()},
\code{\link{genomeToTranscript}()},
\code{\link{proteinToGenome}()},
\code{\link{proteinToTranscript}()},
\code{\link{transcriptToCds}()},
\code{\link{transcriptToProtein}()}
}
\author{
Johannes Rainer
}
\concept{coordinate mapping functions}
