///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "2D.h"
#include "rheolef/ublas_matrix_range.h"
using namespace rheolef;
using namespace std;
using namespace ublas;

// -----
// plane
// -----
//
//       [ 2D_00   2D_01   2D_02 ]
//  2D = [         2D_11   2D_12 ]
//       [                 2D_22 ]
//
// ordered by column: tau = 2D
//
//                  [    u0     u1     u2 ]
//
//  [   tau_00 ] =  [ 2*dx0               ]
//  [ 2*tau_01 ] =  [ 2*dx1  2*dx0        ]
//  [   tau_11 ] =  [        2*dx1        ]
//  [ 2*tau_02 ] =  [ 2*dx2         2*dx0 ]
//  [ 2*tau_12 ] =  [        2*dx2  2*dx1 ]
//  [   tau_22 ] =  [               2*dx2 ]
//
// since the tau_ij, i!=j components have a 2 factor in the
// mass matrix (see mass.cc and inv_mass.cc)
//
// ------------
// axisymmetric
// ------------
//       [ 2D_00   2D_01   0     ]
//  2D = [         2D_11   0     ]
//       [                 2D_22 ]
//
// ordered by column:
//
//                  [    u0     u1 ]
//
//  [   tau_00 ] =  [ 2*dx0        ]
//  [ 2*tau_01 ] =  [ 2*dx1  2*dx0 ]
//  [   tau_11 ] =  [        2*dx1 ]
//  [   tau_22 ] =  [  2/r         ]
//
// the weight is r*dr*dz => the 1/r will disappears for tau_22
//
void
_2D::operator() (const geo_element& K, matrix<Float>& dd) const
{
  switch (K.dimension()) {
    case 1:
      d_dx (K, dd, 0);
      dd *= 2;
      break;
    case 2: {
      matrix<Float> dx0, dx1;
      d_dx (K, dx0, 0);
      d_dx (K, dx1, 1);
      size_type ni = dx0.size1();
      size_type nj = dx0.size2();
      size_type n_comp;
      if (coordinate_system_type() == fem_helper::cartesian) n_comp=3;
      else n_comp=4; // rz or zr axisymmetry
      dd.resize (n_comp*ni, 2*nj);
      dd.clear();
      //  [   tau_00 ] =  [ 2*dx0        ]
      //  [ 2*tau_01 ] =  [ 2*dx1  2*dx0 ]
      //  [   tau_11 ] =  [        2*dx1 ]
      mr_set (dd, range(0,ni),     range(0,nj),       2.*dx0);
      mr_set (dd, range(ni,2*ni),  range(0,nj),       2.*dx1);
      mr_set (dd, range(ni,2*ni),  range(nj,2*nj),    2.*dx0);
      mr_set (dd, range(2*ni,3*ni),range(nj,2*nj),    2.*dx1);
      if (coordinate_system_type() == fem_helper::cartesian) break;
      // otherwise, add the (2/r) ur vr r dr dz = 2 ur vr dr dz 
      ublas::matrix<Float> m;
      set_use_coordinate_system_weight(false);
      build_scalar_mass (K, m);
      set_use_coordinate_system_weight(true);
      size_type k = 0;
      if (coordinate_system_type() == fem_helper::axisymmetric_zr) k++;
      mr_set (dd, range(3*ni,4*ni),    range(k*nj,(k+1)*nj),	2.*m);
      break;
    }
    case 3: {
      matrix<Float> dx0, dx1, dx2;
      d_dx (K, dx0, 0);
      d_dx (K, dx1, 1);
      d_dx (K, dx2, 2);
      size_type ni = dx0.size1();
      size_type nj = dx0.size2();
      dd.resize (6*ni, 3*nj);
      dd.clear();
      //  [   tau_00 ] =  [ 2*dx0               ]
      //  [ 2*tau_01 ] =  [ 2*dx1  2*dx0        ]
      //  [   tau_11 ] =  [        2*dx1        ]
      mr_set (dd, range(0,ni),     range(0,nj),       2.*dx0);
      mr_set (dd, range(ni,2*ni),  range(0,nj),       2.*dx1);
      mr_set (dd, range(ni,2*ni),  range(nj,2*nj),    2.*dx0);
      mr_set (dd, range(2*ni,3*ni),range(nj,2*nj),    2.*dx1);

      //  [ 2*tau_02 ] =  [ 2*dx2         2*dx0 ]
      //  [ 2*tau_12 ] =  [        2*dx2  2*dx1 ]
      //  [   tau_22 ] =  [               2*dx2 ]
      mr_set (dd, range(3*ni,4*ni),range(0,nj),       2.*dx2);
      mr_set (dd, range(3*ni,4*ni),range(2*nj,3*nj),  2.*dx0);
      mr_set (dd, range(4*ni,5*ni),range(nj,2*nj),    2.*dx2);
      mr_set (dd, range(4*ni,5*ni),range(2*nj,3*nj),  2.*dx1);
      mr_set (dd, range(5*ni,6*ni),range(2*nj,3*nj),  2.*dx2);
      break;
    }
    default: {
      fatal_macro ("form `2D' not supported in " << K.dimension() << "d geometry");
    }
  }
}
_2D::size_type
_2D::n_derivative() const
{
  return 1;
}
void
_2D::check_after_initialize () const
{
  size_type d = coordinate_dimension();
  check_macro (
	d == get_first_space().n_component(),
	"unsupported non-vectorial first space for `2D' form");
  check_macro (
	d*(d-1) <= get_second_space().n_component(),
	"unsupported non-tensorial space for `2D' form");
}
