#ifndef _RHEO_BASIS_ON_LATTICE_H
#define _RHEO_BASIS_ON_LATTICE_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

//! Pre-compute a piola transformation polynomial basis
//! on a lattice definied on the reference element.
//! The computation is performed one time for all
//! the first time the reference element appears.
//! Thus, when looping on the mesh for the Lagrange interpolation
//! on a Lagrange nodal basis : the lattice is the set of nodes
//! the evaluation of the fem polynomial basis.
//! The evaluation is performed only one time.

#include "rheolef/tensor.h"
#include "rheolef/basis.h"
#include "rheolef/quadrature.h"
namespace rheolef { 

// -----------------------------------------------------------------------
// basis evaluated on nodal lattice of Lagrange basis
// -----------------------------------------------------------------------
class basis_on_nodal_basis {
public:

    typedef std::vector<Float>::size_type size_type;
    typedef std::vector<Float>::const_iterator const_iterator;

// allocators:

    basis_on_nodal_basis ();
    basis_on_nodal_basis (const basis& nb, const basis& b);

// modifiers:

    void set (const basis& nb, const basis& b);

// accessors:

    const_iterator begin (reference_element hat_K, size_type q) const;
    const_iterator end   (reference_element hat_K, size_type q) const;

// data:
protected:
    basis _nb;
    basis _b;
    mutable std::vector<std::vector<Float> > _val [reference_element::max_variant];
    mutable std::vector<bool     >           _initialized;
    void _initialize (reference_element hat_K) const;
};
// -----------------------------------------------------------------------
// basis evaluated on lattice of quadrature formulae
// -----------------------------------------------------------------------
class basis_on_quadrature {
public:

    typedef std::vector<Float>::size_type size_type;
    typedef std::vector<Float>::const_iterator const_iterator;
    typedef std::vector<point>::const_iterator const_iterator_grad;
    typedef std::vector<tensor >::const_iterator const_iterator_hessian;

// allocators:

    basis_on_quadrature ();
    basis_on_quadrature (const quadrature<Float>& quad, const basis& b);

// modifiers:

    void set (const quadrature<Float>& quad, const basis& b);

// accessors:

    const basis get_basis() const;
    const_iterator begin (reference_element hat_K, size_type q) const;
    const_iterator end   (reference_element hat_K, size_type q) const;

    const_iterator_grad
    begin_grad (reference_element hat_K, size_type q) const;
    const_iterator_grad
    end_grad   (reference_element hat_K, size_type q) const;

    const_iterator_hessian
    begin_hessian (reference_element hat_K, size_type q) const;
    const_iterator_hessian
    end_hessian   (reference_element hat_K, size_type q) const;

// data:
protected:
    const quadrature<Float>* _p_quad;
    basis       _b;
    mutable std::vector<std::vector<Float> > _val      [reference_element::max_variant];
    mutable std::vector<std::vector<point> > _grad_val [reference_element::max_variant];
    mutable std::vector<std::vector<tensor > > _hessian_val [reference_element::max_variant];
    mutable std::vector<bool     >           _initialized;
    mutable std::vector<bool     >           _grad_initialized;
    mutable std::vector<bool     >           _hessian_initialized;
    void _initialize     (reference_element hat_K) const;
    void _grad_initialize (reference_element hat_K) const;
    void _hessian_initialize (reference_element hat_K) const;
};
// ----------------------------------------------------------
// inlined
// ----------------------------------------------------------
inline
basis_on_nodal_basis::basis_on_nodal_basis ()
 : _nb(), _b(), _val(),
   _initialized(reference_element::max_variant, false)
{
}
inline
basis_on_nodal_basis::basis_on_nodal_basis (const basis& nb, const basis& b)
 : _nb(nb), _b(b), _val(),
   _initialized(reference_element::max_variant, false)
{
}
inline
void
basis_on_nodal_basis::set (const basis& nb, const basis& b)
{
    _nb = nb;
    _b  = b;
    std::fill (_initialized.begin(), _initialized.end(), false);
}
inline
basis_on_quadrature::basis_on_quadrature ()
 : _p_quad(0), _b(), _val(), _grad_val(), _hessian_val(),
   _initialized(reference_element::max_variant, false),
   _grad_initialized(reference_element::max_variant, false),
   _hessian_initialized(reference_element::max_variant, false)
{
}
inline
basis_on_quadrature::basis_on_quadrature (const quadrature<Float>& quad, const basis& b)
 : _p_quad(&quad), _b(b), _val(), _grad_val(), _hessian_val(),
   _initialized(reference_element::max_variant, false),
   _grad_initialized(reference_element::max_variant, false),
   _hessian_initialized(reference_element::max_variant, false)
{
}
inline
void
basis_on_quadrature::set (const quadrature<Float>& quad, const basis& b)
{
    // data was not defined on this quadrature formulae : reset
    std::fill (_initialized.begin(), _initialized.end(), false);
    std::fill (_grad_initialized.begin(), _grad_initialized.end(), false);
    std::fill (_hessian_initialized.begin(), _hessian_initialized.end(), false);
    _p_quad = &quad;
    _b  = b;
}
inline
const basis
basis_on_quadrature::get_basis() const
{
  return _b;
}
}// namespace rheolef
#endif // _RHEO_BASIS_ON_LATTICE_H
