///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
//
// Fekete points for degree <= 19
//
// http://www.math.unipd.it/~alvise/software.html
// http://www.math.unipd.it/~alvise/SIMPLEX_SETS/set_simplex_leb_gll.m
// see BriSomVia-2012.pdf
//
#include "rheolef/point.h" 
#include "rheolef/compiler_eigen.h"

namespace rheolef {

template<class T>
void set_simplex_leb_gll (
  size_t                                          degree,
  Eigen::Matrix<point_basic<T>,Eigen::Dynamic,1>& pts)
{ 
     // SET OF POINTS FOR THE UNIT SIMPLEX, HAVING VERTICES 
     // [0,0], [1,0], [0,1] AND GAUSS-LEGENDRE-LOBATTO DISTRIBUTION ON THE
     // SIDES. THE LEBESGUE CONSTANTS ARE PARTICULARLY LOW.
     
     // INPUT: 
     // deg: DEGREE OF THE SET.
     
     // OUTPUT:
     // pts: CARTESIAN COORDINATES, STORED IN A "n x 2" MATRIX.
     // pts_bar: BARYCENTRIC COORDINATES, STORED IN A "n x 3" MATRIX.
     // stats_matrix: MATRIX WITH STATISTICS WITH COLUMNS THAT CONSIST OF
     // DEGREE, LEBESGUE CONSTANT, ABS. VALUE OF VANDERMONDE MATRIX "V" 
     // DETERMINANT (w.r.t DUBINER BASIS), CONDITIONING OF "V".
     
 	 // ------------------------------------------------- 
 	 // |DEGR|  LEBESGUE   | ABS.DET.VAND.| COND. VAND. |
 	 // ------------------------------------------------- 
 	 // |  1 | 1.00000e+00 | 5.87878e+01  | 3.70699e+00 |
 	 // |  2 | 1.66667e+00 | 3.13712e+04  | 1.01810e+01 |
 	 // |  3 | 2.11244e+00 | 3.44701e+08  | 2.03638e+01 |
 	 // |  4 | 2.58725e+00 | 8.99015e+13  | 3.91919e+01 |
 	 // |  5 | 3.08214e+00 | 6.66154e+20  | 5.23357e+01 |
 	 // |  6 | 3.59472e+00 | 1.87052e+29  | 6.79519e+01 |
 	 // |  7 | 4.13763e+00 | 1.88500e+39  | 8.89001e+01 |
 	 // |  8 | 5.21334e+00 | 8.03633e+50  | 1.22730e+02 |
 	 // |  9 | 5.51107e+00 | 2.22811e+64  | 1.36115e+02 |
 	 // | 10 | 5.93288e+00 | 4.57515e+79  | 1.79449e+02 |
 	 // | 11 | 6.66260e+00 | 4.43645e+96  | 1.75462e+02 |
 	 // | 12 | 7.12872e+00 | 1.69089e+115 | 2.44254e+02 |
 	 // | 13 | 7.74381e+00 | 5.70766e+135 | 2.90055e+02 |
 	 // | 14 | 8.30508e+00 | 9.94900e+157 | 3.09248e+02 |
 	 // | 15 | 9.06741e+00 | 1.88891e+182 | 3.63190e+02 |
 	 // | 16 | 8.58342e+00 | 2.70064e+209 | 4.26188e+02 |
 	 // | 17 | 9.11807e+00 | 7.85856e+237 | 4.33208e+02 |
 	 // | 18 | 9.88359e+00 | 1.43972e+268 | 4.89530e+02 |
 	 // ------------------------------------------------- 
     
  pts.resize((degree+1)*(degree+2)/2);
  switch (degree) { 
    case 1: {
        pts[ 0]={ 1.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[ 1]={ 0.00000000000000000000000000000000, 1.00000000000000000000000000000000};
        pts[ 2]={ 0.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        break;
    }
    case 2: {
        pts[ 0]={ 1.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[ 1]={ 0.50000000000000000000000000000000, 0.50000000000000000000000000000000};
        pts[ 2]={ 0.00000000000000000000000000000000, 1.00000000000000000000000000000000};
        pts[ 3]={ 0.50000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[ 4]={ 0.00000000000000000000000000000000, 0.50000000000000000000000000000000};
        pts[ 5]={ 0.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        break;
    }
    case 3: {
        pts[ 0]={ 1.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[ 1]={ 0.72360679193627464034221929978230, 0.27639320806372547068008316273335};
        pts[ 2]={ 0.27639319336619594391635246211081, 0.72360680663380394506134507537354};
        pts[ 3]={ 0.00000000000000000000000000000000, 1.00000000000000000000000000000000};
        pts[ 4]={ 0.72360679404346817111814971212880, 0.00000000000000000000000000000000};
        pts[ 5]={ 0.33333333186264507697060821556079, 0.33333332720603220389321563743579};
        pts[ 6]={ 0.00000000000000000000000000000000, 0.72360678327761007722074282355607};
        pts[ 7]={ 0.27639319567478171091323702057707, 0.00000000000000000000000000000000};
        pts[ 8]={ 0.00000000000000000000000000000000, 0.27639319806372530941018794692354};
        pts[ 9]={ 0.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        break;
    }
    case 4: {
        pts[ 0]={ 0.52713337628873258022821346457931, 0.25042035280809399333534770448750};
        pts[ 1]={ 0.22213227970507090636154146068293, 0.52701897628726301903157036576886};
        pts[ 2]={ 0.25064641146071803268213784576801, 0.22206512551486515261167653534358};
        pts[ 3]={ 1.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[ 4]={ 0.00000000000000000000000000000000, 1.00000000000000000000000000000000};
        pts[ 5]={ 0.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[ 6]={ 0.17267316464601151215774166303163, 0.82732683535398843233110710571054};
        pts[ 7]={ 0.50000000000000000000000000000000, 0.50000000000000000000000000000000};
        pts[ 8]={ 0.82732683535398854335340956822620, 0.17267316464601145664659043177380};
        pts[ 9]={ 0.17267316464601151215774166303163, 0.00000000000000000000000000000000};
        pts[10]={ 0.50000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[11]={ 0.82732683535398854335340956822620, 0.00000000000000000000000000000000};
        pts[12]={ 0.00000000000000000000000000000000, 0.17267316464601151215774166303163};
        pts[13]={ 0.00000000000000000000000000000000, 0.50000000000000000000000000000000};
        pts[14]={ 0.00000000000000000000000000000000, 0.82732683535398854335340956822620};
        break;
    }
    case 5: {
        pts[ 0]={ 0.65277296395854245147916117275599, 0.17390610719231461289702167505311};
        pts[ 1]={ 0.41208046019357558176210432066000, 0.17359751549075921883691364655533};
        pts[ 2]={ 0.17584125722114934964324106658751, 0.65072062351987736139591333994758};
        pts[ 3]={ 0.41216359270868008213284383600694, 0.41115982426654845705016327883641};
        pts[ 4]={ 0.17253420630624241582928846128198, 0.17579824177976616228491479887452};
        pts[ 5]={ 0.17596596710904199856173590887920, 0.41519137386566823400002590460645};
        pts[ 6]={ 1.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[ 7]={ 0.00000000000000000000000000000000, 1.00000000000000000000000000000000};
        pts[ 8]={ 0.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[ 9]={ 0.11747233803526770101743892382729, 0.88252766196473231286034888398717};
        pts[10]={ 0.35738424175967747897786352950789, 0.64261575824032246551098523923429};
        pts[11]={ 0.64261575824032257653328770174994, 0.35738424175967742346671229825006};
        pts[12]={ 0.88252766196473242388265134650283, 0.11747233803526757611734865349717};
        pts[13]={ 0.11747233803526770101743892382729, 0.00000000000000000000000000000000};
        pts[14]={ 0.35738424175967747897786352950789, 0.00000000000000000000000000000000};
        pts[15]={ 0.64261575824032257653328770174994, 0.00000000000000000000000000000000};
        pts[16]={ 0.88252766196473242388265134650283, 0.00000000000000000000000000000000};
        pts[17]={ 0.00000000000000000000000000000000, 0.11747233803526770101743892382729};
        pts[18]={ 0.00000000000000000000000000000000, 0.35738424175967747897786352950789};
        pts[19]={ 0.00000000000000000000000000000000, 0.64261575824032257653328770174994};
        pts[20]={ 0.00000000000000000000000000000000, 0.88252766196473242388265134650283};
        break;
    }
    case 6: {
        pts[ 0]={ 0.33333514472668901440499666932737, 0.33335513526117654947711343993433};
        pts[ 1]={ 0.32498508069223069893993738332938, 0.13627277086369060654647000774276};
        pts[ 2]={ 0.32619161439757726750343636012985, 0.54650094974932839075876245260588};
        pts[ 3]={ 0.11996895293184388353591174336543, 0.75279618524932379042269303681678};
        pts[ 4]={ 0.53867333727111699204215256031603, 0.32504414231257972023314550824580};
        pts[ 5]={ 0.12732523903869941150901468063239, 0.32619845539428404812554163072491};
        pts[ 6]={ 0.12722227928154689058359849695989, 0.12000491662690664085300085162089};
        pts[ 7]={ 0.13629899825536365143285877365997, 0.53869622819612583697335139731877};
        pts[ 8]={ 0.54646952078678912023690372734563, 0.12732893408538545343411385601939};
        pts[ 9]={ 0.75271127399216808395721045599203, 0.12726119072647257279840005139704};
        pts[10]={ 1.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[11]={ 0.00000000000000000000000000000000, 1.00000000000000000000000000000000};
        pts[12]={ 0.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[13]={ 0.08488805186071646247114586003590, 0.91511194813928353752885413996410};
        pts[14]={ 0.26557560326464280109348692349158, 0.73442439673535719890651307650842};
        pts[15]={ 0.50000000000000011102230246251565, 0.49999999999999988897769753748435};
        pts[16]={ 0.73442439673535719890651307650842, 0.26557560326464280109348692349158};
        pts[17]={ 0.91511194813928331548424921493279, 0.08488805186071668451575078506721};
        pts[18]={ 0.08488805186071646247114586003590, 0.00000000000000000000000000000000};
        pts[19]={ 0.26557560326464280109348692349158, 0.00000000000000000000000000000000};
        pts[20]={ 0.50000000000000011102230246251565, 0.00000000000000000000000000000000};
        pts[21]={ 0.73442439673535719890651307650842, 0.00000000000000000000000000000000};
        pts[22]={ 0.91511194813928331548424921493279, 0.00000000000000000000000000000000};
        pts[23]={ 0.00000000000000000000000000000000, 0.08488805186071646247114586003590};
        pts[24]={ 0.00000000000000000000000000000000, 0.26557560326464280109348692349158};
        pts[25]={ 0.00000000000000000000000000000000, 0.50000000000000011102230246251565};
        pts[26]={ 0.00000000000000000000000000000000, 0.73442439673535719890651307650842};
        pts[27]={ 0.00000000000000000000000000000000, 0.91511194813928331548424921493279};
        break;
    }
    case 7: {
        pts[ 0]={ 0.08920695277808962653232072170795, 0.81959064206673137853442767664092};
        pts[ 1]={ 0.26237620402564731936223552111187, 0.10113113042681398656696956095402};
        pts[ 2]={ 0.10640883777534533571884622915604, 0.25381255303781857790568210475612};
        pts[ 3]={ 0.45393752517901786447396261792164, 0.26248795504383504173517849267228};
        pts[ 4]={ 0.44131487577167266556088520701451, 0.45805270293935052006162322868477};
        pts[ 5]={ 0.09120006007472451770023980088808, 0.08921395590945475906341499694463};
        pts[ 6]={ 0.63652758907417983369469993704115, 0.26235092312033286843231394414033};
        pts[ 7]={ 0.63974558349857890693357376221684, 0.10641458455373478675376475166559};
        pts[ 8]={ 0.28355350836197051878073693842452, 0.45395547714644107761827740432636};
        pts[ 9]={ 0.10061575398947217307998869273433, 0.44132716362437623969228184250824};
        pts[10]={ 0.81960688132349979184709809487686, 0.09119423396918546753653345149360};
        pts[11]={ 0.10113367543353200816991233068620, 0.63651800236482647399327561288374};
        pts[12]={ 0.25381791542609144496367434840067, 0.63976115027860425144012879172806};
        pts[13]={ 0.26245092856611090814666908954678, 0.28356768319778830056421270455758};
        pts[14]={ 0.45804618569974664410437981132418, 0.10060335397281700009486371527601};
        pts[15]={ 1.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[16]={ 0.00000000000000000000000000000000, 1.00000000000000000000000000000000};
        pts[17]={ 0.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[18]={ 0.06412992574519671418187272138312, 0.93587007425480328581812727861688};
        pts[19]={ 0.20414990928342899301206614381954, 0.79585009071657097923235824055155};
        pts[20]={ 0.39535039104876057436399605649058, 0.60464960895123942563600394350942};
        pts[21]={ 0.60464960895123931461370148099377, 0.39535039104876068538629851900623};
        pts[22]={ 0.79585009071657131229926562809851, 0.20414990928342868770073437190149};
        pts[23]={ 0.93587007425480317479582481610123, 0.06412992574519682520417518389877};
        pts[24]={ 0.06412992574519671418187272138312, 0.00000000000000000000000000000000};
        pts[25]={ 0.20414990928342899301206614381954, 0.00000000000000000000000000000000};
        pts[26]={ 0.39535039104876057436399605649058, 0.00000000000000000000000000000000};
        pts[27]={ 0.60464960895123931461370148099377, 0.00000000000000000000000000000000};
        pts[28]={ 0.79585009071657131229926562809851, 0.00000000000000000000000000000000};
        pts[29]={ 0.93587007425480317479582481610123, 0.00000000000000000000000000000000};
        pts[30]={ 0.00000000000000000000000000000000, 0.06412992574519671418187272138312};
        pts[31]={ 0.00000000000000000000000000000000, 0.20414990928342899301206614381954};
        pts[32]={ 0.00000000000000000000000000000000, 0.39535039104876057436399605649058};
        pts[33]={ 0.00000000000000000000000000000000, 0.60464960895123931461370148099377};
        pts[34]={ 0.00000000000000000000000000000000, 0.79585009071657131229926562809851};
        pts[35]={ 0.00000000000000000000000000000000, 0.93587007425480317479582481610123};
        break;
    }
    case 8: {
        pts[ 0]={ 0.73902788828497634199266030918807, 0.21077216238173276940948142055277};
        pts[ 1]={ 0.30148586947333583241359633575485, 0.52561231178461031987581009161659};
        pts[ 2]={ 0.64599298345089672945817937943502, 0.20163245998015180693307968340378};
        pts[ 3]={ 0.09465455153292157142175256012706, 0.82672991426915143353681969529134};
        pts[ 4]={ 0.44885416842888509192377455292444, 0.32263207939699595216964667088178};
        pts[ 5]={ 0.28287144260197433220938023623603, 0.66904386378939173507518489714130};
        pts[ 6]={ 0.48964733924590392311060327301675, 0.42906127098337604186539806505607};
        pts[ 7]={ 0.05016312191507527051825832131726, 0.73907048902612704210923766368069};
        pts[ 8]={ 0.17288730300636978576278579566861, 0.30147179058286766029084446927300};
        pts[ 9]={ 0.15236743222623846949481674073468, 0.64599503596700869056945748525322};
        pts[10]={ 0.07864945908433830512596074413523, 0.09464407772943810026600175433487};
        pts[11]={ 0.22849645272874871682056152621954, 0.44886480555703706096792870994250};
        pts[12]={ 0.04808016734010669285259709226921, 0.28285660240774085361081802147964};
        pts[13]={ 0.08128987927664718671216803613788, 0.48963993144526068990174394457426};
        pts[14]={ 0.21075609299984254807469596926239, 0.05017281064612942598213152223252};
        pts[15]={ 0.52561971987592104316178165390738, 0.17288073002878520667557893375488};
        pts[16]={ 0.20164627313113298034963349891768, 0.15236022744483410096272280043195};
        pts[17]={ 0.82673720818600959781008441495942, 0.07861452818420192056869666430430};
        pts[18]={ 0.32264248641333231892858179890027, 0.22850405184638331035351654918486};
        pts[19]={ 0.66904576349975897198163465873222, 0.04808345589080521148561686572975};
        pts[20]={ 0.42904795513803406503683390837978, 0.08129134375082752717034395573137};
        pts[21]={ 1.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[22]={ 0.00000000000000000000000000000000, 1.00000000000000000000000000000000};
        pts[23]={ 0.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[24]={ 0.05012100229426996733650412352290, 0.94987899770573003266349587647710};
        pts[25]={ 0.16140686024463110626037121164700, 0.83859313975536886598405317272409};
        pts[26]={ 0.31844126808691092245240383817872, 0.68155873191308913305874739307910};
        pts[27]={ 0.50000000000000011102230246251565, 0.49999999999999988897769753748435};
        pts[28]={ 0.68155873191308913305874739307910, 0.31844126808691086694125260692090};
        pts[29]={ 0.83859313975536897700635563523974, 0.16140686024463102299364436476026};
        pts[30]={ 0.94987899770572981061889095144579, 0.05012100229427018938110904855421};
        pts[31]={ 0.05012100229426996733650412352290, 0.00000000000000000000000000000000};
        pts[32]={ 0.16140686024463110626037121164700, 0.00000000000000000000000000000000};
        pts[33]={ 0.31844126808691092245240383817872, 0.00000000000000000000000000000000};
        pts[34]={ 0.50000000000000011102230246251565, 0.00000000000000000000000000000000};
        pts[35]={ 0.68155873191308913305874739307910, 0.00000000000000000000000000000000};
        pts[36]={ 0.83859313975536897700635563523974, 0.00000000000000000000000000000000};
        pts[37]={ 0.94987899770572981061889095144579, 0.00000000000000000000000000000000};
        pts[38]={ 0.00000000000000000000000000000000, 0.05012100229426996733650412352290};
        pts[39]={ 0.00000000000000000000000000000000, 0.16140686024463110626037121164700};
        pts[40]={ 0.00000000000000000000000000000000, 0.31844126808691092245240383817872};
        pts[41]={ 0.00000000000000000000000000000000, 0.50000000000000011102230246251565};
        pts[42]={ 0.00000000000000000000000000000000, 0.68155873191308913305874739307910};
        pts[43]={ 0.00000000000000000000000000000000, 0.83859313975536897700635563523974};
        pts[44]={ 0.00000000000000000000000000000000, 0.94987899770572981061889095144579};
        break;
    }
    case 9: {
        pts[ 0]={ 0.33335050528164394956220917265455, 0.33332948549794233494836248610227};
        pts[ 1]={ 0.86294871862490141189994119486073, 0.06678142058431162098042221941796};
        pts[ 2]={ 0.17687331848291526492644720747194, 0.12745710143173966111262984668429};
        pts[ 3]={ 0.06786042562094916907877717449082, 0.57708854093221628733800798727316};
        pts[ 4]={ 0.05288039332007089809994937468218, 0.20599391406261199910510129029717};
        pts[ 5]={ 0.14199816800467993105350217319938, 0.24573088341163071279815710568073};
        pts[ 6]={ 0.19775615397440954312813232718327, 0.03742259972486576546923942032663};
        pts[ 7]={ 0.52426092039671068345541016242350, 0.27375876897388407016364908486139};
        pts[ 8]={ 0.17774983239331398898741554148728, 0.39582884111955335404431366441713};
        pts[ 9]={ 0.53700223304945471358706754472223, 0.05057034179712253940675026342433};
        pts[10]={ 0.07025808041599575226587859333449, 0.86299703783092640563268105324823};
        pts[11]={ 0.69567973868437571471901037511998, 0.17686395241289989965771667357330};
        pts[12]={ 0.35504392624532227218381308375683, 0.06786745457999755759548321520924};
        pts[13]={ 0.74111485932138054799622750579147, 0.05288051498051561660762587280260};
        pts[14]={ 0.61226595820105234047048270440428, 0.14200560678928467184256589916913};
        pts[15]={ 0.76482437500905420257879541168222, 0.19775946787315307195953550944978};
        pts[16]={ 0.20197392429131855173807252867846, 0.52425288109500534083906586602097};
        pts[17]={ 0.42638974135065038639780254925427, 0.17776325993930863766934180603130};
        pts[18]={ 0.41239367537751758785447009358904, 0.53703821052084887988087302801432};
        pts[19]={ 0.06676582155982571653574098036188, 0.07025273146455188189030138801172};
        pts[20]={ 0.12744960259236817745609471330681, 0.69569707961456328515481573049328};
        pts[21]={ 0.57711221759448161350292139104567, 0.35503478478041172738954855958582};
        pts[22]={ 0.20601395268986477660178024962079, 0.74111600822282197587043128805817};
        pts[23]={ 0.24571555217665208559196798887569, 0.61227877390013207836716446763603};
        pts[24]={ 0.03741837763868938182554302329663, 0.76482740160434503629005575930933};
        pts[25]={ 0.27376462188881256487604787253076, 0.20197902420522070854325136224361};
        pts[26]={ 0.39581016736008056700057977650431, 0.42642339069862805756017110070388};
        pts[27]={ 0.05056413179301780247421049807599, 0.41239184622903946042526968085440};
        pts[28]={ 1.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[29]={ 0.00000000000000000000000000000000, 1.00000000000000000000000000000000};
        pts[30]={ 0.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[31]={ 0.04023304591677052260978442177475, 0.95976695408322942881795825087465};
        pts[32]={ 0.13061306744724759942855030203646, 0.86938693255275234506029846670572};
        pts[33]={ 0.26103752509477778920299329001864, 0.73896247490522215528585547872353};
        pts[34]={ 0.41736052116680649737290309531090, 0.58263947883319344711594567343127};
        pts[35]={ 0.58263947883319333609364321091562, 0.41736052116680666390635678908438};
        pts[36]={ 0.73896247490522237733046040375484, 0.26103752509477762266953959624516};
        pts[37]={ 0.86938693255275234506029846670572, 0.13061306744724765493970153329428};
        pts[38]={ 0.95976695408322920677335332584335, 0.04023304591677079322664667415665};
        pts[39]={ 0.04023304591677052260978442177475, 0.00000000000000000000000000000000};
        pts[40]={ 0.13061306744724759942855030203646, 0.00000000000000000000000000000000};
        pts[41]={ 0.26103752509477778920299329001864, 0.00000000000000000000000000000000};
        pts[42]={ 0.41736052116680649737290309531090, 0.00000000000000000000000000000000};
        pts[43]={ 0.58263947883319333609364321091562, 0.00000000000000000000000000000000};
        pts[44]={ 0.73896247490522237733046040375484, 0.00000000000000000000000000000000};
        pts[45]={ 0.86938693255275234506029846670572, 0.00000000000000000000000000000000};
        pts[46]={ 0.95976695408322920677335332584335, 0.00000000000000000000000000000000};
        pts[47]={ 0.00000000000000000000000000000000, 0.04023304591677052260978442177475};
        pts[48]={ 0.00000000000000000000000000000000, 0.13061306744724759942855030203646};
        pts[49]={ 0.00000000000000000000000000000000, 0.26103752509477778920299329001864};
        pts[50]={ 0.00000000000000000000000000000000, 0.41736052116680649737290309531090};
        pts[51]={ 0.00000000000000000000000000000000, 0.58263947883319333609364321091562};
        pts[52]={ 0.00000000000000000000000000000000, 0.73896247490522237733046040375484};
        pts[53]={ 0.00000000000000000000000000000000, 0.86938693255275234506029846670572};
        pts[54]={ 0.00000000000000000000000000000000, 0.95976695408322920677335332584335};
        break;
    }
    case 10: {
        pts[ 0]={ 0.38406096582502163849781595672539, 0.23474765255526919283113329584012};
        pts[ 1]={ 0.38119466728172124447837632033043, 0.38406330980953901077512568917882};
        pts[ 2]={ 0.23474117091297660997462060095131, 0.38118228336079873486141877947375};
        pts[ 3]={ 0.90243852912188127213255484093679, 0.04158966131295448870330844215459};
        pts[ 4]={ 0.05596928591350596587927768155168, 0.90244214267376721139157780271489};
        pts[ 5]={ 0.04158490873523290609625746583333, 0.05596275747554655538573697981519};
        pts[ 6]={ 0.13652319268672216678872644024523, 0.05848932762702901966278545842215};
        pts[ 7]={ 0.80498259526330917523750940745231, 0.13651832875069658257416449487209};
        pts[ 8]={ 0.05850525880060711281238639003277, 0.80498079308799808107011131141917};
        pts[ 9]={ 0.68091527200394486030177176871803, 0.11568955889340039322465969462428};
        pts[10]={ 0.20339281387361696218363249499816, 0.68091712278457805496145738288760};
        pts[11]={ 0.11569526907306046381584963000932, 0.20339270114802737721149128446996};
        pts[12]={ 0.18264203580857607533971531665884, 0.12302439158105102356977056388132};
        pts[13]={ 0.69434538257006594985654146512388, 0.18263816034906410479266014590394};
        pts[14]={ 0.12301616648278329568810818273050, 0.69432387966191522199466135134571};
        pts[15]={ 0.38032706433595131478853090811754, 0.10153439236068378948019130803004};
        pts[16]={ 0.51813949109686741056890468826168, 0.38031521997933798173718855650804};
        pts[17]={ 0.10154147982460165478002522831957, 0.51813824024113575017480570750195};
        pts[18]={ 0.54930352842529783785607833124232, 0.22312821630701878961033912673884};
        pts[19]={ 0.22757728847490968360212093557493, 0.54929842426932362631220030380064};
        pts[20]={ 0.22313684850936207904048558248178, 0.22757158395435561626385378986015};
        pts[21]={ 0.26887607690634096613280235033017, 0.03651675023081095428079123621501};
        pts[22]={ 0.69461022058805821366433974617394, 0.26887364420153464505958140762232};
        pts[23]={ 0.03651885767544953836916121758804, 0.69461386517446466193348442175193};
        pts[24]={ 0.67847724809510845389581845665816, 0.03421210127716058785329877878212};
        pts[25]={ 0.28730258137478770219530588292400, 0.67849031634150991365572735958267};
        pts[26]={ 0.03421113842215239148059069407282, 0.28730784960218458401470797980437};
        pts[27]={ 0.52058064938215187211767442931887, 0.09782476248506824512141122340836};
        pts[28]={ 0.38158435037973337244920912780799, 0.52058700781525291123585930108675};
        pts[29]={ 0.09782900418885427662463172282514, 0.38157726960785326442149312242691};
        pts[30]={ 0.47190451063706917445017552381614, 0.03622610157484423137885798382740};
        pts[31]={ 0.49187290508160352864308606513077, 0.47190612383830643494775358703919};
        pts[32]={ 0.03621738318676882145696893644526, 0.49186241634122401800155444107077};
        pts[33]={ 0.79619601528787520283714229663019, 0.05816796887186514303236251066664};
        pts[34]={ 0.14564929336248674807130498720653, 0.79618137501966990132729051765637};
        pts[35]={ 0.05816441991840956560189113133674, 0.14565541862762565461508756925468};
        pts[36]={ 1.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[37]={ 0.96700071520402963720641764666652, 0.00000000000000000000000000000000};
        pts[38]={ 0.89224173683157226300011188868666, 0.00000000000000000000000000000000};
        pts[39]={ 0.78261766349810257814567648892989, 0.00000000000000000000000000000000};
        pts[40]={ 0.64787906779346970953525897130021, 0.00000000000000000000000000000000};
        pts[41]={ 0.50000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[42]={ 0.35212093220653029046474102869979, 0.00000000000000000000000000000000};
        pts[43]={ 0.21738233650189747736547474232793, 0.00000000000000000000000000000000};
        pts[44]={ 0.10775826316842779251103934257117, 0.00000000000000000000000000000000};
        pts[45]={ 0.03299928479597041830473358459130, 0.00000000000000000000000000000000};
        pts[46]={ 0.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[47]={ 0.00000000000000000000000000000000, 1.00000000000000000000000000000000};
        pts[48]={ 0.00000000000000000000000000000000, 0.96700071520402963720641764666652};
        pts[49]={ 0.00000000000000000000000000000000, 0.89224173683157226300011188868666};
        pts[50]={ 0.00000000000000000000000000000000, 0.78261766349810257814567648892989};
        pts[51]={ 0.00000000000000000000000000000000, 0.64787906779346970953525897130021};
        pts[52]={ 0.00000000000000000000000000000000, 0.50000000000000000000000000000000};
        pts[53]={ 0.00000000000000000000000000000000, 0.35212093220653029046474102869979};
        pts[54]={ 0.00000000000000000000000000000000, 0.21738233650189747736547474232793};
        pts[55]={ 0.00000000000000000000000000000000, 0.10775826316842779251103934257117};
        pts[56]={ 0.00000000000000000000000000000000, 0.03299928479597041830473358459130};
        pts[57]={ 0.03299928479597041830473358459130, 0.96700071520402963720641764666652};
        pts[58]={ 0.10775826316842779251103934257117, 0.89224173683157226300011188868666};
        pts[59]={ 0.21738233650189747736547474232793, 0.78261766349810257814567648892989};
        pts[60]={ 0.35212093220653029046474102869979, 0.64787906779346970953525897130021};
        pts[61]={ 0.50000000000000000000000000000000, 0.50000000000000000000000000000000};
        pts[62]={ 0.64787906779346970953525897130021, 0.35212093220653029046474102869979};
        pts[63]={ 0.78261766349810257814567648892989, 0.21738233650189747736547474232793};
        pts[64]={ 0.89224173683157226300011188868666, 0.10775826316842779251103934257117};
        pts[65]={ 0.96700071520402963720641764666652, 0.03299928479597041830473358459130};
        break;
    }
    case 11: {
        pts[ 0]={ 0.69351770198338746897803730462329, 0.08437752842834599087407099204938};
        pts[ 1]={ 0.22210476958826652627010389551288, 0.69351770198338746897803730462329};
        pts[ 2]={ 0.08437752842834599087407099204938, 0.22210476958826652627010389551288};
        pts[ 3]={ 0.04031147049090379186342403272647, 0.04031147049090379186342403272647};
        pts[ 4]={ 0.91937705901819244402872755017597, 0.04031147049090379186342403272647};
        pts[ 5]={ 0.04031147049090379186342403272647, 0.91937705901819244402872755017597};
        pts[ 6]={ 0.74553090101456620519826401505270, 0.02729822949153083364026706192362};
        pts[ 7]={ 0.22717086949390297156980977888452, 0.74553090101456620519826401505270};
        pts[ 8]={ 0.02729822949153083364026706192362, 0.22717086949390297156980977888452};
        pts[ 9]={ 0.56495374320220637098799443265307, 0.04413159917798487735796086894879};
        pts[10]={ 0.39091465761980875859293860230537, 0.56495374320220637098799443265307};
        pts[11]={ 0.04413159917798487735796086894879, 0.39091465761980875859293860230537};
        pts[12]={ 0.82495126960113840386412675798056, 0.05072213402228054057729877968086};
        pts[13]={ 0.12432659637658105555857446233858, 0.82495126960113840386412675798056};
        pts[14]={ 0.05072213402228054057729877968086, 0.12432659637658105555857446233858};
        pts[15]={ 0.15969983813122679006113457944593, 0.10321977918324988510168083166718};
        pts[16]={ 0.73708038268552333871497239670134, 0.15969983813122679006113457944593};
        pts[17]={ 0.10321977918324988510168083166718, 0.73708038268552333871497239670134};
        pts[18]={ 0.55092060836475587048965962821967, 0.12365705769670772007362558042587};
        pts[19]={ 0.32542233393853642331450259916892, 0.55092060836475587048965962821967};
        pts[20]={ 0.12365705769670772007362558042587, 0.32542233393853642331450259916892};
        pts[21]={ 0.42737137848053580180263111287786, 0.26830798423248086725934058449639};
        pts[22]={ 0.30432063728698338644917953388358, 0.42737137848053580180263111287786};
        pts[23]={ 0.26830798423248086725934058449639, 0.30432063728698338644917953388358};
        pts[24]={ 0.37495473164237586027880411165825, 0.19113296643987093270666832722782};
        pts[25]={ 0.43391230191775326252567879237176, 0.37495473164237586027880411165825};
        pts[26]={ 0.19113296643987093270666832722782, 0.43391230191775326252567879237176};
        pts[27]={ 0.22948861577152193658157841582579, 0.02952208441716225664763761926679};
        pts[28]={ 0.74098929981131578248465530123212, 0.22948861577152193658157841582579};
        pts[29]={ 0.02952208441716225664763761926679, 0.74098929981131578248465530123212};
        pts[30]={ 0.40327364621373307240759231717675, 0.02750554948766919735603231345067};
        pts[31]={ 0.56922080429859778227807964867679, 0.40327364621373307240759231717675};
        pts[32]={ 0.02750554948766919735603231345067, 0.56922080429859778227807964867679};
        pts[33]={ 0.10894013417964920620306656928733, 0.04895820529689973860110185910344};
        pts[34]={ 0.84210166052345103437914985988755, 0.10894013417964920620306656928733};
        pts[35]={ 0.04895820529689973860110185910344, 0.84210166052345103437914985988755};
        pts[36]={ 0.32070128023529026473070757674577, 0.09558815442834922571080369380070};
        pts[37]={ 0.58371056533636056506963996071136, 0.32070128023529026473070757674577};
        pts[38]={ 0.09558815442834922571080369380070, 0.58371056533636056506963996071136};
        pts[39]={ 0.20667240615042631324271837911510, 0.20667240615042631324271837911510};
        pts[40]={ 0.58665518769914726249226077925414, 0.20667240615042631324271837911510};
        pts[41]={ 0.20667240615042631324271837911510, 0.58665518769914726249226077925414};
        pts[42]={ 0.45023971782244887052115700498689, 0.08842665456174254867693917958604};
        pts[43]={ 0.46133362761580859467969162324152, 0.45023971782244887052115700498689};
        pts[44]={ 0.08842665456174254867693917958604, 0.46133362761580859467969162324152};
        pts[45]={ 1.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[46]={ 0.97244963611144108472927882758086, 0.00000000000000000000000000000000};
        pts[47]={ 0.90963966082200331531026904485771, 0.00000000000000000000000000000000};
        pts[48]={ 0.81643807651593025553893312462606, 0.00000000000000000000000000000000};
        pts[49]={ 0.69976547048267445649827322995407, 0.00000000000000000000000000000000};
        pts[50]={ 0.56827646642746376670629615546204, 0.00000000000000000000000000000000};
        pts[51]={ 0.43172353357253623329370384453796, 0.00000000000000000000000000000000};
        pts[52]={ 0.30023452951732554350172677004593, 0.00000000000000000000000000000000};
        pts[53]={ 0.18356192348406968894991564411612, 0.00000000000000000000000000000000};
        pts[54]={ 0.09036033917799668468973095514229, 0.00000000000000000000000000000000};
        pts[55]={ 0.02755036388855891527072117241914, 0.00000000000000000000000000000000};
        pts[56]={ 0.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[57]={ 0.00000000000000000000000000000000, 1.00000000000000000000000000000000};
        pts[58]={ 0.00000000000000000000000000000000, 0.97244963611144108472927882758086};
        pts[59]={ 0.00000000000000000000000000000000, 0.90963966082200331531026904485771};
        pts[60]={ 0.00000000000000000000000000000000, 0.81643807651593025553893312462606};
        pts[61]={ 0.00000000000000000000000000000000, 0.69976547048267445649827322995407};
        pts[62]={ 0.00000000000000000000000000000000, 0.56827646642746376670629615546204};
        pts[63]={ 0.00000000000000000000000000000000, 0.43172353357253623329370384453796};
        pts[64]={ 0.00000000000000000000000000000000, 0.30023452951732554350172677004593};
        pts[65]={ 0.00000000000000000000000000000000, 0.18356192348406968894991564411612};
        pts[66]={ 0.00000000000000000000000000000000, 0.09036033917799668468973095514229};
        pts[67]={ 0.00000000000000000000000000000000, 0.02755036388855891527072117241914};
        pts[68]={ 0.02755036388855891527072117241914, 0.97244963611144108472927882758086};
        pts[69]={ 0.09036033917799668468973095514229, 0.90963966082200331531026904485771};
        pts[70]={ 0.18356192348406968894991564411612, 0.81643807651593025553893312462606};
        pts[71]={ 0.30023452951732554350172677004593, 0.69976547048267445649827322995407};
        pts[72]={ 0.43172353357253623329370384453796, 0.56827646642746376670629615546204};
        pts[73]={ 0.56827646642746376670629615546204, 0.43172353357253623329370384453796};
        pts[74]={ 0.69976547048267445649827322995407, 0.30023452951732554350172677004593};
        pts[75]={ 0.81643807651593025553893312462606, 0.18356192348406968894991564411612};
        pts[76]={ 0.90963966082200331531026904485771, 0.09036033917799668468973095514229};
        pts[77]={ 0.97244963611144108472927882758086, 0.02755036388855891527072117241914};
        break;
    }
    case 12: {
        pts[ 0]={ 0.33333333333333331482961625624739, 0.33333333333333331482961625624739};
        pts[ 1]={ 0.59580556948913909742771011224249, 0.15884144436411473666481697364361};
        pts[ 2]={ 0.90773827285623887028975786961382, 0.04786878168438524899785946331576};
        pts[ 3]={ 0.38660093339102252762984335276997, 0.42318727425997726410145105546690};
        pts[ 4]={ 0.09879451733539568181186041329056, 0.18188416511355362525925727368303};
        pts[ 5]={ 0.09054619448832809669447385658714, 0.74278808765462212004848652213695};
        pts[ 6]={ 0.37669205501454516848980347276665, 0.12559642125338588147798191130278};
        pts[ 7]={ 0.49467415574720130511110482984805, 0.09887135718071371481840969863697};
        pts[ 8]={ 0.13526709910352810450895333360677, 0.03199991091388797170846913786590};
        pts[ 9]={ 0.03684706179902393768932356010737, 0.10886644255912340806258242764670};
        pts[10]={ 0.57475183658016226750930854905164, 0.03233727936972005667648488724808};
        pts[11]={ 0.73237211330090690175609324796824, 0.03061660026751795993238403070791};
        pts[12]={ 0.09496880111811395464016527512285, 0.81203295416178344900970387243433};
        pts[13]={ 0.69791973518465955095280150999315, 0.27487447564560740786987480532844};
        pts[14]={ 0.47190812210997368225662285112776, 0.26692550559951672140002187916252};
        pts[15]={ 0.07945705664599639339584769004432, 0.27857501080175034458008553883701};
        pts[16]={ 0.08075538595871376013057840737019, 0.63561434433040064906350607998320};
        pts[17]={ 0.03687312995329386500209523092053, 0.54279424713712587369229822797934};
        pts[18]={ 0.61496795363012668467206367495237, 0.23043458729110896654468376709701};
        pts[19]={ 0.24535298614674616590747291411390, 0.59580556948913909742771011224249};
        pts[20]={ 0.04439294545937588071238266707041, 0.90773827285623887028975786961382};
        pts[21]={ 0.19021179234900020826870559176314, 0.38660093339102252762984335276997};
        pts[22]={ 0.71932131755105066517330669739749, 0.09879451733539568181186041329056};
        pts[23]={ 0.16666571785704975550146400564699, 0.09054619448832809669447385658714};
        pts[24]={ 0.49771152373206895003221461593057, 0.37669205501454516848980347276665};
        pts[25]={ 0.40645448707208498007048547151498, 0.49467415574720130511110482984805};
        pts[26]={ 0.83273298998258393766036533634178, 0.13526709910352810450895333360677};
        pts[27]={ 0.85428649564185266118698791615316, 0.03684706179902393768932356010737};
        pts[28]={ 0.39291088405011764805863094807137, 0.57475183658016226750930854905164};
        pts[29]={ 0.23701128643157515218931052913831, 0.73237211330090690175609324796824};
        pts[30]={ 0.09299824472010254083897962118499, 0.09496880111811395464016527512285};
        pts[31]={ 0.02720578916973304117732368467841, 0.69791973518465955095280150999315};
        pts[32]={ 0.26116637229050959634335526970972, 0.47190812210997368225662285112776};
        pts[33]={ 0.64196793255225315100176430860301, 0.07945705664599639339584769004432};
        pts[34]={ 0.28363026971088556305033989701769, 0.08075538595871376013057840737019};
        pts[35]={ 0.42033262290958028906118215672905, 0.03687312995329386500209523092053};
        pts[36]={ 0.15459745907876434878325255795062, 0.61496795363012668467206367495237};
        pts[37]={ 0.15884144436411473666481697364361, 0.24535298614674616590747291411390};
        pts[38]={ 0.04786878168438524899785946331576, 0.04439294545937588071238266707041};
        pts[39]={ 0.42318727425997726410145105546690, 0.19021179234900020826870559176314};
        pts[40]={ 0.18188416511355362525925727368303, 0.71932131755105066517330669739749};
        pts[41]={ 0.74278808765462212004848652213695, 0.16666571785704975550146400564699};
        pts[42]={ 0.12559642125338588147798191130278, 0.49771152373206895003221461593057};
        pts[43]={ 0.09887135718071371481840969863697, 0.40645448707208498007048547151498};
        pts[44]={ 0.03199991091388797170846913786590, 0.83273298998258393766036533634178};
        pts[45]={ 0.10886644255912340806258242764670, 0.85428649564185266118698791615316};
        pts[46]={ 0.03233727936972005667648488724808, 0.39291088405011764805863094807137};
        pts[47]={ 0.03061660026751795993238403070791, 0.23701128643157515218931052913831};
        pts[48]={ 0.81203295416178344900970387243433, 0.09299824472010254083897962118499};
        pts[49]={ 0.27487447564560740786987480532844, 0.02720578916973304117732368467841};
        pts[50]={ 0.26692550559951672140002187916252, 0.26116637229050959634335526970972};
        pts[51]={ 0.27857501080175034458008553883701, 0.64196793255225315100176430860301};
        pts[52]={ 0.63561434433040064906350607998320, 0.28363026971088556305033989701769};
        pts[53]={ 0.54279424713712587369229822797934, 0.42033262290958028906118215672905};
        pts[54]={ 0.23043458729110896654468376709701, 0.15459745907876434878325255795062};
        pts[55]={ 1.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[56]={ 0.00000000000000000000000000000000, 1.00000000000000000000000000000000};
        pts[57]={ 0.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[58]={ 0.02334507667891804957860379943213, 0.97665492332108194695194924861426};
        pts[59]={ 0.07682621767406377921716398304852, 0.92317378232593627629398724820931};
        pts[60]={ 0.15690576545912141015470808724785, 0.84309423454087861760086752838106};
        pts[61]={ 0.25854508945433207767194971893332, 0.74145491054566792232805028106668};
        pts[62]={ 0.37535653494687992726142056199024, 0.62464346505312007273857943800976};
        pts[63]={ 0.49999999999999988897769753748435, 0.50000000000000011102230246251565};
        pts[64]={ 0.62464346505311996171627697549411, 0.37535653494688003828372302450589};
        pts[65]={ 0.74145491054566803335035274358233, 0.25854508945433196664964725641767};
        pts[66]={ 0.84309423454087883964547245341237, 0.15690576545912116035452754658763};
        pts[67]={ 0.92317378232593583220477739814669, 0.07682621767406416779522260185331};
        pts[68]={ 0.97665492332108194695194924861426, 0.02334507667891805304805075138574};
        pts[69]={ 0.02334507667891804957860379943213, 0.00000000000000000000000000000000};
        pts[70]={ 0.07682621767406377921716398304852, 0.00000000000000000000000000000000};
        pts[71]={ 0.15690576545912141015470808724785, 0.00000000000000000000000000000000};
        pts[72]={ 0.25854508945433207767194971893332, 0.00000000000000000000000000000000};
        pts[73]={ 0.37535653494687992726142056199024, 0.00000000000000000000000000000000};
        pts[74]={ 0.49999999999999988897769753748435, 0.00000000000000000000000000000000};
        pts[75]={ 0.62464346505311996171627697549411, 0.00000000000000000000000000000000};
        pts[76]={ 0.74145491054566803335035274358233, 0.00000000000000000000000000000000};
        pts[77]={ 0.84309423454087883964547245341237, 0.00000000000000000000000000000000};
        pts[78]={ 0.92317378232593583220477739814669, 0.00000000000000000000000000000000};
        pts[79]={ 0.97665492332108194695194924861426, 0.00000000000000000000000000000000};
        pts[80]={ 0.00000000000000000000000000000000, 0.02334507667891804957860379943213};
        pts[81]={ 0.00000000000000000000000000000000, 0.07682621767406377921716398304852};
        pts[82]={ 0.00000000000000000000000000000000, 0.15690576545912141015470808724785};
        pts[83]={ 0.00000000000000000000000000000000, 0.25854508945433207767194971893332};
        pts[84]={ 0.00000000000000000000000000000000, 0.37535653494687992726142056199024};
        pts[85]={ 0.00000000000000000000000000000000, 0.49999999999999988897769753748435};
        pts[86]={ 0.00000000000000000000000000000000, 0.62464346505311996171627697549411};
        pts[87]={ 0.00000000000000000000000000000000, 0.74145491054566803335035274358233};
        pts[88]={ 0.00000000000000000000000000000000, 0.84309423454087883964547245341237};
        pts[89]={ 0.00000000000000000000000000000000, 0.92317378232593583220477739814669};
        pts[90]={ 0.00000000000000000000000000000000, 0.97665492332108194695194924861426};
        break;
    }
    case 13: {
        pts[ 0]={ 0.48254395679038292188423042716749, 0.02180164938752872372251800925369};
        pts[ 1]={ 0.49565439382208842378219060265110, 0.48254395679038292188423042716749};
        pts[ 2]={ 0.02180164938752872372251800925369, 0.49565439382208842378219060265110};
        pts[ 3]={ 0.27035531005290119876960375222552, 0.02118779795175457572597998989750};
        pts[ 4]={ 0.70845689199534411795156074731494, 0.27035531005290119876960375222552};
        pts[ 5]={ 0.02118779795175457572597998989750, 0.70845689199534411795156074731494};
        pts[ 6]={ 0.37175753808656258891929269339016, 0.04080287762488947478356493547835};
        pts[ 7]={ 0.58743958428854803344165702583268, 0.37175753808656258891929269339016};
        pts[ 8]={ 0.04080287762488947478356493547835, 0.58743958428854803344165702583268};
        pts[ 9]={ 0.27949525761827420655691867068526, 0.06864720203665519715485743290628};
        pts[10]={ 0.65185754034507059628822389640845, 0.27949525761827420655691867068526};
        pts[11]={ 0.06864720203665519715485743290628, 0.65185754034507059628822389640845};
        pts[12]={ 0.77520423964579876496827637311071, 0.03294538357494599528640932817325};
        pts[13]={ 0.19185037677925523280642039480881, 0.77520423964579876496827637311071};
        pts[14]={ 0.03294538357494599528640932817325, 0.19185037677925523280642039480881};
        pts[15]={ 0.58840573881630142683007989035104, 0.06712503774907915832237392805837};
        pts[16]={ 0.34446922343461938709197056596167, 0.58840573881630142683007989035104};
        pts[17]={ 0.06712503774907915832237392805837, 0.34446922343461938709197056596167};
        pts[18]={ 0.13305442687245128130513194264495, 0.02293678414999906697846476788527};
        pts[19]={ 0.84400878897754960661359291407280, 0.13305442687245128130513194264495};
        pts[20]={ 0.02293678414999906697846476788527, 0.84400878897754960661359291407280};
        pts[21]={ 0.33808504766535157903817321312090, 0.21810242661584300360999577605980};
        pts[22]={ 0.44381252571880536184067977956147, 0.33808504766535157903817321312090};
        pts[23]={ 0.21810242661584300360999577605980, 0.44381252571880536184067977956147};
        pts[24]={ 0.18116385225097764166690694764839, 0.15120191226325155331444705097965};
        pts[25]={ 0.66763423548577083277422161700088, 0.18116385225097764166690694764839};
        pts[26]={ 0.15120191226325155331444705097965, 0.66763423548577083277422161700088};
        pts[27]={ 0.32183509682876670909834615486034, 0.13300919379865164260401400042610};
        pts[28]={ 0.54515570937258173156436669160030, 0.32183509682876670909834615486034};
        pts[29]={ 0.13300919379865164260401400042610, 0.54515570937258173156436669160030};
        pts[30]={ 0.03961855581957707955753633655149, 0.02989107137428834159997848018975};
        pts[31]={ 0.93049037280613455802580347153707, 0.03961855581957707955753633655149};
        pts[32]={ 0.02989107137428834159997848018975, 0.93049037280613455802580347153707};
        pts[33]={ 0.14714091454213237830650484738726, 0.06254456537779155833600697178554};
        pts[34]={ 0.79031452008007607723527598864166, 0.14714091454213237830650484738726};
        pts[35]={ 0.06254456537779155833600697178554, 0.79031452008007607723527598864166};
        pts[36]={ 0.45815699799597475738721641391749, 0.15974444694755424167809110258531};
        pts[37]={ 0.38209855505647100093469248349720, 0.45815699799597475738721641391749};
        pts[38]={ 0.15974444694755424167809110258531, 0.38209855505647100093469248349720};
        pts[39]={ 0.08027843740008795692109089259247, 0.05953784448029870307594890732616};
        pts[40]={ 0.86018371811961324979733944928739, 0.08027843740008795692109089259247};
        pts[41]={ 0.05953784448029870307594890732616, 0.86018371811961324979733944928739};
        pts[42]={ 0.68979175262087999431770413139020, 0.06684334085711833572940321346323};
        pts[43]={ 0.24336490652200165607510484733211, 0.68979175262087999431770413139020};
        pts[44]={ 0.06684334085711833572940321346323, 0.24336490652200165607510484733211};
        pts[45]={ 0.60321881295188140015284261608031, 0.14801120509892598531287433161197};
        pts[46]={ 0.24876998194919261453428305230773, 0.60321881295188140015284261608031};
        pts[47]={ 0.14801120509892598531287433161197, 0.24876998194919261453428305230773};
        pts[48]={ 0.43929956988621743363410132587887, 0.07084823734066676703680087712200};
        pts[49]={ 0.48985219277311581320688560481358, 0.43929956988621743363410132587887};
        pts[50]={ 0.07084823734066676703680087712200, 0.48985219277311581320688560481358};
        pts[51]={ 0.23516747335230311910514444662113, 0.23516125307306637304272101118841};
        pts[52]={ 0.52967127357463050785213454219047, 0.23516747335230311910514444662113};
        pts[53]={ 0.23516125307306637304272101118841, 0.52967127357463050785213454219047};
        pts[54]={ 0.75875390221366512122358471970074, 0.08393722540159115341751316918817};
        pts[55]={ 0.15730887238474372535890211111109, 0.75875390221366512122358471970074};
        pts[56]={ 0.08393722540159115341751316918817, 0.15730887238474372535890211111109};
        pts[57]={ 0.40235017434608083553015944744402, 0.26067230244277200812419437170320};
        pts[58]={ 0.33697752321114710083449494959495, 0.40235017434608083553015944744402};
        pts[59]={ 0.26067230244277200812419437170320, 0.33697752321114710083449494959495};
        pts[60]={ 0.88476340755026594564469633041881, 0.03411072930099425665195767010118};
        pts[61]={ 0.08112586314873979076445209557278, 0.88476340755026594564469633041881};
        pts[62]={ 0.03411072930099425665195767010118, 0.08112586314873979076445209557278};
        pts[63]={ 0.64827611647845850839644299412612, 0.02230602753970508181358134436323};
        pts[64]={ 0.32941785598183642713721042127872, 0.64827611647845850839644299412612};
        pts[65]={ 0.02230602753970508181358134436323, 0.32941785598183642713721042127872};
        pts[66]={ 1.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[67]={ 0.97996752263363040569288386905100, 0.00000000000000000000000000000000};
        pts[68]={ 0.93390052691517366767470775812399, 0.00000000000000000000000000000000};
        pts[69]={ 0.86443429954566308737184954225086, 0.00000000000000000000000000000000};
        pts[70]={ 0.77531970146432360735389011097141, 0.00000000000000000000000000000000};
        pts[71]={ 0.67136200667135637054627750330837, 0.00000000000000000000000000000000};
        pts[72]={ 0.55816593444185191863482486951398, 0.00000000000000000000000000000000};
        pts[73]={ 0.44183406555814808136517513048602, 0.00000000000000000000000000000000};
        pts[74]={ 0.32863799332864357394257126543380, 0.00000000000000000000000000000000};
        pts[75]={ 0.22468029853567644815726112028642, 0.00000000000000000000000000000000};
        pts[76]={ 0.13556570045433691262815045774914, 0.00000000000000000000000000000000};
        pts[77]={ 0.06609947308482638783644347313384, 0.00000000000000000000000000000000};
        pts[78]={ 0.02003247736636953879596489969117, 0.00000000000000000000000000000000};
        pts[79]={ 0.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[80]={ 0.00000000000000000000000000000000, 1.00000000000000000000000000000000};
        pts[81]={ 0.00000000000000000000000000000000, 0.97996752263363040569288386905100};
        pts[82]={ 0.00000000000000000000000000000000, 0.93390052691517366767470775812399};
        pts[83]={ 0.00000000000000000000000000000000, 0.86443429954566308737184954225086};
        pts[84]={ 0.00000000000000000000000000000000, 0.77531970146432360735389011097141};
        pts[85]={ 0.00000000000000000000000000000000, 0.67136200667135637054627750330837};
        pts[86]={ 0.00000000000000000000000000000000, 0.55816593444185191863482486951398};
        pts[87]={ 0.00000000000000000000000000000000, 0.44183406555814808136517513048602};
        pts[88]={ 0.00000000000000000000000000000000, 0.32863799332864357394257126543380};
        pts[89]={ 0.00000000000000000000000000000000, 0.22468029853567644815726112028642};
        pts[90]={ 0.00000000000000000000000000000000, 0.13556570045433691262815045774914};
        pts[91]={ 0.00000000000000000000000000000000, 0.06609947308482638783644347313384};
        pts[92]={ 0.00000000000000000000000000000000, 0.02003247736636953879596489969117};
        pts[93]={ 0.02003247736636953879596489969117, 0.97996752263363040569288386905100};
        pts[94]={ 0.06609947308482638783644347313384, 0.93390052691517366767470775812399};
        pts[95]={ 0.13556570045433691262815045774914, 0.86443429954566308737184954225086};
        pts[96]={ 0.22468029853567644815726112028642, 0.77531970146432360735389011097141};
        pts[97]={ 0.32863799332864357394257126543380, 0.67136200667135637054627750330837};
        pts[98]={ 0.44183406555814808136517513048602, 0.55816593444185191863482486951398};
        pts[99]={ 0.55816593444185191863482486951398, 0.44183406555814808136517513048602};
        pts[100]={ 0.67136200667135637054627750330837, 0.32863799332864357394257126543380};
        pts[101]={ 0.77531970146432360735389011097141, 0.22468029853567644815726112028642};
        pts[102]={ 0.86443429954566308737184954225086, 0.13556570045433691262815045774914};
        pts[103]={ 0.93390052691517366767470775812399, 0.06609947308482638783644347313384};
        pts[104]={ 0.97996752263363040569288386905100, 0.02003247736636953879596489969117};
        break;
    }
    case 14: {
        pts[ 0]={ 0.04625042453028135880366278342990, 0.01977030152020083086927826343526};
        pts[ 1]={ 0.93397927394951774093811991406255, 0.04625042453028135880366278342990};
        pts[ 2]={ 0.01977030152020083086927826343526, 0.93397927394951774093811991406255};
        pts[ 3]={ 0.92632659737287770251867868864792, 0.01820016267574078167035267483698};
        pts[ 4]={ 0.05547323995138151581096863651510, 0.92632659737287770251867868864792};
        pts[ 5]={ 0.01820016267574078167035267483698, 0.05547323995138151581096863651510};
        pts[ 6]={ 0.32703462743708006099652152443014, 0.02836070535722972490999538308643};
        pts[ 7]={ 0.64460466720569020715458918857621, 0.32703462743708006099652152443014};
        pts[ 8]={ 0.02836070535722972490999538308643, 0.64460466720569020715458918857621};
        pts[ 9]={ 0.60061284473864329225278879675898, 0.01814314606602200966034565965401};
        pts[10]={ 0.38124400919533468767852468772617, 0.60061284473864329225278879675898};
        pts[11]={ 0.01814314606602200966034565965401, 0.38124400919533468767852468772617};
        pts[12]={ 0.52180999892365897974144672843977, 0.06190681978029687754228049811900};
        pts[13]={ 0.41628318129604413577737886953400, 0.52180999892365897974144672843977};
        pts[14]={ 0.06190681978029687754228049811900, 0.41628318129604413577737886953400};
        pts[15]={ 0.32395972089484870171105512781651, 0.29417523955654206746501699853980};
        pts[16]={ 0.38186503954860923082392787364370, 0.32395972089484870171105512781651};
        pts[17]={ 0.29417523955654206746501699853980, 0.38186503954860923082392787364370};
        pts[18]={ 0.28514037742473380410146432950569, 0.07584624717245121161557364075634};
        pts[19]={ 0.63901337540281488713844737503678, 0.28514037742473380410146432950569};
        pts[20]={ 0.07584624717245121161557364075634, 0.63901337540281488713844737503678};
        pts[21]={ 0.61318041430696523796939345629653, 0.06727346851679433248349937457533};
        pts[22]={ 0.31954611717624042954710716912814, 0.61318041430696523796939345629653};
        pts[23]={ 0.06727346851679433248349937457533, 0.31954611717624042954710716912814};
        pts[24]={ 0.73400669070495827561728674481856, 0.01894488439901751211658087470369};
        pts[25]={ 0.24704842489602421573557933243137, 0.73400669070495827561728674481856};
        pts[26]={ 0.01894488439901751211658087470369, 0.24704842489602421573557933243137};
        pts[27]={ 0.40635332371784116389790142420679, 0.07985847987007041459683875928022};
        pts[28]={ 0.51378819641208839374968420088408, 0.40635332371784116389790142420679};
        pts[29]={ 0.07985847987007041459683875928022, 0.51378819641208839374968420088408};
        pts[30]={ 0.12900790858202898503392930251721, 0.03252985109902303134665757511357};
        pts[31]={ 0.83846224031894789341379237157525, 0.12900790858202898503392930251721};
        pts[32]={ 0.03252985109902303134665757511357, 0.83846224031894789341379237157525};
        pts[33]={ 0.21380717697122103571061302318412, 0.03335666682746493694189382495097};
        pts[34]={ 0.75283615620131394408076630497817, 0.21380717697122103571061302318412};
        pts[35]={ 0.03335666682746493694189382495097, 0.75283615620131394408076630497817};
        pts[36]={ 0.16975705122904435895314634308306, 0.12348723696213652345488043238220};
        pts[37]={ 0.70675571180881913146976103234920, 0.16975705122904435895314634308306};
        pts[38]={ 0.12348723696213652345488043238220, 0.70675571180881913146976103234920};
        pts[39]={ 0.87568274655439903852993666077964, 0.05694225904903640889109794898104};
        pts[40]={ 0.06737499439656455257896539023932, 0.87568274655439903852993666077964};
        pts[41]={ 0.05694225904903640889109794898104, 0.06737499439656455257896539023932};
        pts[42]={ 0.83115363065833536282411841966677, 0.02826552006289960103369018895592};
        pts[43]={ 0.14058084927876504655053224723815, 0.83115363065833536282411841966677};
        pts[44]={ 0.02826552006289960103369018895592, 0.14058084927876504655053224723815};
        pts[45]={ 0.51248647535317315426794948507450, 0.12169602901547478535082547068669};
        pts[46]={ 0.36581749563135207425901285205327, 0.51248647535317315426794948507450};
        pts[47]={ 0.12169602901547478535082547068669, 0.36581749563135207425901285205327};
        pts[48]={ 0.78872899184734601707447154694819, 0.09207025500684888619229440109848};
        pts[49]={ 0.11920075314580509673323405195333, 0.78872899184734601707447154694819};
        pts[50]={ 0.09207025500684888619229440109848, 0.11920075314580509673323405195333};
        pts[51]={ 0.49301718193275584356527474483300, 0.19295747045566474797695377674245};
        pts[52]={ 0.31402534761157946396892270968237, 0.49301718193275584356527474483300};
        pts[53]={ 0.19295747045566474797695377674245, 0.31402534761157946396892270968237};
        pts[54]={ 0.32480621143418386687073962093564, 0.13314682439648198308823623392527};
        pts[55]={ 0.54204696416933417779659976076800, 0.32480621143418386687073962093564};
        pts[56]={ 0.13314682439648198308823623392527, 0.54204696416933417779659976076800};
        pts[57]={ 0.64001591711854499688882924601785, 0.13719362348223684833214974787552};
        pts[58]={ 0.22279045939921815477902100610663, 0.64001591711854499688882924601785};
        pts[59]={ 0.13719362348223684833214974787552, 0.22279045939921815477902100610663};
        pts[60]={ 0.27227276488947277366392540898232, 0.21552003579406989186573184724693};
        pts[61]={ 0.51220719931645741773706959065748, 0.27227276488947277366392540898232};
        pts[62]={ 0.21552003579406989186573184724693, 0.51220719931645741773706959065748};
        pts[63]={ 0.22634306457364944398236161759996, 0.16293922236272717629645967463148};
        pts[64]={ 0.61071771306362332421002747651073, 0.22634306457364944398236161759996};
        pts[65]={ 0.16293922236272717629645967463148, 0.61071771306362332421002747651073};
        pts[66]={ 0.45350667224656393550219490862219, 0.03243544401597714399443361799058};
        pts[67]={ 0.51405788373745897601452270464506, 0.45350667224656393550219490862219};
        pts[68]={ 0.03243544401597714399443361799058, 0.51405788373745897601452270464506};
        pts[69]={ 0.71854633569919634794587182113901, 0.06703560495633466287390689331005};
        pts[70]={ 0.21441805934446900305800909336540, 0.71854633569919634794587182113901};
        pts[71]={ 0.06703560495633466287390689331005, 0.21441805934446900305800909336540};
        pts[72]={ 0.15839343085936782862255256532080, 0.07074495301407555947204031099318};
        pts[73]={ 0.77086161612655657027204370024265, 0.15839343085936782862255256532080};
        pts[74]={ 0.07074495301407555947204031099318, 0.77086161612655657027204370024265};
        pts[75]={ 0.42060092672949378478364224065444, 0.16959660738383425249153901859245};
        pts[76]={ 0.40980246588667196272481874075311, 0.42060092672949378478364224065444};
        pts[77]={ 0.16959660738383425249153901859245, 0.40980246588667196272481874075311};
        pts[78]={ 1.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[79]={ 0.98262296325191922363728735945188, 0.00000000000000000000000000000000};
        pts[80]={ 0.94254102211148815815278112495434, 0.00000000000000000000000000000000};
        pts[81]={ 0.88175984497590764732422030647285, 0.00000000000000000000000000000000};
        pts[82]={ 0.80312660273492286755470104253618, 0.00000000000000000000000000000000};
        pts[83]={ 0.71031902735683627447826893330785, 0.00000000000000000000000000000000};
        pts[84]={ 0.60767697768189710139097314822720, 0.00000000000000000000000000000000};
        pts[85]={ 0.50000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[86]={ 0.39232302231810289860902685177280, 0.00000000000000000000000000000000};
        pts[87]={ 0.28968097264316372552173106669215, 0.00000000000000000000000000000000};
        pts[88]={ 0.19687339726507713244529895746382, 0.00000000000000000000000000000000};
        pts[89]={ 0.11824015502409240818693092478497, 0.00000000000000000000000000000000};
        pts[90]={ 0.05745897788851184184721887504566, 0.00000000000000000000000000000000};
        pts[91]={ 0.01737703674808072085156140929030, 0.00000000000000000000000000000000};
        pts[92]={ 0.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[93]={ 0.00000000000000000000000000000000, 1.00000000000000000000000000000000};
        pts[94]={ 0.00000000000000000000000000000000, 0.98262296325191922363728735945188};
        pts[95]={ 0.00000000000000000000000000000000, 0.94254102211148815815278112495434};
        pts[96]={ 0.00000000000000000000000000000000, 0.88175984497590764732422030647285};
        pts[97]={ 0.00000000000000000000000000000000, 0.80312660273492286755470104253618};
        pts[98]={ 0.00000000000000000000000000000000, 0.71031902735683627447826893330785};
        pts[99]={ 0.00000000000000000000000000000000, 0.60767697768189710139097314822720};
        pts[100]={ 0.00000000000000000000000000000000, 0.50000000000000000000000000000000};
        pts[101]={ 0.00000000000000000000000000000000, 0.39232302231810289860902685177280};
        pts[102]={ 0.00000000000000000000000000000000, 0.28968097264316372552173106669215};
        pts[103]={ 0.00000000000000000000000000000000, 0.19687339726507713244529895746382};
        pts[104]={ 0.00000000000000000000000000000000, 0.11824015502409240818693092478497};
        pts[105]={ 0.00000000000000000000000000000000, 0.05745897788851184184721887504566};
        pts[106]={ 0.00000000000000000000000000000000, 0.01737703674808072085156140929030};
        pts[107]={ 0.01737703674808072085156140929030, 0.98262296325191922363728735945188};
        pts[108]={ 0.05745897788851184184721887504566, 0.94254102211148815815278112495434};
        pts[109]={ 0.11824015502409240818693092478497, 0.88175984497590764732422030647285};
        pts[110]={ 0.19687339726507713244529895746382, 0.80312660273492286755470104253618};
        pts[111]={ 0.28968097264316372552173106669215, 0.71031902735683627447826893330785};
        pts[112]={ 0.39232302231810289860902685177280, 0.60767697768189710139097314822720};
        pts[113]={ 0.50000000000000000000000000000000, 0.50000000000000000000000000000000};
        pts[114]={ 0.60767697768189710139097314822720, 0.39232302231810289860902685177280};
        pts[115]={ 0.71031902735683627447826893330785, 0.28968097264316372552173106669215};
        pts[116]={ 0.80312660273492286755470104253618, 0.19687339726507713244529895746382};
        pts[117]={ 0.88175984497590764732422030647285, 0.11824015502409240818693092478497};
        pts[118]={ 0.94254102211148815815278112495434, 0.05745897788851184184721887504566};
        pts[119]={ 0.98262296325191922363728735945188, 0.01737703674808072085156140929030};
        break;
    }
    case 15: {
        pts[ 0]={ 0.29772517071393372889431816474826, 0.24359923713976261439029258326627};
        pts[ 1]={ 0.45871520129517395769624954482424, 0.29772517071393372889431816474826};
        pts[ 2]={ 0.24355962799089223014270544354076, 0.45867559214630360120423802072764};
        pts[ 3]={ 0.85636571232988145663256318584899, 0.01645981251637753176031431223691};
        pts[ 4]={ 0.12717447515374102895435726168216, 0.85633633646903384306625639510457};
        pts[ 5]={ 0.01644957922835469169009492418354, 0.12721408430261141320194440140767};
        pts[ 6]={ 0.12828799467131124512064843656844, 0.02527544170027049008631081505882};
        pts[ 7]={ 0.84640442703335050378399273540708, 0.12828799467131124512064843656844};
        pts[ 8]={ 0.02531505084914088474223881064518, 0.84639695447954788054545360864722};
        pts[ 9]={ 0.19134047337696047375033003845601, 0.13556461966954921893879770777858};
        pts[10]={ 0.67309490695349027955529663813650, 0.19134047337696047375033003845601};
        pts[11]={ 0.13556461966954921893879770777858, 0.67309490695349027955529663813650};
        pts[12]={ 0.27770339196950660776153085862461, 0.06412210488233743843622391977988};
        pts[13]={ 0.65814853357909464115493847202742, 0.27770973854317232420996219843801};
        pts[14]={ 0.06413453581278555359901361043740, 0.65814086321873987994734989115386};
        pts[15]={ 0.65090975175497389759726729607792, 0.11903384015880381918695007925635};
        pts[16]={ 0.23004848600541871750380096273148, 0.65090975175497389759726729607792};
        pts[17]={ 0.11900879755791970993072936835233, 0.23004936071366074989974492837064};
        pts[18]={ 0.83491830879959127997835821588524, 0.08256065017463956601240226973459};
        pts[19]={ 0.08252104102576914013145170656571, 0.83491830879959127997835821588524};
        pts[20]={ 0.08256065017463956601240226973459, 0.08252104102576914013145170656571};
        pts[21]={ 0.34896458732606477903814834462537, 0.02337648680789271987690014498185};
        pts[22]={ 0.62765892586604254965720883774338, 0.34896458732606477903814834462537};
        pts[23]={ 0.02337648680789271987690014498185, 0.62765892586604254965720883774338};
        pts[24]={ 0.55018093422721159324595419093384, 0.16913233225993690167499039489485};
        pts[25]={ 0.28068673351285150507905541417131, 0.55018093422721159324595419093384};
        pts[26]={ 0.16913233225993690167499039489485, 0.28068673351285150507905541417131};
        pts[27]={ 0.37638155555094326976117713456915, 0.16949660193514853867569058820663};
        pts[28]={ 0.45408223336503766853766705935413, 0.37638155555094326976117713456915};
        pts[29]={ 0.16953621108401895067885334356106, 0.45408223336503766853766705935413};
        pts[30]={ 0.76401087449961246900187461505993, 0.04309557978443596948370242216697};
        pts[31]={ 0.19289091240223524281738320951263, 0.76401350781332866279882409799029};
        pts[32]={ 0.04310180795345228133008674831217, 0.19289028361887017304354685620638};
        pts[33]={ 0.20299972210146460804836010538565, 0.20296011295259422380077296566014};
        pts[34]={ 0.59407977409481160790960529993754, 0.20296011295259422380077296566014};
        pts[35]={ 0.20296011295259422380077296566014, 0.59404016494594114039529131332529};
        pts[36]={ 0.41978308765081184805723069075611, 0.23895792564099965771262645830575};
        pts[37]={ 0.34129859585705890623330560629256, 0.41978308765081184805723069075611};
        pts[38]={ 0.23891831649212927346503931858024, 0.34129859585705890623330560629256};
        pts[39]={ 0.48128348054748987250306413443468, 0.14707033065520427883576815020206};
        pts[40]={ 0.37164618879730593192789456225000, 0.48127555846668629291329466468596};
        pts[41]={ 0.14707825273600785842553761995077, 0.37164618879730593192789456225000};
        pts[42]={ 0.82586314137512784050443315209122, 0.05716760478909379411538438375828};
        pts[43]={ 0.11695839536784852219497565783968, 0.82587399984305776001747290138155};
        pts[44]={ 0.05719373015582445568538716429430, 0.11696557889075041847259939231662};
        pts[45]={ 0.36488976025302932715277393072029, 0.07014518674370906481385645747650};
        pts[46]={ 0.56497794639780285486097000102745, 0.36491046607516952615668515136349};
        pts[47]={ 0.07015119667589796159656856389120, 0.56496505300326160803336961180321};
        pts[48]={ 0.56880178379178703007568174143671, 0.08205497451493805327604746935322};
        pts[49]={ 0.34913531961247146195859158979147, 0.56878995026022516334762713086093};
        pts[50]={ 0.08207473012730347183829593404880, 0.34913531961247146195859158979147};
        pts[51]={ 0.67387057915150183617924994905479, 0.06057375456608002362024478770763};
        pts[52]={ 0.26556360723103583643123215551896, 0.67387228649348307474298280794756};
        pts[53]={ 0.06055874017082331361994107510327, 0.26559396048081163055343267842545};
        pts[54]={ 0.59490121684573937788087505396106, 0.02684138135991032819704926737359};
        pts[55]={ 0.37825191616234721259104389901040, 0.59487501540967568924855868317536};
        pts[56]={ 0.02687306842797714326320779321122, 0.37825191616234721259104389901040};
        pts[57]={ 0.89934739237709815373733590604388, 0.03391082873899081001622946018870};
        pts[58]={ 0.06674177888391095991860169078791, 0.89934739237709815373733590604388};
        pts[59]={ 0.03387121959012041882974841655596, 0.06678138803278137192176444614233};
        pts[60]={ 0.22917969404895796592391832291469, 0.01729944557888743808526932355107};
        pts[61]={ 0.75350847156187883424394158282666, 0.22918732441570413560505414807267};
        pts[62]={ 0.01733905472775783621064427109104, 0.75348125122328424296824778139126};
        pts[63]={ 0.46683639986632474716543583781458, 0.04846177771628102509682278764558};
        pts[64]={ 0.48470974449819770324410228568013, 0.46682998940459563552707322742208};
        pts[65]={ 0.04846026609720658490099154391828, 0.48470974449819770324410228568013};
        pts[66]={ 0.18071771312625431460396896454768, 0.06786320659382230613410058595036};
        pts[67]={ 0.75141908027992332375077921824413, 0.18070812877401393192755563177343};
        pts[68]={ 0.06787415191534210134882698639558, 0.75141908027992332375077921824413};
        pts[69]={ 0.95069312831508090422261147978134, 0.01380219222734517699191503226075};
        pts[70]={ 0.03550216724029300885412396837637, 0.95065769434707014795549184782431};
        pts[71]={ 0.01384013841263680676119118828638, 0.03550216724029300885412396837637};
        pts[72]={ 0.48280938968506409425529568579805, 0.01462826973931436423670415791776};
        pts[73]={ 0.50253686363455463137484002800193, 0.48283486662613106510377747326856};
        pts[74]={ 0.01462826973931436423670415791776, 0.50253686363455463137484002800193};
        pts[75]={ 0.27814191728006981696452726282587, 0.14827070868291958305462685530074};
        pts[76]={ 0.57358737403701065549199711313122, 0.27814191728006981696452726282587};
        pts[77]={ 0.14827070868291958305462685530074, 0.57358737403701065549199711313122};
        pts[78]={ 0.43937672226149543375228745389904, 0.09991706670852805449722922048750};
        pts[79]={ 0.46070621102997655338384674905683, 0.43938048777478033724008810168016};
        pts[80]={ 0.09990083766729404568529560037859, 0.46072049127991798744830020950758};
        pts[81]={ 0.72828593218189452418442897396744, 0.01454048401272346746626595859198};
        pts[82]={ 0.25715524121137256496893996882136, 0.72830427477590398144258188040112};
        pts[83]={ 0.01455352393053594740879308488957, 0.25718181044243049182540516994777};
        pts[84]={ 0.75524422666823320948736864011153, 0.10925939213194978127940970580312};
        pts[85]={ 0.13545677205094669437457355343213, 0.75524422666823320948736864011153};
        pts[86]={ 0.10925939213194978127940970580312, 0.13545677205094669437457355343213};
        pts[87]={ 0.06084181457071281134973972370972, 0.02165749099660987594950789514314};
        pts[88]={ 0.91747955314931306514125708417851, 0.06084181457071281134973972370972};
        pts[89]={ 0.02163902313110372191418129261820, 0.91747955314931306514125708417851};
        pts[90]={ 1.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[91]={ 0.98478402313510904342308549530571, 0.00000000000000000000000000000000};
        pts[92]={ 0.94960026654673601598233290133066, 0.00000000000000000000000000000000};
        pts[93]={ 0.89600414593090760284610496455571, 0.00000000000000000000000000000000};
        pts[94]={ 0.82619435144124653280783832087764, 0.00000000000000000000000000000000};
        pts[95]={ 0.74302971094356884407261532032862, 0.00000000000000000000000000000000};
        pts[96]={ 0.64991523445038157369424425269244, 0.00000000000000000000000000000000};
        pts[97]={ 0.55066313676097466967007676430512, 0.00000000000000000000000000000000};
        pts[98]={ 0.44933686323902527481877200443705, 0.00000000000000000000000000000000};
        pts[99]={ 0.35008476554961842630575574730756, 0.00000000000000000000000000000000};
        pts[100]={ 0.25697028905643115592738467967138, 0.00000000000000000000000000000000};
        pts[101]={ 0.17380564855875346719216167912236, 0.00000000000000000000000000000000};
        pts[102]={ 0.10399585406909245266504626670212, 0.00000000000000000000000000000000};
        pts[103]={ 0.05039973345326398401766709866934, 0.00000000000000000000000000000000};
        pts[104]={ 0.01521597686489101208806573595211, 0.00000000000000000000000000000000};
        pts[105]={ 0.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[106]={ 0.00000000000000000000000000000000, 1.00000000000000000000000000000000};
        pts[107]={ 0.00000000000000000000000000000000, 0.98478402313510904342308549530571};
        pts[108]={ 0.00000000000000000000000000000000, 0.94960026654673601598233290133066};
        pts[109]={ 0.00000000000000000000000000000000, 0.89600414593090760284610496455571};
        pts[110]={ 0.00000000000000000000000000000000, 0.82619435144124653280783832087764};
        pts[111]={ 0.00000000000000000000000000000000, 0.74302971094356884407261532032862};
        pts[112]={ 0.00000000000000000000000000000000, 0.64991523445038157369424425269244};
        pts[113]={ 0.00000000000000000000000000000000, 0.55066313676097466967007676430512};
        pts[114]={ 0.00000000000000000000000000000000, 0.44933686323902527481877200443705};
        pts[115]={ 0.00000000000000000000000000000000, 0.35008476554961842630575574730756};
        pts[116]={ 0.00000000000000000000000000000000, 0.25697028905643115592738467967138};
        pts[117]={ 0.00000000000000000000000000000000, 0.17380564855875346719216167912236};
        pts[118]={ 0.00000000000000000000000000000000, 0.10399585406909245266504626670212};
        pts[119]={ 0.00000000000000000000000000000000, 0.05039973345326398401766709866934};
        pts[120]={ 0.00000000000000000000000000000000, 0.01521597686489101208806573595211};
        pts[121]={ 0.01521597686489101208806573595211, 0.98478402313510904342308549530571};
        pts[122]={ 0.05039973345326398401766709866934, 0.94960026654673601598233290133066};
        pts[123]={ 0.10399585406909245266504626670212, 0.89600414593090760284610496455571};
        pts[124]={ 0.17380564855875346719216167912236, 0.82619435144124653280783832087764};
        pts[125]={ 0.25697028905643115592738467967138, 0.74302971094356884407261532032862};
        pts[126]={ 0.35008476554961842630575574730756, 0.64991523445038157369424425269244};
        pts[127]={ 0.44933686323902527481877200443705, 0.55066313676097466967007676430512};
        pts[128]={ 0.55066313676097466967007676430512, 0.44933686323902527481877200443705};
        pts[129]={ 0.64991523445038157369424425269244, 0.35008476554961842630575574730756};
        pts[130]={ 0.74302971094356884407261532032862, 0.25697028905643115592738467967138};
        pts[131]={ 0.82619435144124653280783832087764, 0.17380564855875346719216167912236};
        pts[132]={ 0.89600414593090760284610496455571, 0.10399585406909245266504626670212};
        pts[133]={ 0.94960026654673601598233290133066, 0.05039973345326398401766709866934};
        pts[134]={ 0.98478402313510904342308549530571, 0.01521597686489101208806573595211};
        pts[135]={ 0.33333333333333331482961625624739, 0.33333333333333331482961625624739};
        break;
    }
    case 16: {
        pts[ 0]={ 0.25515331892265313840084672847297, 0.15591927414369158300111450898839};
        pts[ 1]={ 0.58892740693365530635361437816755, 0.25515331892265313840084672847297};
        pts[ 2]={ 0.15591927414369158300111450898839, 0.58892740693365530635361437816755};
        pts[ 3]={ 0.86403386919974600743898918153718, 0.01822865300036489999424560437546};
        pts[ 4]={ 0.11773747779988909256676521408735, 0.86403386919974600743898918153718};
        pts[ 5]={ 0.01822865300036489999424560437546, 0.11773747779988909256676521408735};
        pts[ 6]={ 0.31886869347913843197872552082117, 0.28363437830191418065339803433744};
        pts[ 7]={ 0.39749692821894733185672521358356, 0.31886869347913843197872552082117};
        pts[ 8]={ 0.28363437830191418065339803433744, 0.39749692821894733185672521358356};
        pts[ 9]={ 0.18393603889431450748581653442670, 0.13090750663905176276458064421604};
        pts[10]={ 0.68515645446663375750517843698617, 0.18393603889431450748581653442670};
        pts[11]={ 0.13090750663905176276458064421604, 0.68515645446663375750517843698617};
        pts[12]={ 0.10634675919178909753171069496602, 0.10406283232907509450626548641594};
        pts[13]={ 0.78959040847913586347317504987586, 0.10634675919178909753171069496602};
        pts[14]={ 0.10406283232907509450626548641594, 0.78959040847913586347317504987586};
        pts[15]={ 0.06491585857590222163882032191395, 0.02945529355838272245193820708664};
        pts[16]={ 0.90562884786571506978702927881386, 0.06491585857590222163882032191395};
        pts[17]={ 0.02945529355838272245193820708664, 0.90562884786571506978702927881386};
        pts[18]={ 0.10645992700380052520259255288693, 0.01573062568689507362273971580180};
        pts[19]={ 0.87780944730930443586913725084742, 0.10645992700380052520259255288693};
        pts[20]={ 0.01573062568689507362273971580180, 0.87780944730930443586913725084742};
        pts[21]={ 0.61518016423017496396852266116184, 0.15642373011992249942636590276379};
        pts[22]={ 0.22839610564990253660511143607437, 0.61518016423017496396852266116184};
        pts[23]={ 0.15642373011992249942636590276379, 0.22839610564990253660511143607437};
        pts[24]={ 0.34092600755102270282392851186160, 0.04709933202215083741792867044751};
        pts[25]={ 0.61197466042682635567473425908247, 0.34092600755102270282392851186160};
        pts[26]={ 0.04709933202215083741792867044751, 0.61197466042682635567473425908247};
        pts[27]={ 0.20255140048153236209671490541950, 0.02128452121817701533923461454378};
        pts[28]={ 0.77616407830029066072796695152647, 0.20255140048153236209671490541950};
        pts[29]={ 0.02128452121817701533923461454378, 0.77616407830029066072796695152647};
        pts[30]={ 0.40344188383906653960764288058272, 0.23836966169360085388007064466365};
        pts[31]={ 0.35818845446733260651228647475364, 0.40344188383906653960764288058272};
        pts[32]={ 0.23836966169360085388007064466365, 0.35818845446733260651228647475364};
        pts[33]={ 0.43627935051797706034193424784462, 0.01507970327356023855502886732438};
        pts[34]={ 0.54864094620846270977665426471503, 0.43627935051797706034193424784462};
        pts[35]={ 0.01507970327356023855502886732438, 0.54864094620846270977665426471503};
        pts[36]={ 0.85129665947546140358781485701911, 0.05584402149801173664833697785070};
        pts[37]={ 0.09285931902652685976384816513018, 0.85129665947546140358781485701911};
        pts[38]={ 0.05584402149801173664833697785070, 0.09285931902652685976384816513018};
        pts[39]={ 0.25097510400894773585633856782806, 0.23778457269911967841835576109588};
        pts[40]={ 0.51124032329193258572530567107606, 0.25097510400894773585633856782806};
        pts[41]={ 0.23778457269911967841835576109588, 0.51124032329193258572530567107606};
        pts[42]={ 0.61956525237229098657110171188833, 0.09467197351515263537002908833529};
        pts[43]={ 0.28576277411255635030329358414747, 0.61956525237229098657110171188833};
        pts[44]={ 0.09467197351515263537002908833529, 0.28576277411255635030329358414747};
        pts[45]={ 0.34481644015751750087517280007887, 0.17674087222546266962019956281438};
        pts[46]={ 0.47844268761701980174905202147784, 0.34481644015751750087517280007887};
        pts[47]={ 0.17674087222546266962019956281438, 0.47844268761701980174905202147784};
        pts[48]={ 0.36927348756674033269931101131078, 0.10953736833793063787378940787676};
        pts[49]={ 0.52118914409532912657141423551366, 0.36927348756674033269931101131078};
        pts[50]={ 0.10953736833793063787378940787676, 0.52118914409532912657141423551366};
        pts[51]={ 0.51265200691670032018265601436724, 0.08866860377641500035039712201979};
        pts[52]={ 0.39867938930688467946694686361297, 0.51265200691670032018265601436724};
        pts[53]={ 0.08866860377641500035039712201979, 0.39867938930688467946694686361297};
        pts[54]={ 0.18214435570638423667055860732944, 0.06088700366133469044083170729209};
        pts[55]={ 0.75696864063228108676639749319293, 0.18214435570638423667055860732944};
        pts[56]={ 0.06088700366133469044083170729209, 0.75696864063228108676639749319293};
        pts[57]={ 0.71114693883535240459536908019800, 0.10630610443578662749342100823924};
        pts[58]={ 0.18254695672886095403342210374831, 0.71114693883535240459536908019800};
        pts[59]={ 0.10630610443578662749342100823924, 0.18254695672886095403342210374831};
        pts[60]={ 0.44666959090079005889606378332246, 0.05494368174550146727286303871551};
        pts[61]={ 0.49838672735370848076996708186925, 0.44666959090079005889606378332246};
        pts[62]={ 0.05494368174550146727286303871551, 0.49838672735370848076996708186925};
        pts[63]={ 0.92591337936627438054415506485384, 0.02157747846196626448356248317850};
        pts[64]={ 0.05250914217175935150283550001404, 0.92591337936627438054415506485384};
        pts[65]={ 0.02157747846196626448356248317850, 0.05250914217175935150283550001404};
        pts[66]={ 0.25990799867759534214073369184916, 0.10137629001168226017259854643271};
        pts[67]={ 0.63871571131072235605330433827476, 0.25990799867759534214073369184916};
        pts[68]={ 0.10137629001168226017259854643271, 0.63871571131072235605330433827476};
        pts[69]={ 0.50056284757716251743175916999462, 0.16905310354945313200403234077385};
        pts[70]={ 0.33038404887338435056420848923153, 0.50056284757716251743175916999462};
        pts[71]={ 0.16905310354945313200403234077385, 0.33038404887338435056420848923153};
        pts[72]={ 0.02749298190695764754210017599689, 0.01570514706965205758981696249066};
        pts[73]={ 0.95680187102339031568476457323413, 0.02749298190695764754210017599689};
        pts[74]={ 0.01570514706965205758981696249066, 0.95680187102339031568476457323413};
        pts[75]={ 0.78791709294041245570383580343332, 0.04898156610340578809292821915733};
        pts[76]={ 0.16310134095618175620323597740935, 0.78791709294041245570383580343332};
        pts[77]={ 0.04898156610340578809292821915733, 0.16310134095618175620323597740935};
        pts[78]={ 0.31066022793966152626410348602803, 0.01655687042736561642719550491165};
        pts[79]={ 0.67278290163297282955312539343140, 0.31066022793966152626410348602803};
        pts[80]={ 0.01655687042736561642719550491165, 0.67278290163297282955312539343140};
        pts[81]={ 0.71011117829207237583943879144499, 0.05279167578105559144319158804137};
        pts[82]={ 0.23709714592687203271736962051364, 0.71011117829207237583943879144499};
        pts[83]={ 0.05279167578105559144319158804137, 0.23709714592687203271736962051364};
        pts[84]={ 0.42220960028294618471278454308049, 0.13486798115263390340778926201892};
        pts[85]={ 0.44292241856441993963500181052950, 0.42220960028294618471278454308049};
        pts[86]={ 0.13486798115263390340778926201892, 0.44292241856441993963500181052950};
        pts[87]={ 0.11340354354198708963785691139492, 0.04194793662248177262830139966354};
        pts[88]={ 0.84464851983553113079494778503431, 0.11340354354198708963785691139492};
        pts[89]={ 0.04194793662248177262830139966354, 0.84464851983553113079494778503431};
        pts[90]={ 0.77021335570576110463036911824020, 0.01462667740963627220629472702740};
        pts[91]={ 0.21515996688460262142861267875560, 0.77021335570576110463036911824020};
        pts[92]={ 0.01462667740963627220629472702740, 0.21515996688460262142861267875560};
        pts[93]={ 0.65346977505869741076338641505572, 0.02150936631898192619161136462935};
        pts[94]={ 0.32502085862232066304500222031493, 0.65346977505869741076338641505572};
        pts[95]={ 0.02150936631898192619161136462935, 0.32502085862232066304500222031493};
        pts[96]={ 0.55134971587723180519446941616479, 0.02476102169289638721405566457179};
        pts[97]={ 0.42388926242987179371368711144896, 0.55134971587723180519446941616479};
        pts[98]={ 0.02476102169289638721405566457179, 0.42388926242987179371368711144896};
        pts[99]={ 0.25618667888885587879954641721270, 0.05082919462592933035338660374691};
        pts[100]={ 0.69298412648521479084706697904039, 0.25618667888885587879954641721270};
        pts[101]={ 0.05082919462592933035338660374691, 0.69298412648521479084706697904039};
        pts[102]={ 0.57864532101011234566101393284043, 0.05229611289321199618651547780246};
        pts[103]={ 0.36905856609667564427468278154265, 0.57864532101011234566101393284043};
        pts[104]={ 0.05229611289321199618651547780246, 0.36905856609667564427468278154265};
        pts[105]={ 1.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[106]={ 0.98656608831570924422749158111401, 0.00000000000000000000000000000000};
        pts[107]={ 0.95543999795778677963653535698541, 0.00000000000000000000000000000000};
        pts[108]={ 0.90784812561088512605778078068397, 0.00000000000000000000000000000000};
        pts[109]={ 0.84551449031384229293450971454149, 0.00000000000000000000000000000000};
        pts[110]={ 0.77069269966505071778328783693723, 0.00000000000000000000000000000000};
        pts[111]={ 0.68608721678273854038820900314022, 0.00000000000000000000000000000000};
        pts[112]={ 0.59475598675915875013231470802566, 0.00000000000000000000000000000000};
        pts[113]={ 0.50000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[114]={ 0.40524401324084130537883652323217, 0.00000000000000000000000000000000};
        pts[115]={ 0.31391278321726145961179099685978, 0.00000000000000000000000000000000};
        pts[116]={ 0.22930730033494922670556093180494, 0.00000000000000000000000000000000};
        pts[117]={ 0.15448550968615765155433905420068, 0.00000000000000000000000000000000};
        pts[118]={ 0.09215187438911487394221921931603, 0.00000000000000000000000000000000};
        pts[119]={ 0.04456000204221322036346464301459, 0.00000000000000000000000000000000};
        pts[120]={ 0.01343391168429081128365965014382, 0.00000000000000000000000000000000};
        pts[121]={ 0.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[122]={ 0.00000000000000000000000000000000, 1.00000000000000000000000000000000};
        pts[123]={ 0.00000000000000000000000000000000, 0.98656608831570924422749158111401};
        pts[124]={ 0.00000000000000000000000000000000, 0.95543999795778677963653535698541};
        pts[125]={ 0.00000000000000000000000000000000, 0.90784812561088512605778078068397};
        pts[126]={ 0.00000000000000000000000000000000, 0.84551449031384229293450971454149};
        pts[127]={ 0.00000000000000000000000000000000, 0.77069269966505071778328783693723};
        pts[128]={ 0.00000000000000000000000000000000, 0.68608721678273854038820900314022};
        pts[129]={ 0.00000000000000000000000000000000, 0.59475598675915875013231470802566};
        pts[130]={ 0.00000000000000000000000000000000, 0.50000000000000000000000000000000};
        pts[131]={ 0.00000000000000000000000000000000, 0.40524401324084130537883652323217};
        pts[132]={ 0.00000000000000000000000000000000, 0.31391278321726145961179099685978};
        pts[133]={ 0.00000000000000000000000000000000, 0.22930730033494922670556093180494};
        pts[134]={ 0.00000000000000000000000000000000, 0.15448550968615765155433905420068};
        pts[135]={ 0.00000000000000000000000000000000, 0.09215187438911487394221921931603};
        pts[136]={ 0.00000000000000000000000000000000, 0.04456000204221322036346464301459};
        pts[137]={ 0.00000000000000000000000000000000, 0.01343391168429081128365965014382};
        pts[138]={ 0.01343391168429081128365965014382, 0.98656608831570924422749158111401};
        pts[139]={ 0.04456000204221322036346464301459, 0.95543999795778677963653535698541};
        pts[140]={ 0.09215187438911487394221921931603, 0.90784812561088512605778078068397};
        pts[141]={ 0.15448550968615765155433905420068, 0.84551449031384229293450971454149};
        pts[142]={ 0.22930730033494922670556093180494, 0.77069269966505071778328783693723};
        pts[143]={ 0.31391278321726145961179099685978, 0.68608721678273854038820900314022};
        pts[144]={ 0.40524401324084130537883652323217, 0.59475598675915875013231470802566};
        pts[145]={ 0.50000000000000000000000000000000, 0.50000000000000000000000000000000};
        pts[146]={ 0.59475598675915875013231470802566, 0.40524401324084130537883652323217};
        pts[147]={ 0.68608721678273854038820900314022, 0.31391278321726145961179099685978};
        pts[148]={ 0.77069269966505071778328783693723, 0.22930730033494922670556093180494};
        pts[149]={ 0.84551449031384229293450971454149, 0.15448550968615765155433905420068};
        pts[150]={ 0.90784812561088512605778078068397, 0.09215187438911487394221921931603};
        pts[151]={ 0.95543999795778677963653535698541, 0.04456000204221322036346464301459};
        pts[152]={ 0.98656608831570924422749158111401, 0.01343391168429081128365965014382};
        break;
    }
    case 17: {
        pts[ 0]={ 0.71683136220809728911262936890125, 0.01418369789941874639649377343176};
        pts[ 1]={ 0.26898493989248395408253600180615, 0.71683136220809728911262936890125};
        pts[ 2]={ 0.01418369789941874639649377343176, 0.26898493989248395408253600180615};
        pts[ 3]={ 0.78771739787696515300297050998779, 0.05649356465138096611289242332532};
        pts[ 4]={ 0.15578903747165387394524316277966, 0.78771739787696515300297050998779};
        pts[ 5]={ 0.05649356465138096611289242332532, 0.15578903747165387394524316277966};
        pts[ 6]={ 0.34619714737722045994061659257568, 0.27228587894146638603842802694999};
        pts[ 7]={ 0.38151697368131309850980414921651, 0.34619714737722045994061659257568};
        pts[ 8]={ 0.27228587894146638603842802694999, 0.38151697368131309850980414921651};
        pts[ 9]={ 0.10512577105666533194838763165535, 0.09752415461677049757227564441564};
        pts[10]={ 0.79735007432656423986827576300129, 0.10512577105666533194838763165535};
        pts[11]={ 0.09752415461677049757227564441564, 0.79735007432656423986827576300129};
        pts[12]={ 0.56904316653812692106129134117509, 0.12027038844952203655580547092541};
        pts[13]={ 0.31068644501235104238290318789950, 0.56904316653812692106129134117509};
        pts[14]={ 0.12027038844952203655580547092541, 0.31068644501235104238290318789950};
        pts[15]={ 0.68299086001599851325494228149182, 0.08326834652743357600979123844809};
        pts[16]={ 0.23374079345656789685747867224563, 0.68299086001599851325494228149182};
        pts[17]={ 0.08326834652743357600979123844809, 0.23374079345656789685747867224563};
        pts[18]={ 0.30283136648192138240531789961096, 0.15837797929540872732268042000214};
        pts[19]={ 0.53879065422266991802757729601581, 0.30283136648192138240531789961096};
        pts[20]={ 0.15837797929540872732268042000214, 0.53879065422266991802757729601581};
        pts[21]={ 0.26989551913114212533528757376189, 0.22532151487422846791730535187526};
        pts[22]={ 0.50478296599462946225855830562068, 0.26989551913114212533528757376189};
        pts[23]={ 0.22532151487422846791730535187526, 0.50478296599462946225855830562068};
        pts[24]={ 0.72412697107428003473472699624836, 0.12195866201846243537243452692564};
        pts[25]={ 0.15391436690725751601505066901154, 0.72412697107428003473472699624836};
        pts[26]={ 0.12195866201846243537243452692564, 0.15391436690725751601505066901154};
        pts[27]={ 0.04527657923950216545261326928085, 0.01815423970669129138943986845334};
        pts[28]={ 0.93656918105380659866909809352364, 0.04527657923950216545261326928085};
        pts[29]={ 0.01815423970669129138943986845334, 0.93656918105380659866909809352364};
        pts[30]={ 0.14379881215919843029382718668785, 0.04439191482703121888198083411226};
        pts[31]={ 0.81180927301377037164087369092158, 0.14379881215919843029382718668785};
        pts[32]={ 0.04439191482703121888198083411226, 0.81180927301377037164087369092158};
        pts[33]={ 0.49590878706952951082698177742714, 0.01998222995581780403462346384913};
        pts[34]={ 0.48410898297465265738281914309482, 0.49590878706952951082698177742714};
        pts[35]={ 0.01998222995581780403462346384913, 0.48410898297465265738281914309482};
        pts[36]={ 0.45023871976102836089950187670183, 0.05526210991762155883000318112863};
        pts[37]={ 0.49449917032135010108717665389122, 0.45023871976102836089950187670183};
        pts[38]={ 0.05526210991762155883000318112863, 0.49449917032135010108717665389122};
        pts[39]={ 0.54928665819820066129608449045918, 0.19735304306319778611111814825563};
        pts[40]={ 0.25336029873860155259279736128519, 0.54928665819820066129608449045918};
        pts[41]={ 0.19735304306319778611111814825563, 0.25336029873860155259279736128519};
        pts[42]={ 0.56841606731765359494801259643282, 0.03890130231099912100356519317756};
        pts[43]={ 0.39268263037134726323174049866793, 0.56841606731765359494801259643282};
        pts[44]={ 0.03890130231099912100356519317756, 0.39268263037134726323174049866793};
        pts[45]={ 0.46071019599056906024259205878479, 0.23450565007787188553933788170980};
        pts[46]={ 0.30478415393155905421807005950541, 0.46071019599056906024259205878479};
        pts[47]={ 0.23450565007787188553933788170980, 0.30478415393155905421807005950541};
        pts[48]={ 0.96187740751332395650052831115318, 0.01213999608421929658763627202234};
        pts[49]={ 0.02598259640245674517711194084768, 0.96187740751332395650052831115318};
        pts[50]={ 0.01213999608421929658763627202234, 0.02598259640245674517711194084768};
        pts[51]={ 0.80517274273296390862952875977498, 0.02069355738854714368835097104693};
        pts[52]={ 0.17413369987848895115156722113170, 0.80517274273296390862952875977498};
        pts[53]={ 0.02069355738854714368835097104693, 0.17413369987848895115156722113170};
        pts[54]={ 0.37791655655563644033989589843259, 0.19774955376374930837712895481673};
        pts[55]={ 0.42433388968061430679412637800851, 0.37791655655563644033989589843259};
        pts[56]={ 0.19774955376374930837712895481673, 0.42433388968061430679412637800851};
        pts[57]={ 0.19681701329921219567609114164952, 0.01474522459527022322234302009747};
        pts[58]={ 0.78843776210551763661271706951084, 0.19681701329921219567609114164952};
        pts[59]={ 0.01474522459527022322234302009747, 0.78843776210551763661271706951084};
        pts[60]={ 0.08580393150955704928417588917000, 0.05299038476452450074916100675182};
        pts[61]={ 0.86120568372591854711117775877938, 0.08580393150955704928417588917000};
        pts[62]={ 0.05299038476452450074916100675182, 0.86120568372591854711117775877938};
        pts[63]={ 0.22558333734735291908890530976350, 0.04560756681477428492454961883595};
        pts[64]={ 0.72880909583787278904765116749331, 0.22558333734735291908890530976350};
        pts[65]={ 0.04560756681477428492454961883595, 0.72880909583787278904765116749331};
        pts[66]={ 0.71710756467630842880822683582664, 0.04145516033527813642978188113375};
        pts[67]={ 0.24143727498841344170088518694683, 0.71710756467630842880822683582664};
        pts[68]={ 0.04145516033527813642978188113375, 0.24143727498841344170088518694683};
        pts[69]={ 0.91743773315339094409637255012058, 0.02726402909714513433936389219525};
        pts[70]={ 0.05529823774946392156426355768417, 0.91743773315339094409637255012058};
        pts[71]={ 0.02726402909714513433936389219525, 0.05529823774946392156426355768417};
        pts[72]={ 0.63380449344161871039204925182275, 0.14600349097315251656326040574641};
        pts[73]={ 0.22019201558522877304469034243084, 0.63380449344161871039204925182275};
        pts[74]={ 0.14600349097315251656326040574641, 0.22019201558522877304469034243084};
        pts[75]={ 0.25911472386313016968983902188484, 0.09023850992161454309670176598956};
        pts[76]={ 0.65064676621525530109124701994006, 0.25911472386313016968983902188484};
        pts[77]={ 0.09023850992161454309670176598956, 0.65064676621525530109124701994006};
        pts[78]={ 0.63725020045307201321804768667789, 0.05277981116659663518575484886242};
        pts[79]={ 0.30996998838033135159619746445969, 0.63725020045307201321804768667789};
        pts[80]={ 0.05277981116659663518575484886242, 0.30996998838033135159619746445969};
        pts[81]={ 0.21522374480863756818216359079088, 0.16240986781221589119184045557631};
        pts[82]={ 0.62236638737914651287042033800390, 0.21522374480863756818216359079088};
        pts[83]={ 0.16240986781221589119184045557631, 0.62236638737914651287042033800390};
        pts[84]={ 0.35887613440457094560187556453457, 0.08977952266066784037246151228828};
        pts[85]={ 0.55134434293476120014787511536269, 0.35887613440457094560187556453457};
        pts[86]={ 0.08977952266066784037246151228828, 0.55134434293476120014787511536269};
        pts[87]={ 0.47950530924884221528614602902962, 0.14112411684572320758057628609095};
        pts[88]={ 0.37937057390543460488885330050834, 0.47950530924884221528614602902962};
        pts[89]={ 0.14112411684572320758057628609095, 0.37937057390543460488885330050834};
        pts[90]={ 0.39143034732965098498880252009258, 0.02183581874360724511818787618722};
        pts[91]={ 0.58673383392674172132075227636960, 0.39143034732965098498880252009258};
        pts[92]={ 0.02183581874360724511818787618722, 0.58673383392674172132075227636960};
        pts[93]={ 0.33932150802276972445525871080463, 0.04978087446049472841735905603855};
        pts[94]={ 0.61089761751673554712738223315682, 0.33932150802276972445525871080463};
        pts[95]={ 0.04978087446049472841735905603855, 0.61089761751673554712738223315682};
        pts[96]={ 0.52731130330054576393905563236331, 0.08108469251758988749489986957997};
        pts[97]={ 0.39160400418186436244383230587118, 0.52731130330054576393905563236331};
        pts[98]={ 0.08108469251758988749489986957997, 0.39160400418186436244383230587118};
        pts[99]={ 0.29950385603663909339644533247338, 0.01672219116759078583189790379038};
        pts[100]={ 0.68377395279577013464944457155070, 0.29950385603663909339644533247338};
        pts[101]={ 0.01672219116759078583189790379038, 0.68377395279577013464944457155070};
        pts[102]={ 0.89028402121870242602597045333823, 0.01470387603715043491059422109402};
        pts[103]={ 0.09501210274414713385926489763733, 0.89028402121870242602597045333823};
        pts[104]={ 0.01470387603715043491059422109402, 0.09501210274414713385926489763733};
        pts[105]={ 0.10993821079547889951744110703657, 0.01408389783956228921701914202913};
        pts[106]={ 0.87597789136495873840715375990840, 0.10993821079547889951744110703657};
        pts[107]={ 0.01408389783956228921701914202913, 0.87597789136495873840715375990840};
        pts[108]={ 0.17383399202414168138730587997998, 0.10098980444934264888434682916341};
        pts[109]={ 0.72517620352651568360613509867107, 0.17383399202414168138730587997998};
        pts[110]={ 0.10098980444934264888434682916341, 0.72517620352651568360613509867107};
        pts[111]={ 0.41272044351923065086396036349470, 0.11598232415962334007630829546542};
        pts[112]={ 0.47129723232114600905973134103988, 0.41272044351923065086396036349470};
        pts[113]={ 0.11598232415962334007630829546542, 0.47129723232114600905973134103988};
        pts[114]={ 0.85677779639565510017007454734994, 0.04280777260382494231460981382043};
        pts[115]={ 0.10041443100051995751531563882963, 0.85677779639565510017007454734994};
        pts[116]={ 0.04280777260382494231460981382043, 0.10041443100051995751531563882963};
        pts[117]={ 0.61316811613748112286259583925130, 0.01494346599519905637032390899321};
        pts[118]={ 0.37188841786731979821567506405700, 0.61316811613748112286259583925130};
        pts[119]={ 0.01494346599519905637032390899321, 0.37188841786731979821567506405700};
        pts[120]={ 1.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[121]={ 0.98805277870609931056833374896087, 0.00000000000000000000000000000000};
        pts[122]={ 0.96032459267376690803530436824076, 0.00000000000000000000000000000000};
        pts[123]={ 0.91779676760904504995153274649056, 0.00000000000000000000000000000000};
        pts[124]={ 0.86183966464162131693171886581695, 0.00000000000000000000000000000000};
        pts[125]={ 0.79425241715933081465550458233338, 0.00000000000000000000000000000000};
        pts[126]={ 0.71720751845606200980398625688395, 0.00000000000000000000000000000000};
        pts[127]={ 0.63318132643914049051403480916633, 0.00000000000000000000000000000000};
        pts[128]={ 0.54487454674232604912020860865596, 0.00000000000000000000000000000000};
        pts[129]={ 0.45512545325767395087979139134404, 0.00000000000000000000000000000000};
        pts[130]={ 0.36681867356085950948596519083367, 0.00000000000000000000000000000000};
        pts[131]={ 0.28279248154393799019601374311605, 0.00000000000000000000000000000000};
        pts[132]={ 0.20574758284066912983334418640879, 0.00000000000000000000000000000000};
        pts[133]={ 0.13816033535837868306828113418305, 0.00000000000000000000000000000000};
        pts[134]={ 0.08220323239095489453731602225162, 0.00000000000000000000000000000000};
        pts[135]={ 0.03967540732623303645354440050141, 0.00000000000000000000000000000000};
        pts[136]={ 0.01194722129390074494281748229696, 0.00000000000000000000000000000000};
        pts[137]={ 0.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[138]={ 0.00000000000000000000000000000000, 1.00000000000000000000000000000000};
        pts[139]={ 0.00000000000000000000000000000000, 0.98805277870609931056833374896087};
        pts[140]={ 0.00000000000000000000000000000000, 0.96032459267376690803530436824076};
        pts[141]={ 0.00000000000000000000000000000000, 0.91779676760904504995153274649056};
        pts[142]={ 0.00000000000000000000000000000000, 0.86183966464162131693171886581695};
        pts[143]={ 0.00000000000000000000000000000000, 0.79425241715933081465550458233338};
        pts[144]={ 0.00000000000000000000000000000000, 0.71720751845606200980398625688395};
        pts[145]={ 0.00000000000000000000000000000000, 0.63318132643914049051403480916633};
        pts[146]={ 0.00000000000000000000000000000000, 0.54487454674232604912020860865596};
        pts[147]={ 0.00000000000000000000000000000000, 0.45512545325767395087979139134404};
        pts[148]={ 0.00000000000000000000000000000000, 0.36681867356085950948596519083367};
        pts[149]={ 0.00000000000000000000000000000000, 0.28279248154393799019601374311605};
        pts[150]={ 0.00000000000000000000000000000000, 0.20574758284066912983334418640879};
        pts[151]={ 0.00000000000000000000000000000000, 0.13816033535837868306828113418305};
        pts[152]={ 0.00000000000000000000000000000000, 0.08220323239095489453731602225162};
        pts[153]={ 0.00000000000000000000000000000000, 0.03967540732623303645354440050141};
        pts[154]={ 0.00000000000000000000000000000000, 0.01194722129390074494281748229696};
        pts[155]={ 0.01194722129390074494281748229696, 0.98805277870609931056833374896087};
        pts[156]={ 0.03967540732623303645354440050141, 0.96032459267376690803530436824076};
        pts[157]={ 0.08220323239095489453731602225162, 0.91779676760904504995153274649056};
        pts[158]={ 0.13816033535837868306828113418305, 0.86183966464162131693171886581695};
        pts[159]={ 0.20574758284066912983334418640879, 0.79425241715933081465550458233338};
        pts[160]={ 0.28279248154393799019601374311605, 0.71720751845606200980398625688395};
        pts[161]={ 0.36681867356085950948596519083367, 0.63318132643914049051403480916633};
        pts[162]={ 0.45512545325767395087979139134404, 0.54487454674232604912020860865596};
        pts[163]={ 0.54487454674232604912020860865596, 0.45512545325767395087979139134404};
        pts[164]={ 0.63318132643914049051403480916633, 0.36681867356085950948596519083367};
        pts[165]={ 0.71720751845606200980398625688395, 0.28279248154393799019601374311605};
        pts[166]={ 0.79425241715933081465550458233338, 0.20574758284066912983334418640879};
        pts[167]={ 0.86183966464162131693171886581695, 0.13816033535837868306828113418305};
        pts[168]={ 0.91779676760904504995153274649056, 0.08220323239095489453731602225162};
        pts[169]={ 0.96032459267376690803530436824076, 0.03967540732623303645354440050141};
        pts[170]={ 0.98805277870609931056833374896087, 0.01194722129390074494281748229696};
        break;
    }
    case 18: {
        pts[ 0]={ 0.71038982376255532802389325297554, 0.01210958148309153326049880661230};
        pts[ 1]={ 0.27750059475435312483782013259770, 0.71038982376255532802389325297554};
        pts[ 2]={ 0.01210958148309153326049880661230, 0.27750059475435312483782013259770};
        pts[ 3]={ 0.14893155105524155668028640775447, 0.03976636762487085335893155502163};
        pts[ 4]={ 0.81130208131988756914410032550222, 0.14893155105524155668028640775447};
        pts[ 5]={ 0.03976636762487085335893155502163, 0.81130208131988756914410032550222};
        pts[ 6]={ 0.76103300646853133670788338349666, 0.03277512259112753262302319967603};
        pts[ 7]={ 0.20619187094034113760798732073454, 0.76103300646853133670788338349666};
        pts[ 8]={ 0.03277512259112753262302319967603, 0.20619187094034113760798732073454};
        pts[ 9]={ 0.41056760590618934791962146846345, 0.03624500038642432253599423574997};
        pts[10]={ 0.55318739370738634342217210360104, 0.41056760590618934791962146846345};
        pts[11]={ 0.03624500038642432253599423574997, 0.55318739370738634342217210360104};
        pts[12]={ 0.67073064284931038780968037826824, 0.03702252030158931922487752785855};
        pts[13]={ 0.29224683684910029990433599778044, 0.67073064284931038780968037826824};
        pts[14]={ 0.03702252030158931922487752785855, 0.29224683684910029990433599778044};
        pts[15]={ 0.82592229343640155558858850781689, 0.03022060356777456213706045673462};
        pts[16]={ 0.14385710299582388227435103544849, 0.82592229343640155558858850781689};
        pts[17]={ 0.03022060356777456213706045673462, 0.14385710299582388227435103544849};
        pts[18]={ 0.39555041178380373478873366366315, 0.11931570562534324353443082600279};
        pts[19]={ 0.48513388259085299392125989470514, 0.39555041178380373478873366366315};
        pts[20]={ 0.11931570562534324353443082600279, 0.48513388259085299392125989470514};
        pts[21]={ 0.81416794542000014978100352891488, 0.01082298827636249405337309781316};
        pts[22]={ 0.17500906630363735616562337327196, 0.81416794542000014978100352891488};
        pts[23]={ 0.01082298827636249405337309781316, 0.17500906630363735616562337327196};
        pts[24]={ 0.26522931351791112497551239357563, 0.13638627417978055333946940663736};
        pts[25]={ 0.59838441230230832168501819978701, 0.26522931351791112497551239357563};
        pts[26]={ 0.13638627417978055333946940663736, 0.59838441230230832168501819978701};
        pts[27]={ 0.05145645798590730429777906351774, 0.01960835570918360859504758764160};
        pts[28]={ 0.92893518630490912180164286837680, 0.05145645798590730429777906351774};
        pts[29]={ 0.01960835570918360859504758764160, 0.92893518630490912180164286837680};
        pts[30]={ 0.60067086215312481822792278762790, 0.01563154850217922040012119566654};
        pts[31]={ 0.38369758934469594402472125693748, 0.60067086215312481822792278762790};
        pts[32]={ 0.01563154850217922040012119566654, 0.38369758934469594402472125693748};
        pts[33]={ 0.26620013756642180169365019537508, 0.21821879478743855251821059937356};
        pts[34]={ 0.51558106764613964578813920525135, 0.26620013756642180169365019537508};
        pts[35]={ 0.21821879478743855251821059937356, 0.51558106764613964578813920525135};
        pts[36]={ 0.50398767986704340327008821986965, 0.19396710855527177219670420527109};
        pts[37]={ 0.30204521157768482453320757485926, 0.50398767986704340327008821986965};
        pts[38]={ 0.19396710855527177219670420527109, 0.30204521157768482453320757485926};
        pts[39]={ 0.73998096895470488743029591205413, 0.08505313526782211364096752959085};
        pts[40]={ 0.17496589577747301280652436616947, 0.73998096895470488743029591205413};
        pts[41]={ 0.08505313526782211364096752959085, 0.17496589577747301280652436616947};
        pts[42]={ 0.54350285675381049266263744357275, 0.12781833651469109991438699580613};
        pts[43]={ 0.32867880673149840742297556062113, 0.54350285675381049266263744357275};
        pts[44]={ 0.12781833651469109991438699580613, 0.32867880673149840742297556062113};
        pts[45]={ 0.08193019295653271305557296955158, 0.05314524152604391954435314460170};
        pts[46]={ 0.86492456551742336046117998193949, 0.08193019295653271305557296955158};
        pts[47]={ 0.05314524152604391954435314460170, 0.86492456551742336046117998193949};
        pts[48]={ 0.14589210907509408965410102609894, 0.07600988365601799023885121187050};
        pts[49]={ 0.77809800726888789235147214640165, 0.14589210907509408965410102609894};
        pts[50]={ 0.07600988365601799023885121187050, 0.77809800726888789235147214640165};
        pts[51]={ 0.19980794074180641128890556501574, 0.09955975924134374821505844010971};
        pts[52]={ 0.70063230001684984049603599487455, 0.19980794074180641128890556501574};
        pts[53]={ 0.09955975924134374821505844010971, 0.70063230001684984049603599487455};
        pts[54]={ 0.37880737677766102011744919764169, 0.01379045070118107944867613667839};
        pts[55]={ 0.60740217252115791257693899751757, 0.37880737677766102011744919764169};
        pts[56]={ 0.01379045070118107944867613667839, 0.60740217252115791257693899751757};
        pts[57]={ 0.42799284807036724886941669865337, 0.18133108613095175187801544325339};
        pts[58]={ 0.39067606579868108251929470497998, 0.42799284807036724886941669865337};
        pts[59]={ 0.18133108613095175187801544325339, 0.39067606579868108251929470497998};
        pts[60]={ 0.52477479692275397216150167878368, 0.03608617710052490490379994980685};
        pts[61]={ 0.43913902597672110905691056359501, 0.52477479692275397216150167878368};
        pts[62]={ 0.03608617710052490490379994980685, 0.43913902597672110905691056359501};
        pts[63]={ 0.42885951199870764316557369966176, 0.24665729200266003906527600975096};
        pts[64]={ 0.32448319599863229001357467495836, 0.42885951199870764316557369966176};
        pts[65]={ 0.24665729200266003906527600975096, 0.32448319599863229001357467495836};
        pts[66]={ 0.20101251899050642801825006245053, 0.17600773169411015883767390732828};
        pts[67]={ 0.62297974931538346865522726147901, 0.20101251899050642801825006245053};
        pts[68]={ 0.17600773169411015883767390732828, 0.62297974931538346865522726147901};
        pts[69]={ 0.27412306196502284061367049616820, 0.07566141210367714753370904645635};
        pts[70]={ 0.65021552593130005348598388081882, 0.27412306196502284061367049616820};
        pts[71]={ 0.07566141210367714753370904645635, 0.65021552593130005348598388081882};
        pts[72]={ 0.02070961798831303171675166652221, 0.01064733954432504209330900835084};
        pts[73]={ 0.96864304246736188108712894972996, 0.02070961798831303171675166652221};
        pts[74]={ 0.01064733954432504209330900835084, 0.96864304246736188108712894972996};
        pts[75]={ 0.49529109813865263012289119615161, 0.10129153976517932256307119587291};
        pts[76]={ 0.40341736209616796404731076108874, 0.49529109813865263012289119615161};
        pts[77]={ 0.10129153976517932256307119587291, 0.40341736209616796404731076108874};
        pts[78]={ 0.33769470319797956214813439146383, 0.16322317028282673501138333449489};
        pts[79]={ 0.49908212651919370284048227404128, 0.33769470319797956214813439146383};
        pts[80]={ 0.16322317028282673501138333449489, 0.49908212651919370284048227404128};
        pts[81]={ 0.65759824611908923586156561214011, 0.12092806349013841826778303811807};
        pts[82]={ 0.22147369039077235974843915755628, 0.65759824611908923586156561214011};
        pts[83]={ 0.12092806349013841826778303811807, 0.22147369039077235974843915755628};
        pts[84]={ 0.26734041196440633259001629085105, 0.01374663793865008899752755411328};
        pts[85]={ 0.71891295009694367035280038180645, 0.26734041196440633259001629085105};
        pts[86]={ 0.01374663793865008899752755411328, 0.71891295009694367035280038180645};
        pts[87]={ 0.49335316525936800946539051437867, 0.01133513496215798714572553507196};
        pts[88]={ 0.49531169977847400165416047457256, 0.49335316525936800946539051437867};
        pts[89]={ 0.01133513496215798714572553507196, 0.49531169977847400165416047457256};
        pts[90]={ 0.93741312121193309803146576086874, 0.02045783267263008228797893650608};
        pts[91]={ 0.04212904611543681621110835067157, 0.93741312121193309803146576086874};
        pts[92]={ 0.02045783267263008228797893650608, 0.04212904611543681621110835067157};
        pts[93]={ 0.09559229280608330658886018227349, 0.01284397605058827315782643552211};
        pts[94]={ 0.89156373114332843066165423806524, 0.09559229280608330658886018227349};
        pts[95]={ 0.01284397605058827315782643552211, 0.89156373114332843066165423806524};
        pts[96]={ 0.89919441294213442450455886501004, 0.01355693350522054439755148536051};
        pts[97]={ 0.08724865355264503630206007755987, 0.89919441294213442450455886501004};
        pts[98]={ 0.01355693350522054439755148536051, 0.08724865355264503630206007755987};
        pts[99]={ 0.17670786259161005982143421988440, 0.01597724877452631059293608473126};
        pts[100]={ 0.80731488863386358101337236803374, 0.17670786259161005982143421988440};
        pts[101]={ 0.01597724877452631059293608473126, 0.80731488863386358101337236803374};
        pts[102]={ 0.44056682953284115322745151388517, 0.07308623780482333132102468198354};
        pts[103]={ 0.48634693266233558484046284320357, 0.44056682953284115322745151388517};
        pts[104]={ 0.07308623780482333132102468198354, 0.48634693266233558484046284320357};
        pts[105]={ 0.82978592994847744090947117001633, 0.07420528900845844988420907384352};
        pts[106]={ 0.09600878104306410920631975614015, 0.82978592994847744090947117001633};
        pts[107]={ 0.07420528900845844988420907384352, 0.09600878104306410920631975614015};
        pts[108]={ 0.65837903678940146168230285184109, 0.07360353631169014310131615275168};
        pts[109]={ 0.26801742689890839521638099540723, 0.65837903678940146168230285184109};
        pts[110]={ 0.07360353631169014310131615275168, 0.26801742689890839521638099540723};
        pts[111]={ 0.31705863417413504556918724119896, 0.03658324556669936833408485199470};
        pts[112]={ 0.64635812025916561385230352243525, 0.31705863417413504556918724119896};
        pts[113]={ 0.03658324556669936833408485199470, 0.64635812025916561385230352243525};
        pts[114]={ 0.58071906287123287171425545238890, 0.17325514647294307235547705658973};
        pts[115]={ 0.24602579065582405593026749102137, 0.58071906287123287171425545238890};
        pts[116]={ 0.17325514647294307235547705658973, 0.24602579065582405593026749102137};
        pts[117]={ 0.58686000048975528464723083743593, 0.05986404483540469501079428482626};
        pts[118]={ 0.35327595467484002034197487773781, 0.58686000048975528464723083743593};
        pts[119]={ 0.05986404483540469501079428482626, 0.35327595467484002034197487773781};
        pts[120]={ 0.88747539463515101587631761503872, 0.04550883786193480468051930643014};
        pts[121]={ 0.06701576750291418638205698243837, 0.88747539463515101587631761503872};
        pts[122]={ 0.04550883786193480468051930643014, 0.06701576750291418638205698243837};
        pts[123]={ 0.34514648042267931371540612417448, 0.08501797433328461828239142050734};
        pts[124]={ 0.56983554524403601249105122406036, 0.34514648042267931371540612417448};
        pts[125]={ 0.08501797433328461828239142050734, 0.56983554524403601249105122406036};
        pts[126]={ 0.31854409565954938665655049589986, 0.25352454362284659872273095970741};
        pts[127]={ 0.42793136071760407013186977565056, 0.31854409565954938665655049589986};
        pts[128]={ 0.25352454362284659872273095970741, 0.42793136071760407013186977565056};
        pts[129]={ 0.73029425779457490808255215597455, 0.13475633020021907104712965974613};
        pts[130]={ 0.13494941200520602087031818427931, 0.73029425779457490808255215597455};
        pts[131]={ 0.13475633020021907104712965974613, 0.13494941200520602087031818427931};
        pts[132]={ 0.21422107853641420005530449088837, 0.04529135109275838483311460436198};
        pts[133]={ 0.74048757037082735266153576958459, 0.21422107853641420005530449088837};
        pts[134]={ 0.04529135109275838483311460436198, 0.74048757037082735266153576958459};
        pts[135]={ 1.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[136]={ 0.98930588311103995202699934452539, 0.00000000000000000000000000000000};
        pts[137]={ 0.96445076407629315440317441243678, 0.00000000000000000000000000000000};
        pts[138]={ 0.92623028889832303978124627974466, 0.00000000000000000000000000000000};
        pts[139]={ 0.87574710127630650546848301019054, 0.00000000000000000000000000000000};
        pts[140]={ 0.81445406863261027297085092868656, 0.00000000000000000000000000000000};
        pts[141]={ 0.74411464284035677696493848998216, 0.00000000000000000000000000000000};
        pts[142]={ 0.66675242391224931459703384462046, 0.00000000000000000000000000000000};
        pts[143]={ 0.58459301170464073038601782172918, 0.00000000000000000000000000000000};
        pts[144]={ 0.50000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[145]={ 0.41540698829535921410283094701299, 0.00000000000000000000000000000000};
        pts[146]={ 0.33324757608775068540296615537954, 0.00000000000000000000000000000000};
        pts[147]={ 0.25588535715964322303506151001784, 0.00000000000000000000000000000000};
        pts[148]={ 0.18554593136738972702914907131344, 0.00000000000000000000000000000000};
        pts[149]={ 0.12425289872369349453151698980946, 0.00000000000000000000000000000000};
        pts[150]={ 0.07376971110167696021875372025534, 0.00000000000000000000000000000000};
        pts[151]={ 0.03554923592370684559682558756322, 0.00000000000000000000000000000000};
        pts[152]={ 0.01069411688895999246184942421678, 0.00000000000000000000000000000000};
        pts[153]={ 0.00000000000000000000000000000000, 0.00000000000000000000000000000000};
        pts[154]={ 0.00000000000000000000000000000000, 1.00000000000000000000000000000000};
        pts[155]={ 0.00000000000000000000000000000000, 0.98930588311103995202699934452539};
        pts[156]={ 0.00000000000000000000000000000000, 0.96445076407629315440317441243678};
        pts[157]={ 0.00000000000000000000000000000000, 0.92623028889832303978124627974466};
        pts[158]={ 0.00000000000000000000000000000000, 0.87574710127630650546848301019054};
        pts[159]={ 0.00000000000000000000000000000000, 0.81445406863261027297085092868656};
        pts[160]={ 0.00000000000000000000000000000000, 0.74411464284035677696493848998216};
        pts[161]={ 0.00000000000000000000000000000000, 0.66675242391224931459703384462046};
        pts[162]={ 0.00000000000000000000000000000000, 0.58459301170464073038601782172918};
        pts[163]={ 0.00000000000000000000000000000000, 0.50000000000000000000000000000000};
        pts[164]={ 0.00000000000000000000000000000000, 0.41540698829535921410283094701299};
        pts[165]={ 0.00000000000000000000000000000000, 0.33324757608775068540296615537954};
        pts[166]={ 0.00000000000000000000000000000000, 0.25588535715964322303506151001784};
        pts[167]={ 0.00000000000000000000000000000000, 0.18554593136738972702914907131344};
        pts[168]={ 0.00000000000000000000000000000000, 0.12425289872369349453151698980946};
        pts[169]={ 0.00000000000000000000000000000000, 0.07376971110167696021875372025534};
        pts[170]={ 0.00000000000000000000000000000000, 0.03554923592370684559682558756322};
        pts[171]={ 0.00000000000000000000000000000000, 0.01069411688895999246184942421678};
        pts[172]={ 0.01069411688895999246184942421678, 0.98930588311103995202699934452539};
        pts[173]={ 0.03554923592370684559682558756322, 0.96445076407629315440317441243678};
        pts[174]={ 0.07376971110167696021875372025534, 0.92623028889832303978124627974466};
        pts[175]={ 0.12425289872369349453151698980946, 0.87574710127630650546848301019054};
        pts[176]={ 0.18554593136738972702914907131344, 0.81445406863261027297085092868656};
        pts[177]={ 0.25588535715964322303506151001784, 0.74411464284035677696493848998216};
        pts[178]={ 0.33324757608775068540296615537954, 0.66675242391224931459703384462046};
        pts[179]={ 0.41540698829535921410283094701299, 0.58459301170464073038601782172918};
        pts[180]={ 0.50000000000000000000000000000000, 0.50000000000000000000000000000000};
        pts[181]={ 0.58459301170464073038601782172918, 0.41540698829535921410283094701299};
        pts[182]={ 0.66675242391224931459703384462046, 0.33324757608775068540296615537954};
        pts[183]={ 0.74411464284035677696493848998216, 0.25588535715964322303506151001784};
        pts[184]={ 0.81445406863261027297085092868656, 0.18554593136738972702914907131344};
        pts[185]={ 0.87574710127630650546848301019054, 0.12425289872369349453151698980946};
        pts[186]={ 0.92623028889832303978124627974466, 0.07376971110167696021875372025534};
        pts[187]={ 0.96445076407629315440317441243678, 0.03554923592370684559682558756322};
        pts[188]={ 0.98930588311103995202699934452539, 0.01069411688895999246184942421678};
        pts[189]={ 0.33333333333333331482961625624739, 0.33333333333333331482961625624739};
        break;
    }
    default:
 	error_macro ("fekete node set: unsupported degree "<<degree);
  }
}

// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
#define _RHEOLEF_instanciation(T)                                             	\
template void set_simplex_leb_gll (size_t degree, Eigen::Matrix<point_basic<T>,Eigen::Dynamic,1>& pts);

_RHEOLEF_instanciation(Float)

}// namespace rheolef
