/*
 * Copyright (C) 2023, KylinSoft Co., Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
**/
#include "unifiedoutputconfig.h"
#include "resolutionslider.h"
#include "scalesize.h"
#include "utils.h"
#include "ukcccommon.h"
using namespace ukcc;

#include <QComboBox>
#include <QIcon>
#include <QLabel>
#include <QGridLayout>
#include <QSpacerItem>
#include <QCheckBox>
#include <QGroupBox>
#include <QVBoxLayout>
#include <QFormLayout>
#include <QStyledItemDelegate>
#include <QFile>
#include <QVector>
#include <QJsonDocument>
#include <QJsonArray>
#include <QJsonObject>
#include <QStandardPaths>
#include <QStringBuilder>

#include <KF5/KScreen/kscreen/output.h>
#include <KF5/KScreen/kscreen/config.h>
#include <KF5/KScreen/kscreen/getconfigoperation.h>
#include <KF5/KScreen/kscreen/configmonitor.h>
#include <KF5/KScreen/kscreen/getconfigoperation.h>
#include <KF5/KScreen/kscreen/setconfigoperation.h>
using namespace KScreen;

bool operator<(const QSize &s1, const QSize &s2)
{
    return s1.width() * s1.height() < s2.width() * s2.height();
}

template<>
bool qMapLessThanKey(const QSize &s1, const QSize &s2)
{
    return s1 < s2;
}

UnifiedOutputConfig::UnifiedOutputConfig(const KScreen::ConfigPtr &config, QWidget *parent) :
    OutputConfig(parent),
    mConfig(config)
{
    mIsRestore = true;
}

UnifiedOutputConfig::~UnifiedOutputConfig()
{
}

void UnifiedOutputConfig::setOutput(const KScreen::OutputPtr &output)
{
    mOutput = output;

    mClones.clear();
    mClones.reserve(mOutput->clones().count());
    Q_FOREACH (int id, mOutput->clones()) {
        mClones << mConfig->output(id);
    }
    mClones << mOutput;

    OutputConfig::setOutput(output);
}

void UnifiedOutputConfig::initUi()
{
    QVBoxLayout *vbox = new QVBoxLayout(this);
    vbox->setContentsMargins(0, 0, 0, 0);
    vbox->setSpacing(1);
    setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Preferred);

    KScreen::OutputPtr fakeOutput = createFakeOutput();
    mResolution = new ResolutionSlider(fakeOutput, this);

    mResolution->setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Minimum);
    mResolution->setMinimumWidth(402);

    //监听，否则无法处理修改分辨率/刷新率未保存
    connect(mOutput.data(), &KScreen::Output::currentModeIdChanged,
            this, &UnifiedOutputConfig::slotRestoreResoltion);

    connect(mOutput.data(), &KScreen::Output::rotationChanged,
            this, &UnifiedOutputConfig::slotRestoreRatation);

    QLabel *resLabel = new QLabel(this);
    resLabel->setText(tr("resolution"));
    resLabel->setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Minimum);
    resLabel->setFixedWidth(108);

    QHBoxLayout *resLayout = new QHBoxLayout();
    resLayout->setContentsMargins(16, 0, 16, 0);
    resLayout->addWidget(resLabel);
    resLayout->addWidget(mResolution);
    resLayout->setSpacing(16);

    UkccFrame *resFrame = new UkccFrame(this, UkccFrame::BorderRadiusStyle::None, true);
    resFrame->setLayout(resLayout);
    vbox->addWidget(resFrame);

    connect(mResolution, &ResolutionSlider::resolutionChanged,
            this, [=](QSize size, bool emitFlag){
                slotResolutionChanged(size, emitFlag);
                emit scaleChanged(size);
                UkccCommon::buriedSettings(QString("display"), QString("U-mResolution"), QString("select"), Utils::sizeToString(size));
            });

    // 方向下拉框
    mRotation = new QComboBox(this);
    mRotation->setMinimumWidth(402);
    mRotation->setMaximumWidth(16777215);

    QLabel *rotateLabel = new QLabel(this);
    rotateLabel->setText(tr("orientation"));
    rotateLabel->setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Minimum);
    rotateLabel->setFixedWidth(108);

    mRotation->addItem(tr("arrow-up"), KScreen::Output::None);
    mRotation->addItem(tr("90° arrow-right"), KScreen::Output::Right);
    mRotation->addItem(tr("90° arrow-left"), KScreen::Output::Left);
    mRotation->addItem(tr("arrow-down"), KScreen::Output::Inverted);

    int index = mRotation->findData(mOutput->rotation());
    mRotation->setCurrentIndex(index);

    connect(mRotation, static_cast<void (QComboBox::*)(int)>(&QComboBox::currentIndexChanged),
            this, &UnifiedOutputConfig::slotRotationChangedDerived);

    QHBoxLayout *roatateLayout = new QHBoxLayout();
    roatateLayout->setContentsMargins(16, 0, 16, 0);
    roatateLayout->addWidget(rotateLabel);
    roatateLayout->addWidget(mRotation);
    roatateLayout->setSpacing(16);

    mRotateFrame = new UkccFrame(this, UkccFrame::BorderRadiusStyle::None, true);
    mRotateFrame->setLayout(roatateLayout);
    vbox->addWidget(mRotateFrame);

    // 自动旋转
    mAutoRotation = new KSwitchButton(this);

    QLabel *autoRotateLabel = new QLabel(this);
    //~ contents_path /Display/auto rotation
    autoRotateLabel->setText(tr("auto rotation"));
    autoRotateLabel->setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Minimum);
    autoRotateLabel->setFixedWidth(108);

    QHBoxLayout *autoRotateLayout = new QHBoxLayout();
    autoRotateLayout->setContentsMargins(16, 0, 16, 0);
    autoRotateLayout->setSpacing(16);
    autoRotateLayout->addWidget(autoRotateLabel);
    autoRotateLayout->addStretch();
    autoRotateLayout->addWidget(mAutoRotation);

    mAutoRotationFrame = new UkccFrame(this, UkccFrame::BorderRadiusStyle::None, true);
    mAutoRotationFrame->setLayout(autoRotateLayout);

    vbox->addWidget(mAutoRotationFrame);
    mAutoRotationFrame->setVisible(false);

    mStatusManager = new QDBusInterface("com.kylin.statusmanager.interface",
                                        "/",
                                        "com.kylin.statusmanager.interface",
                                        QDBusConnection::sessionBus(), this);

    if (mStatusManager->isValid()) {
        // 仅平板模式下，且硬件支持自动旋转才显示
        QDBusReply<bool> isSupportedAuto = mStatusManager->call("is_supported_autorotation");
        QDBusReply<bool> tabletmode = mStatusManager->call("get_current_tabletmode");
        qDebug() << "tabletmode = " << tabletmode << "isSupportedAuto = " << isSupportedAuto;
        if (tabletmode && isSupportedAuto) {
            mAutoRotationFrame->setVisible(true);
        }
        connect(mStatusManager, SIGNAL(mode_change_signal(bool)), this, SLOT(tabletChanged(bool)));

        QDBusReply<bool> isAutoRotation = mStatusManager->call("get_auto_rotation");
        mAutoRotation->setChecked(isAutoRotation);
        mRotation->setEnabled(!isAutoRotation);
        connect(mStatusManager, SIGNAL(auto_rotation_change_signal(bool)), this, SLOT(rotationDbusSlot(bool)));
        connect(mStatusManager, SIGNAL(rotations_change_signal(QString)), this, SLOT(rotationDirectionSlot(QString)));
    }

    connect(mAutoRotation, &KSwitchButton::stateChanged, this, [=](bool checked){
        mRotation->setEnabled(!checked);
        if (mStatusManager->isValid()) {
            mStatusManager->call("set_auto_rotation", checked, "ukcc", "set_auto_rotation");
        }
    });

    // 统一输出刷新率下拉框
    mRefreshRate = new QComboBox(this);
    mRefreshRate->setMinimumWidth(402);
    mRefreshRate->setMaximumWidth(16777215);

    QLabel *freshLabel = new QLabel(this);
    freshLabel->setText(tr("frequency"));
    freshLabel->setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Minimum);
    freshLabel->setFixedWidth(108);

    QHBoxLayout *freshLayout = new QHBoxLayout();
    freshLayout->setContentsMargins(16, 0, 16, 0);
    freshLayout->setSpacing(16);
    freshLayout->addWidget(freshLabel);
    freshLayout->addWidget(mRefreshRate);

    UkccFrame *freshFrame = new UkccFrame(this, UkccFrame::BorderRadiusStyle::None, true);
    freshFrame->setLayout(freshLayout);
    vbox->addWidget(freshFrame);


    slotResolutionChanged(mResolution->currentResolution(), true);
    connect(mRefreshRate, static_cast<void (QComboBox::*)(int)>(&QComboBox::currentIndexChanged),
        this, &UnifiedOutputConfig::slotRefreshRateChanged);
    QObject::connect(new KScreen::GetConfigOperation(), &KScreen::GetConfigOperation::finished, this,
                         [&](KScreen::ConfigOperation *op) {
        KScreen::ConfigPtr sConfig = qobject_cast<KScreen::GetConfigOperation *>(op)->config();
        KScreen::OutputPtr sOutput = sConfig -> primaryOutput();

        for (int i = 0; i < mRefreshRate->count(); ++i) {
            if (!sOutput.isNull() && !sOutput->currentMode().isNull() && mRefreshRate->itemText(i) == refreshRateToText(sOutput->currentMode()->refreshRate())) {
                mRefreshRate->setCurrentIndex(i);
            }
        }
    });

    if (!(UkccCommon::isOpenkylin() && UkccCommon::isWayland()))
        return;

    UkccFrame *scaleFrame = new UkccFrame(this, UkccFrame::BorderRadiusStyle::None, true);

    QHBoxLayout *scaleLayout = new QHBoxLayout(scaleFrame);
    scaleLayout->setContentsMargins(16, 0, 16, 0);
    scaleLayout->setSpacing(16);
    mScaleCombox = new QComboBox(this);
    mScaleCombox->setObjectName("scaleCombox");

    QLabel *scaleLabel = new QLabel(this);
    //~ contents_path /Display/screen zoom
    scaleLabel->setText(tr("screen zoom"));
    scaleLabel->setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Minimum);
    scaleLabel->setFixedWidth(108);

    scaleLayout->addWidget(scaleLabel);
    scaleLayout->addWidget(mScaleCombox);

    vbox->addWidget(scaleFrame);

    connect(mOutput.data(), &KScreen::Output::scaleChanged,
            this, &OutputConfig::initScaleItem);

    connect(mOutput.data(), &KScreen::Output::currentModeIdChanged,
            this, &UnifiedOutputConfig::initScaleItem);
//    scaleFrame->hide();
    initScaleItem();
}

KScreen::OutputPtr UnifiedOutputConfig::createFakeOutput()
{
    // Find set of common resolutions
    QMap<QSize, int> commonSizes;
    Q_FOREACH (const KScreen::OutputPtr &clone, mClones) {
        QList<QSize> processedSizes;
        Q_FOREACH (const KScreen::ModePtr &mode, clone->modes()) {
            // Make sure we don't count some modes multiple times because of different
            // refresh rates
            if (processedSizes.contains(mode->size())) {
                continue;
            }

            processedSizes << mode->size();

            if (commonSizes.contains(mode->size())) {
                commonSizes[mode->size()]++;
            } else {
                commonSizes.insert(mode->size(), 1);
            }
        }
    }

    KScreen::OutputPtr fakeOutput(new KScreen::Output);

    // This will give us list of resolution that are shared by all outputs
    QList<QSize> commonResults = commonSizes.keys(mClones.count());
    // If there are no common resolution, fallback to smallest preferred mode
    if (commonResults.isEmpty()) {
        QSize smallestMode;
        Q_FOREACH (const KScreen::OutputPtr &clone, mClones) {
            if (!smallestMode.isValid() || clone->preferredMode()->size() < smallestMode) {
                smallestMode = clone->preferredMode()->size();
            }
        }
        commonResults << smallestMode;
    }
    std::sort(commonResults.begin(), commonResults.end());

    KScreen::ModeList modes;
    Q_FOREACH (const QSize &size, commonResults) {
        KScreen::ModePtr mode(new KScreen::Mode);
        mode->setSize(size);
        mode->setId(Utils::sizeToString(size));
        mode->setName(mode->id());
        modes.insert(mode->id(), mode);
    }
    fakeOutput->setModes(modes);
    if (!mOutput->currentModeId().isEmpty()) {
        fakeOutput->setCurrentModeId(Utils::sizeToString(mOutput->currentMode()->size()));
    } else {
        fakeOutput->setCurrentModeId(Utils::sizeToString(commonResults.last()));
    }

    return fakeOutput;
}

void UnifiedOutputConfig::slotResolutionChanged(const QSize &size, bool emitFlag)
{
    // Ignore disconnected outputs
    if (!size.isValid()) {
        return;
    }
    bool mIsModeInit = true;
    QVector<QString>Vrefresh;
    bool mIsCloneMode = isCloneMode();
    mRefreshRate->blockSignals(true);
    mRefreshRate->clear();
    mRefreshRate->blockSignals(false);
    Q_FOREACH (const KScreen::OutputPtr &clone, mClones) {
        const QString &id = findBestMode(clone, size);
        if (id.isEmpty()) {
            // FIXME: Error?
            return;
        }
        //本来就是镜像模式且当前分辨率就是选中分辨率，就不需要重新设置显示参数
        //用于镜像模式下刚打开控制面板时的显示，否则显示的不是实际刷新率而是findBestMode
        if (!mIsCloneMode || size != clone->currentMode()->size()) {
            mIsModeInit = false;
            clone->blockSignals(true); //必须加blockSignals，否则在这里就会触发currentModeIdChanged的信号
            mIsRestore = false; //修改配置会触发currentModeIdChanged时，不运行slotRestoreResoltion
            clone->setCurrentModeId(id);
            clone->setPos(QPoint(0, 0));
            clone->blockSignals(false);
        }

        QList<KScreen::ModePtr> modes;
        Q_FOREACH (const KScreen::ModePtr &mode, clone->modes()) {
            if (mode->size() == size) {
                modes << mode;
            }
        }

        QVector<QString>VrefreshTemp;
        for (int i = 0, total = modes.count(); i < total; ++i) {
           const KScreen::ModePtr mode = modes.at(i);

           bool alreadyExisted = false; //判断该显示器的刷新率是否有重复的，确保同一刷新率在一个屏幕上只出现一次
           for (int j = 0; j < VrefreshTemp.size(); ++j) {
               if (refreshRateToText(mode->refreshRate()) == VrefreshTemp[j]) {
                   alreadyExisted = true;
                   break;
               }
           }
           if (alreadyExisted == false) {   //不添加重复的项
               VrefreshTemp.append(refreshRateToText(mode->refreshRate()));
           }
        }

        for (int i = 0; i < VrefreshTemp.size(); ++i) {
            Vrefresh.append(VrefreshTemp[i]);
        }
    }

    for (int i = 0; i < Vrefresh.size(); ++i) {
        if (Vrefresh.count(Vrefresh[i]) == mClones.size()) { //该刷新率出现次数等于屏幕数，即每个屏幕都有该刷新率
            bool existFlag = false;
            for (int j = 0; j < mRefreshRate->count(); ++j) {  //已经存在就不再添加
                if (Vrefresh[i] == mRefreshRate->itemText(j)) {
                    existFlag = true;
                    break;
                }
            }
            if (existFlag == false) {  //不存在添加到容器中
                mRefreshRate->blockSignals(true);
                mRefreshRate->addItem(Vrefresh[i]);
                mRefreshRate->blockSignals(false);
            }
        }
    }

    if (mRefreshRate->count() > 1) {
        float currentRereshRate = mClones[0]->currentMode()->refreshRate();
        for (int i = 0; i < mRefreshRate->count(); i++) {
            if (refreshRateToText(currentRereshRate) == mRefreshRate->itemText(i)) {
                mRefreshRate->blockSignals(true);
                mRefreshRate->setCurrentIndex(i);
                mRefreshRate->blockSignals(false);
                break;
            }
        }
    }

    if (mRefreshRate->count() == 0) {
        mRefreshRate->blockSignals(true);
        mRefreshRate->addItem(tr("auto"), -1);
        mRefreshRate->blockSignals(false);
    }
    if (emitFlag && !mIsModeInit){
        changeItm = RESOLUTION;
        Q_EMIT changed();
    }
}

void UnifiedOutputConfig::slotRefreshRateChanged(int index)
{
    Q_FOREACH (const KScreen::OutputPtr &clone, mClones) {
        Q_FOREACH (const KScreen::ModePtr &mode, clone->modes()) {
            if (mode->size() == mResolution->currentResolution() && \
                    refreshRateToText(mode->refreshRate()) == mRefreshRate->itemText(index)) {
                mIsRestore = false;
                clone->blockSignals(true);
                clone->setCurrentModeId(mode->id());
                clone->blockSignals(false);
            }
        }
    }
    changeItm = FREQUENCY;
    Q_EMIT changed();
    UkccCommon::buriedSettings(QString("display"), QString("U-mRefreshRate"), QString("select"), mRefreshRate->currentText());
}

void UnifiedOutputConfig::slotScaleChanged(int index)
{
    mScaleCombox->blockSignals(true);
    qreal kscreenScale = mScaleCombox->itemData(index).toDouble();

    Q_FOREACH (const KScreen::OutputPtr &clone, mClones) {
        clone->setScale(kscreenScale);
    }

    changeItm = SCALE;
    Q_EMIT changed();
    mScaleCombox->blockSignals(false);
}

QString UnifiedOutputConfig::findBestMode(const KScreen::OutputPtr &output, const QSize &size)
{
    float refreshRate = 0;
    QString id;
    Q_FOREACH (const KScreen::ModePtr &mode, output->modes()) {
        if (mode->size() == size && mode->refreshRate() > refreshRate) {
            refreshRate = mode->refreshRate();
            id = mode->id();
        }
    }
    return id;
}

// 统一输出方向信号改变
void UnifiedOutputConfig::slotRotationChangedDerived(int index)
{
    KScreen::Output::Rotation rotation = static_cast<KScreen::Output::Rotation>(mRotation->itemData(index).toInt());
    Q_FOREACH (const KScreen::OutputPtr &clone, mClones) {
        if (clone->isConnected() && clone->isEnabled()) {
            clone->blockSignals(true);
            clone->setRotation(rotation);
            clone->setPos(QPoint(0, 0));
            clone->blockSignals(false);
        }
    }
    changeItm = ORIENTATION;
    Q_EMIT changed();
    UkccCommon::buriedSettings(QString("display"), QString("U-mRotation"), QString("select"), QString::number(index));
}

void UnifiedOutputConfig::rotationDbusSlot(bool autoRotation)
{
    mRotation->setEnabled(!autoRotation);
    mAutoRotation->blockSignals(true);
    mAutoRotation->setChecked(autoRotation);
    mAutoRotation->blockSignals(false);
}

void UnifiedOutputConfig::tabletChanged(bool tabletMode)
{
    QDBusReply<bool> isSupportedAuto = mStatusManager->call("is_supported_autorotation");
    qDebug() << "tabletmode = " << tabletMode << "isSupportedAuto = " << isSupportedAuto;
    if (tabletMode && isSupportedAuto) {
        mAutoRotationFrame->setVisible(true);
    } else {
        mAutoRotationFrame->setVisible(false);
    }
}

void UnifiedOutputConfig::rotationDirectionSlot(QString rotationDirection)
{
    mRotation->blockSignals(true);
    if (rotationDirection.compare("normal") == 0) {
        mRotation->setCurrentIndex(KScreen::Output::None);
    } else if (rotationDirection.compare("upside-down") == 0) {
        mRotation->setCurrentIndex(KScreen::Output::Inverted);
    } else if (rotationDirection.compare("left") == 0) {
        mRotation->setCurrentIndex(KScreen::Output::Left);
    } else if (rotationDirection.compare("right") == 0) {
        mRotation->setCurrentIndex(KScreen::Output::Right);
    }
    mRotation->blockSignals(false);
}

void UnifiedOutputConfig::slotRestoreResoltion()
{
    if (!mIsRestore || !mOutput->currentMode()) {
        mIsRestore = true;
        if (!mSetRestore) {
            return;
        }
    }
    mSetRestore = false;
    if (mResolution->currentResolution() != mOutput->currentMode()->size()) { //分辨率改变时，触发该信号重新加载刷新率，用于修改分辨率之后但未保存
        mResolution->setResolution(mOutput->currentMode()->size()); //这里面不会触发分辨率改变信号
        Q_EMIT scaleChanged(mOutput->currentMode()->size());
        slotResolutionChanged(mOutput->currentMode()->size(), false);
    } else { //分辨率未修改，刷新率修改,用于修改刷新率之后但未保存
        for (int i = 0; i < mRefreshRate->count(); i++) {
           if (refreshRateToText(mOutput->currentMode()->refreshRate()) == mRefreshRate->itemText(i)\
                   || mRefreshRate->count() == 1) {
               mRefreshRate->blockSignals(true);
               mRefreshRate->setCurrentIndex(i);
               mRefreshRate->blockSignals(false);
               break;
           }
        }
    }
}

void UnifiedOutputConfig::slotRestoreRatation()
{
    mRotation->blockSignals(true);
    mRotation->setCurrentIndex(mRotation->findData(mOutput->rotation()));
    mRotation->blockSignals(false);
}

bool UnifiedOutputConfig::isCloneMode()
{
    /*
     *不能直接用isVisible判断是否为镜像模式
     *设置镜像模式时，visiable总是true，但此时还未设置currentMode
     *导致某些情况异常
     */
    //return this->isVisible();    //显示则表示是统一输出
    if (!mClones.isEmpty() && mClones[0] && mClones[0]->currentMode()) {
        QSize cloneSize(mClones[0]->currentMode()->size());
        QPoint clonePos(mClones[0]->pos());
        Q_FOREACH (const KScreen::OutputPtr &clone, mClones) {
            if (clone->currentMode() && (clone->currentMode()->size() != cloneSize || clone->pos() != clonePos)) {
                return false;
            }
        }
        return true;
    } else {
        return false;
    }

}

void UnifiedOutputConfig::initScaleItem()
{
    if (!UkccCommon::isOpenkylin() && !UkccCommon::isWayland())
        return;
    mScaleCombox->blockSignals(true);
    if (!mOutput->currentMode())
        return;
    QSize scalesize = mOutput->currentMode()->size();

    mScaleCombox->addItem("100%", 1.0);
    if (scalesize.width() > 1024 ) {
        mScaleCombox->addItem("125%", 1.25);
    }
    if (scalesize.width() == 1920 ) {
        mScaleCombox->addItem("150%", 1.5);
    }
    if (scalesize.width() > 1920) {
        mScaleCombox->addItem("150%", 1.5);
        mScaleCombox->addItem("175%", 1.75);
    }
    if (scalesize.width() >= 2160) {
        mScaleCombox->addItem("200%", 2.0);
    }
    if (scalesize.width() > 2560) {
        mScaleCombox->addItem("225%", 2.25);
    }
    if (scalesize.width() > 3072) {
        mScaleCombox->addItem("250%", 2.5);
    }
    if (scalesize.width() > 3840) {
        mScaleCombox->addItem("275%", 2.75);
    }

    if (mScaleCombox->findData(mOutput->scale()) == -1) {
        Q_FOREACH (const KScreen::OutputPtr &clone, mClones) {
            clone->setScale(1);
        }
    }

    mScaleCombox->setCurrentText(QString::number(mOutput->scale() * 100) + "%");
    mScaleCombox->blockSignals(false);
}

double UnifiedOutputConfig::getGlobalData(OutputPtr output)
{
    QString hash = mConfig->connectedOutputsHash();

    QString scaleDir = QStandardPaths::writableLocation(QStandardPaths::GenericDataLocation)
            %QStringLiteral("/kscreen/scale/");
    QFile file(scaleDir % hash);
    if (!file.open(QIODevice::ReadOnly)) {
        qDebug() << "Failed to open file" << file.fileName();
        return 0;
    }

    QByteArray readBy=file.readAll();
    QJsonParseError error;
    QJsonDocument readDoc=QJsonDocument::fromJson(readBy,&error);
    QJsonArray obj=readDoc.array();

    for (int i = 0 ; i < obj.size(); i++) {
        QJsonObject faObj= obj[i].toObject();
        if (faObj["id"].toString() == output->hashMd5())
            return faObj["scale"].toDouble();
    }
    return 0;
}

QFrame *UnifiedOutputConfig::setLine(QFrame *frame)
{
    QFrame *line = new QFrame(frame);
    line->setMinimumSize(QSize(0, 1));
    line->setMaximumSize(QSize(16777215, 1));
    line->setLineWidth(0);
    line->setFrameShape(QFrame::HLine);
    line->setFrameShadow(QFrame::Sunken);
    return line;
}
